package org.phosphoresce.webcore.ext.struts.util;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.Iterator;

import javax.servlet.http.HttpServletRequest;

import org.apache.struts.action.ActionMessage;
import org.apache.struts.action.ActionMessages;
import org.apache.struts.action.RequestProcessor;
import org.apache.struts.config.ModuleConfig;
import org.phosphoresce.lib.commons.util.AnnotationUtil;
import org.phosphoresce.lib.commons.util.ArrayUtil;
import org.phosphoresce.lib.commons.util.ReflectionUtil;
import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.ext.struts.StrutsConfigNames;
import org.phosphoresce.webcore.ext.struts.StrutsConstants;
import org.phosphoresce.webcore.ext.struts.action.AbstractAction;
import org.phosphoresce.webcore.ext.struts.action.AbstractForm;
import org.phosphoresce.webcore.ext.struts.action.AbstractLogic;
import org.phosphoresce.webcore.ext.struts.action.ScreenSession;
import org.phosphoresce.webcore.ext.struts.action.ServletContainer;
import org.phosphoresce.webcore.ext.struts.action.WindowSessionContainer;
import org.phosphoresce.webcore.ext.struts.annotation.StrutsAction;
import org.phosphoresce.webcore.ext.struts.annotation.StrutsForm;
import org.phosphoresce.webcore.ext.struts.annotation.StrutsLogic;
import org.phosphoresce.webcore.ext.struts.annotation.StrutsMethod;
import org.phosphoresce.webcore.ext.struts.annotation.StrutsSession;
import org.phosphoresce.webcore.ext.struts.exception.StrutsProcessInternalException;

/**
 * Strutsモジュールユーティリティクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/19	Kitagawa		新規作成
 * 2012/07/04	Kitagawa		全体的に再構築
 * 2012/12/01	Kitagawa		ランダムな文字列生成のメソッドを追加
 *-->
 */
public final class StrutsUtil implements StrutsConstants {

	/**
	 * コンストラクタ<br>
	 */
	private StrutsUtil() {
		super();
	}

	/**
	 * リクエストされた各種情報からウィンドウIDを取得します。<br>
	 * ウィンドウIDはアクションフォーム、リクエストパラメータ、リクエストアトリビュートの優先順位で取得します。<br>
	 * @param request リクエストオブジェクト
	 * @param form フォームオブジェクト
	 * @return ウィンドウID
	 */
	public static String getRequestWindowId(HttpServletRequest request, AbstractForm<? extends Serializable> form) {
		String windowId = "";

		// アクションフォームフィールド取得
		if (StringUtil.isEmpty(windowId) && form != null) {
			windowId = form.getControlFieldWindowId();
		}

		// リクエストパラメータ取得
		if (StringUtil.isEmpty(windowId) && request != null) {
			windowId = request.getParameter(FIELD_KEY_WINDOW_ID);
		}

		// リクエストアトリビュート取得
		if (StringUtil.isEmpty(windowId) && request != null) {
			windowId = (String) request.getAttribute(FIELD_KEY_WINDOW_ID);
		}

		return windowId;
	}

	/**
	 * リクエストされた各種情報からトークンIDを取得します。<br>
	 * トークンIDはアクションフォーム、リクエストパラメータ、リクエストアトリビュートの優先順位で取得します。<br>
	 * @param request リクエストオブジェクト
	 * @param form フォームオブジェクト
	 * @return トークンID
	 */
	public static String getRequestTokenId(HttpServletRequest request, AbstractForm<? extends Serializable> form) {
		String tokenId = "";

		// アクションフォームフィールド取得
		if (StringUtil.isEmpty(tokenId) && form != null) {
			tokenId = form.getControlFieldTokenId();
		}

		// リクエストパラメータ取得
		if (StringUtil.isEmpty(tokenId) && request != null) {
			tokenId = request.getParameter(FIELD_KEY_TOKEN_ID);
		}

		// リクエストアトリビュート取得
		if (StringUtil.isEmpty(tokenId) && request != null) {
			tokenId = (String) request.getAttribute(FIELD_KEY_TOKEN_ID);
		}

		return tokenId;
	}

	/**
	 * リクエストされた各種情報からポップアップウィンドウリクエストであるか判定します。<br>
	 * @param request リクエストオブジェクト
	 * @param form フォームオブジェクト
	 * @return ポップアップウィンドウリクエストである場合にtrueを返却
	 */
	public static boolean isPopupRequest(HttpServletRequest request, AbstractForm<? extends Serializable> form) {
		String popupWindow = "";

		// アクションフォームフィールド取得
		if (StringUtil.isEmpty(popupWindow) && form != null) {
			popupWindow = form.getControlFieldPopupWindow();
		}

		// リクエストパラメータ取得
		if (StringUtil.isEmpty(popupWindow) && request != null) {
			popupWindow = request.getParameter(FIELD_KEY_POPUP_WINDOW);
		}

		// リクエストアトリビュート取得
		if (StringUtil.isEmpty(popupWindow) && request != null) {
			popupWindow = (String) request.getAttribute(FIELD_KEY_POPUP_WINDOW);
		}

		return !StringUtil.isEmpty(popupWindow);
	}

	/**
	 * リクエストされた各種情報からJSONリクエストであるか判定します。<br>
	 * @param request リクエストオブジェクト
	 * @param form フォームオブジェクト
	 * @return JSONリクエストである場合にtrueを返却
	 */
	public static boolean isJSONReuqest(HttpServletRequest request, AbstractForm<? extends Serializable> form) {
		String jsonRequest = "";

		// アクションフォームフィールド取得
		if (StringUtil.isEmpty(jsonRequest) && form != null) {
			jsonRequest = form.getControlFieldJsonRequest();
		}

		// リクエストパラメータ取得
		if (StringUtil.isEmpty(jsonRequest) && request != null) {
			jsonRequest = request.getParameter(FIELD_KEY_JSON_REQUEST);
		}

		// リクエストアトリビュート取得
		if (StringUtil.isEmpty(jsonRequest) && request != null) {
			jsonRequest = (String) request.getAttribute(FIELD_KEY_JSON_REQUEST);
		}

		return !StringUtil.isEmpty(jsonRequest);
	}

	/**
	 * アクションパスを取得します。<br>
	 * リクエストされたURLからアクションサーブレットパスを除くパス文字列を取得します。<br>
	 * <pre>
	 * リクエストURL      : http://localhost:8080/alias/!f/Function/login?foo=bar&hoge=hoge
	 * アクションパス     : /Function/login
	 * リクエスト機能     : Function
	 * 機能メソッド       : login
	 * </pre>
	 * @param request HttpServletRequestオブジェクト
	 * @param moduleConfig ModuleConfigオブジェクト
	 * @return アクションパス
	 */
	public static String getRequestActionPath(HttpServletRequest request, ModuleConfig moduleConfig) {
		String path = null;

		// リクエスト属性パス情報解析
		path = (String) request.getAttribute(RequestProcessor.INCLUDE_PATH_INFO);
		if (path == null) {
			path = request.getPathInfo();
		}
		if (path != null && path.length() > 0) {
			return path;
		}

		// サーブレットパス解析
		path = (String) request.getAttribute(RequestProcessor.INCLUDE_SERVLET_PATH);
		if (path == null) {
			path = request.getServletPath();
		}
		String prefix = moduleConfig.getPrefix();
		if (!path.startsWith(prefix)) {
			throw new StrutsProcessInternalException("FSTR00005", new Object[] { path });
		}
		path = path.substring(prefix.length());
		int slash = path.lastIndexOf("/");
		int period = path.lastIndexOf(".");
		if (period >= 0 && period > slash) {
			path = path.substring(0, period);
		}

		return path;
	}

	/**
	 * リクエスト機能を取得します。<br>
	 * リクエストされたURLからリクエスト機能(アクションクラスプレフィクスと一致)を取得します。<br>
	 * <pre>
	 * リクエストURL      : http://localhost:8080/alias/!f/Function/login?foo=bar&hoge=hoge
	 * アクションパス     : /Function/login
	 * リクエスト機能     : Function
	 * 機能メソッド       : login
	 * </pre>
	 * @param actionPath アクションパス
	 * @return リクエスト昨日
	 */
	public static String getRequestFunction(String actionPath) {
		if (StringUtil.isEmpty(actionPath)) {
			throw new StrutsProcessInternalException("FSTR00006", new Object[] { actionPath });
		}
		if (actionPath.indexOf("/", 1) >= 0) {
			actionPath = actionPath.substring(0, actionPath.indexOf("/", 1));
		}
		if (actionPath.indexOf("/") >= 0) {
			return StringUtil.right(actionPath, actionPath.length() - 1);
		} else {
			return actionPath;
		}
	}

	/**
	 * リクエスト機能メソッドを取得します。
	 * リクエストされたURLからリクエスト機能メソッド(アクションメソッドと一致)を取得します。<br>
	 * <pre>
	 * リクエストURL      : http://localhost:8080/alias/!f/Function/login?foo=bar&hoge=hoge
	 * アクションパス     : /Function/login
	 * リクエスト機能     : Function
	 * 機能メソッド       : login
	 * </pre>
	 * @param actionPath リクエストパス文字列
	 * @return 対象アクションメソッド名
	 */
	public static String getRequestFunctionMethod(String actionPath) {
		String method = StringUtil.right(actionPath, actionPath.length() - 1);
		if (method.indexOf("/") < 0) {
			method = Config.getString(StrutsConfigNames.STRUTS_ACTION_DEFAULT);
		} else {
			method = method.substring(method.indexOf("/") + 1);
		}
		if (StringUtil.isEmpty(method)) {
			method = Config.getString(StrutsConfigNames.STRUTS_ACTION_DEFAULT);
		}
		if (method.endsWith("/")) {
			method = StringUtil.left(method, method.length() - 1);
		}
		if (method.indexOf("/") >= 0) {
			throw new StrutsProcessInternalException("FSTR00007", new Object[] { actionPath });
		}
		return method;
	}

	/**
	 * 指定されたアクションパスの定数インタフェースクラス名(SimpleName)を取得します。<br>
	 * @param actionPath アクションパス
	 * @return 定数インタフェースクラス名(SimpleName)
	 */
	public static String getConstantsClassSimpleName(String actionPath) {
		String function = getRequestFunction(actionPath);
		String className = EMPTY_STRING;
		if (!StringUtil.isEmpty(function)) {
			className = function + Config.getString(StrutsConfigNames.STRUTS_CLASS_SUFFIX_CONSTANTS);
		}
		return className;
	}

	/**
	 * 指定されたアクションパスのアクションクラス名(SimpleName)を取得します。<br>
	 * @param actionPath アクションパス
	 * @return アクションクラス名(SimpleName)
	 */
	public static String getActionClassSimpleName(String actionPath) {
		String function = getRequestFunction(actionPath);
		String className = EMPTY_STRING;
		if (!StringUtil.isEmpty(function)) {
			className = function + Config.getString(StrutsConfigNames.STRUTS_CLASS_SUFFIX_ACTION);
		}
		return className;
	}

	/**
	 * 指定されたアクションパスのフォームクラス名(SimpleName)を取得します。<br>
	 * @param actionPath アクションパス
	 * @return フォームクラス名(SimpleName)
	 */
	public static String getFormClassSimpleName(String actionPath) {
		String function = getRequestFunction(actionPath);
		String className = EMPTY_STRING;
		if (!StringUtil.isEmpty(function)) {
			className = function + Config.getString(StrutsConfigNames.STRUTS_CLASS_SUFFIX_FORM);
		}
		return className;
	}

	/**
	 * 指定されたアクションパスのロジッククラス名(SimpleName)を取得します。<br>
	 * @param actionPath アクションパス
	 * @return ロジッククラス名(SimpleName)
	 */
	public static String getLogicClassSimpleName(String actionPath) {
		String function = getRequestFunction(actionPath);
		String className = EMPTY_STRING;
		if (!StringUtil.isEmpty(function)) {
			className = function + Config.getString(StrutsConfigNames.STRUTS_CLASS_SUFFIX_LOGIC);
		}
		return className;
	}

	/**
	 * 指定されたアクションパスのセッションクラス名(SimpleName)を取得します。<br>
	 * @param actionPath アクションパス
	 * @return セッションクラス名(SimpleName)
	 */
	public static String getSessionClassSimpleName(String actionPath) {
		String function = getRequestFunction(actionPath);
		String className = EMPTY_STRING;
		if (!StringUtil.isEmpty(function)) {
			className = function + Config.getString(StrutsConfigNames.STRUTS_CLASS_SUFFIX_SESSION);
		}
		return className;
	}

	/**
	 * 指定されたアクションパスのユーティリティクラス名(SimpleName)を取得します。<br>
	 * @param actionPath アクションパス
	 * @return ユーティリティクラス名(SimpleName)
	 */
	public static String getUtilClassSimpleName(String actionPath) {
		String function = getRequestFunction(actionPath);
		String className = EMPTY_STRING;
		if (!StringUtil.isEmpty(function)) {
			className = function + Config.getString(StrutsConfigNames.STRUTS_CLASS_SUFFIX_UTIL);
		}
		return className;
	}

	/**
	 * 指定されたアクションパスの定数インタフェースクラス名(FQCN)を取得します。<br>
	 * @param actionPath アクションパス
	 * @return 定数インタフェースクラス名(FQCN)
	 */
	public static String getConstantsClassName(String actionPath) {
		String className = getConstantsClassSimpleName(actionPath);
		if (!StringUtil.isEmpty(className)) {
			String function = getRequestFunction(actionPath);
			className = Config.getString(StrutsConfigNames.STRUTS_ACTION_PACKAGE) + "." + function + "." + className;
		}
		return className;
	}

	/**
	 * 指定されたアクションパスのアクションクラス名(FQCN)を取得します。<br>
	 * @param actionPath アクションパス
	 * @return アクションクラス名(FQCN)
	 */
	public static String getActionClassName(String actionPath) {
		String className = getActionClassSimpleName(actionPath);
		if (!StringUtil.isEmpty(className)) {
			String function = getRequestFunction(actionPath);
			className = Config.getString(StrutsConfigNames.STRUTS_ACTION_PACKAGE) + "." + function + "." + className;
		}
		return className;
	}

	/**
	 * 指定されたアクションパスのフォームクラス名(FQCN)を取得します。<br>
	 * @param actionPath アクションパス
	 * @return フォームクラス名(FQCN)
	 */
	public static String getFormClassName(String actionPath) {
		String className = getFormClassSimpleName(actionPath);
		if (!StringUtil.isEmpty(className)) {
			String function = getRequestFunction(actionPath);
			className = Config.getString(StrutsConfigNames.STRUTS_ACTION_PACKAGE) + "." + function + "." + className;
		}
		return className;
	}

	/**
	 * 指定されたアクションパスのロジッククラス名(FQCN)を取得します。<br>
	 * @param actionPath アクションパス
	 * @return ロジッククラス名(FQCN)
	 */
	public static String getLogicClassName(String actionPath) {
		String className = getLogicClassSimpleName(actionPath);
		if (!StringUtil.isEmpty(className)) {
			String function = getRequestFunction(actionPath);
			className = Config.getString(StrutsConfigNames.STRUTS_ACTION_PACKAGE) + "." + function + "." + className;
		}
		return className;
	}

	/**
	 * 指定されたアクションパスのセッションクラス名(FQCN)を取得します。<br>
	 * @param actionPath アクションパス
	 * @return セッションクラス名(FQCN)
	 */
	public static String getSessionClassName(String actionPath) {
		String className = getSessionClassSimpleName(actionPath);
		if (!StringUtil.isEmpty(className)) {
			String function = getRequestFunction(actionPath);
			className = Config.getString(StrutsConfigNames.STRUTS_ACTION_PACKAGE) + "." + function + "." + className;
		}
		return className;
	}

	/**
	 * 指定されたアクションパスのユーティリティクラス名(FQCN)を取得します。<br>
	 * @param actionPath アクションパス
	 * @return ユーティリティクラス名(FQCN)
	 */
	public static String getUtilClassName(String actionPath) {
		String className = getUtilClassSimpleName(actionPath);
		if (!StringUtil.isEmpty(className)) {
			String function = getRequestFunction(actionPath);
			className = Config.getString(StrutsConfigNames.STRUTS_ACTION_PACKAGE) + "." + function + "." + className;
		}
		return className;
	}

	/**
	 * 指定されたアクションパスの定数インタフェースクラスを取得します。<br>
	 * @param actionPath アクションパス
	 * @return アクションクラス
	 */
	public static Class<?> getConstantsClass(String actionPath) {
		try {
			Class<?> clazz = null;
			String className = getConstantsClassName(actionPath);
			if (!StringUtil.isEmpty(className)) {
				clazz = (Class<?>) Class.forName(className);
			}
			return clazz;
		} catch (ClassNotFoundException e) {
			return null;
		}
	}

	/**
	 * 指定されたアクションパスのアクションクラスを取得します。<br>
	 * @param actionPath アクションパス
	 * @return アクションクラス
	 */
	@SuppressWarnings("unchecked")
	public static Class<? extends AbstractAction<? extends Serializable, AbstractForm<? extends Serializable>>> getActionClass(String actionPath) {
		try {
			Class<? extends AbstractAction<? extends Serializable, AbstractForm<? extends Serializable>>> clazz = null;
			String className = getActionClassName(actionPath);
			if (!StringUtil.isEmpty(className)) {
				clazz = (Class<? extends AbstractAction<? extends Serializable, AbstractForm<? extends Serializable>>>) Class.forName(className);
			}
			return clazz;
		} catch (ClassNotFoundException e) {
			return null;
		}
	}

	/**
	 * 指定されたアクションパスのフォームクラスを取得します。<br>
	 * @param actionPath アクションパス
	 * @return フォームクラス
	 */
	@SuppressWarnings("unchecked")
	public static Class<? extends AbstractForm<? extends Serializable>> getFormClass(String actionPath) {
		try {
			Class<? extends AbstractForm<? extends Serializable>> clazz = null;
			String className = getFormClassName(actionPath);
			if (!StringUtil.isEmpty(className)) {
				clazz = (Class<? extends AbstractForm<? extends Serializable>>) Class.forName(className);
			}
			return clazz;
		} catch (ClassNotFoundException e) {
			return null;
		}
	}

	/**
	 * 指定されたアクションパスのロジッククラスを取得します。<br>
	 * @param actionPath アクションパス
	 * @return ロジッククラス
	 */
	@SuppressWarnings("unchecked")
	public static Class<? extends AbstractLogic<? extends Serializable>> getLogicClass(String actionPath) {
		try {
			Class<? extends AbstractLogic<? extends Serializable>> clazz = null;
			String className = getLogicClassName(actionPath);
			if (!StringUtil.isEmpty(className)) {
				clazz = (Class<? extends AbstractLogic<? extends Serializable>>) Class.forName(className);
			}
			return clazz;
		} catch (ClassNotFoundException e) {
			return null;
		}
	}

	/**
	 * 指定されたアクションパスのセッションクラスを取得します。<br>
	 * @param actionPath アクションパス
	 * @return セッションクラス
	 */
	@SuppressWarnings("unchecked")
	public static Class<?> getSessionClass(String actionPath) {
		try {
			Class<?> clazz = null;
			String className = getSessionClassName(actionPath);
			if (!StringUtil.isEmpty(className)) {
				clazz = (Class<? extends AbstractLogic<? extends Serializable>>) Class.forName(className);
			}
			return clazz;
		} catch (ClassNotFoundException e) {
			return null;
		}
	}

	/**
	 * 指定されたアクションパスのユーティリティクラスを取得します。<br>
	 * @param actionPath アクションパス
	 * @return ユーティリティクラス
	 */
	@SuppressWarnings("unchecked")
	public static Class<?> getUtilClass(String actionPath) {
		try {
			Class<?> clazz = null;
			String className = getUtilClassName(actionPath);
			if (!StringUtil.isEmpty(className)) {
				clazz = (Class<? extends AbstractLogic<? extends Serializable>>) Class.forName(className);
			}
			return clazz;
		} catch (ClassNotFoundException e) {
			return null;
		}
	}

	/**
	 * 指定されたアクションパスのアクションメソッドを取得します。<br>
	 * @param actionPath アクションパス
	 * @return アクションメソッド
	 */
	public static Method getActionMethod(String actionPath) {
		Class<? extends AbstractAction<? extends Serializable, AbstractForm<? extends Serializable>>> actionClass = getActionClass(actionPath);
		Class<? extends AbstractForm<? extends Serializable>> formClass = getFormClass(actionPath);
		String methodName = getRequestFunctionMethod(actionPath);
		if (actionClass == null) {
			return null;
		}
		Method method = ReflectionUtil.getMethod(actionClass, methodName, formClass, WindowSessionContainer.class, ServletContainer.class);
		return method;
	}

	/**
	 * 指定されたActionMessagesにActionMessageが含まれているか判定します。<br>
	 * @param savedMessages ActionMessagesオブジェクト
	 * @param message ActionMessageオブジェクト
	 * @return 含まれている場合にtrueを返却
	 */
	public static boolean containsActionMessage(ActionMessages savedMessages, ActionMessage message) {
		if (savedMessages == null || message == null) {
			return false;
		}

		for (@SuppressWarnings("unchecked")
		Iterator<String> iterator1 = savedMessages.properties(); iterator1.hasNext();) {
			String key = iterator1.next();
			for (@SuppressWarnings("unchecked")
			Iterator<ActionMessage> iterator2 = savedMessages.get(key); iterator2.hasNext();) {
				ActionMessage saveedMessage = (ActionMessage) iterator2.next();

				if (saveedMessage.getKey().equals(message.getKey())) {
					Object[] savedMessageValues = saveedMessage.getValues();
					Object[] messageValues = message.getValues();
					if (savedMessageValues == null && messageValues == null) {
						return true;
					}
					if (savedMessageValues == null || messageValues == null) {
						return false;
					}
					if (savedMessageValues.length != messageValues.length) {
						return false;
					}
					for (int i = 0; i <= savedMessageValues.length - 1; i++) {
						Object savedMessageValue = savedMessageValues[i];
						Object messageValue = messageValues[i];
						if (savedMessageValue == null && messageValue != null) {
							return false;
						}
						if (savedMessageValue != null && messageValue == null) {
							return false;
						}
						if (savedMessageValue != null && messageValue != null) {
							if (savedMessageValue.equals(messageValue)) {
								return true;
							}
						}
					}
				}
			}
		}

		return false;
	}

	/**
	 * 指定されたアクションパスの各種機能クラスの構成が不正でないか判定します。<br>
	 * 不正な場合は例外がスローされます。<br>
	 * @param actionPath アクションパス
	 */
	public static void checkClassStructure(String actionPath) {
		Class<?> constantsClass = getConstantsClass(actionPath);
		Class<? extends AbstractAction<? extends Serializable, AbstractForm<? extends Serializable>>> actionClass = getActionClass(actionPath);
		Class<? extends AbstractForm<? extends Serializable>> formClass = getFormClass(actionPath);
		Class<? extends AbstractLogic<? extends Serializable>> logicClass = getLogicClass(actionPath);
		Class<?> sessionClass = getSessionClass(actionPath);
		Class<?> utilClass = getUtilClass(actionPath);

		/*
		 * クラス存在チェック
		 */
		if (constantsClass == null) {
			throw new StrutsProcessInternalException("FSTR00010", new Object[] { getConstantsClassSimpleName(actionPath) });
		}
		if (actionClass == null) {
			throw new StrutsProcessInternalException("FSTR00010", new Object[] { getActionClassSimpleName(actionPath) });
		}
		if (formClass == null) {
			throw new StrutsProcessInternalException("FSTR00010", new Object[] { getFormClassSimpleName(actionPath) });
		}
		if (logicClass == null) {
			throw new StrutsProcessInternalException("FSTR00010", new Object[] { getLogicClassSimpleName(actionPath) });
		}
		if (sessionClass == null) {
			throw new StrutsProcessInternalException("FSTR00010", new Object[] { getSessionClassSimpleName(actionPath) });
		}
		if (utilClass == null) {
			throw new StrutsProcessInternalException("FSTR00010", new Object[] { getUtilClassSimpleName(actionPath) });
		}

		/*
		 * インタフェース実装チェック	
		 */
		if (!ArrayUtil.contains(ReflectionUtil.getInterfaces(actionClass).toArray(), constantsClass)) {
			throw new StrutsProcessInternalException("FSTR00011", new Object[] { getActionClassSimpleName(actionPath) });
		}
		if (!ArrayUtil.contains(ReflectionUtil.getInterfaces(formClass).toArray(), constantsClass)) {
			throw new StrutsProcessInternalException("FSTR00011", new Object[] { getFormClassSimpleName(actionPath) });
		}
		if (!ArrayUtil.contains(ReflectionUtil.getInterfaces(logicClass).toArray(), constantsClass)) {
			throw new StrutsProcessInternalException("FSTR00011", new Object[] { getLogicClassSimpleName(actionPath) });
		}
		if (!ArrayUtil.contains(ReflectionUtil.getInterfaces(sessionClass).toArray(), constantsClass)) {
			throw new StrutsProcessInternalException("FSTR00011", new Object[] { getSessionClassSimpleName(actionPath) });
		}
		if (!ArrayUtil.contains(ReflectionUtil.getInterfaces(utilClass).toArray(), constantsClass)) {
			throw new StrutsProcessInternalException("FSTR00011", new Object[] { getUtilClassSimpleName(actionPath) });
		}
		if (!ArrayUtil.contains(ReflectionUtil.getInterfaces(sessionClass).toArray(), ScreenSession.class)) {
			throw new StrutsProcessInternalException("FSTR00012", new Object[] { getSessionClassSimpleName(actionPath) });
		}

		/*
		 * アノテーションチェック
		 */
		if (AnnotationUtil.getClassAnnotation(StrutsAction.class, actionClass) == null) {
			throw new StrutsProcessInternalException("FSTR00013", new Object[] { getActionClassSimpleName(actionPath) });
		}
		if (AnnotationUtil.getClassAnnotation(StrutsForm.class, formClass) == null) {
			throw new StrutsProcessInternalException("FSTR00013", new Object[] { getFormClassSimpleName(actionPath) });
		}
		if (AnnotationUtil.getClassAnnotation(StrutsLogic.class, logicClass) == null) {
			throw new StrutsProcessInternalException("FSTR00013", new Object[] { getLogicClassSimpleName(actionPath) });
		}
		if (AnnotationUtil.getClassAnnotation(StrutsSession.class, sessionClass) == null) {
			throw new StrutsProcessInternalException("FSTR00013", new Object[] { getSessionClassSimpleName(actionPath) });
		}
	}

	/**
	 * 指定されたアクションパスのクラスメソッド構成が不正でないか判定します。<br>
	 * 不正な場合は例外がスローされます。<br>
	 * @param actionPath アクションパス
	 */
	public static void checkClassMethodStructure(String actionPath) {
		String methodName = getRequestFunctionMethod(actionPath);
		Method method = getActionMethod(actionPath);

		if (StringUtil.isEmpty(methodName)) {
			return; // メソッド名未指定の場合はディフォルト遷移させるのでここではチェック除外
		}
		if (method == null) {
			throw new StrutsProcessInternalException("FSTR00014", new Object[] { getActionClassSimpleName(actionPath), methodName });
		}
		if (method.getAnnotation(StrutsMethod.class) == null) {
			throw new StrutsProcessInternalException("FSTR00015", new Object[] { getActionClassSimpleName(actionPath), methodName });
		}
	}
}
