package org.phosphoresce.webcore.ext.struts.action;

import java.io.Serializable;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpSession;

import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.phosphoresce.webcore.core.transaction.TransactionUser;
import org.phosphoresce.webcore.ext.struts.StrutsConstants;
import org.phosphoresce.webcore.ext.struts.exception.StrutsSessionTimeoutException;
import org.slf4j.LoggerFactory;

/**
 * セッションコンテナクラス<br>
 * <br>
 * フレームワーク上で制御されるセッションスコープの情報を管理するクラスです。<br>
 * HttpSessionを直接利用することによる情報乱立を回避する為に設けられたクラスで、フレームワーク配下のモジュールは全てのセッションアクセスを当クラスで一元管理します。<br>
 * セッションコンテナクラスのライフサイクルは通常のセッションと同様となります(アプリケーションサーバー側セッションタイムアウト時間依存)。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/19	Kitagawa		新規作成
 * 2012/07/04	Kitagawa		全体的に再構築
 *-->
 */
public final class SessionContainer<L extends TransactionUser> implements Serializable, StrutsConstants {

	/** ロガーオブジェクト */
	private static transient CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(SessionContainer.class));

	/** 破棄済みセッションフラグ */
	private boolean disposed;

	/** セッションID */
	private String sessionId;

	/** ログインユーザー情報 */
	private LoginUser<L> loginUser;

	/** ウィンドウ毎のセッション情報 */
	private Map<String, WindowSessionContainer<L>> windowSession;

	/** ウィンドウセッションタイムアウト監視オブジェクト */
	private transient WindowSessionTimeoutMonitor<L> windowSessionMonitor;

	/**
	 * コンストラクタ<br>
	 * @param sessionId セッションID
	 */
	private SessionContainer(String sessionId) {
		super();
		this.disposed = false;
		this.sessionId = sessionId;
		this.loginUser = null;
		this.windowSession = new HashMap<String, WindowSessionContainer<L>>();
		this.windowSessionMonitor = null;
	}

	/**
	 * ロガーオブジェクトを取得します。<br>
	 * transientフィールドである為、nullとなった場合にインスタンスを生成して提供します。<br>
	 * @return ローがオブジェクト
	 */
	private static CodeConvertLogger getLog() {
		if (log == null) {
			log = new CodeConvertLogger(LoggerFactory.getLogger(SessionContainer.class));
		}
		return log;
	}

	/**
	 * 新規にセッションコンテナをHTTPセッション上に生成します。<br>
	 * @param httpSession HTTPセッションオブジェクト
	 * @return 生成したセッションコンテナオブジェクト
	 */
	static <L extends TransactionUser> SessionContainer<L> createNewSessionContainer(HttpSession httpSession) {
		SessionContainer<L> sessionContainer = new SessionContainer<L>(httpSession.getId());
		httpSession.setAttribute(SESSION_KEY_CONTAINER, sessionContainer);
		getLog().output("FSTR03002");
		return sessionContainer;
	}

	/**
	 * セッション内容をすべて破棄します。<br>
	 * ウィンドウセッション破棄後は各種操作呼び出し時にタイムアウト例外が発生することに注意して下さい。<br>
	 */
	void dispose() {
		synchronized (this) {
			// 既に破棄済みである場合は処理中断
			if (disposed) {
				return;
			}

			// ロギング用情報取得
			String activeSessionId = sessionId;

			// ウィンドウセッションタイムアウトモニタ停止
			if (windowSessionMonitor != null) {
				windowSessionMonitor.halt();
			}

			// ウィンドウセッション破棄処理
			synchronized (windowSession) {
				for (WindowSessionContainer<L> windowSessionInstance : windowSession.values()) {
					windowSessionInstance.dispose();
				}
				windowSession.clear();
			}

			// 各種フィールド初期化
			sessionId = null;
			loginUser = null;
			windowSession = null;
			windowSessionMonitor = null;

			// 破棄済みフラグ設定
			disposed = true;

			getLog().output("FSTR03001", activeSessionId);
		}
	}

	/**
	 * セッションが破棄済みであるか判定します。<br>
	 * @return セッションが破棄済みである場合にtrueを返却
	 */
	public boolean isDisposed() {
		return disposed;
	}

	/**
	 * ログインユーザー情報を取得します。<br>
	 * @return ログインユーザー情報
	 */
	public LoginUser<L> getLoginUser() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return loginUser;
	}

	/**
	 * ログインユーザー情報を設定します。<br>
	 * @param loginUser ログインユーザー情報
	 */
	public void setLoginUser(LoginUser<L> loginUser) {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		this.loginUser = loginUser;
	}

	/**
	 * セッションIDを取得します。<br>
	 * @return セッションID
	 */
	public String getSessionId() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return sessionId;
	}

	/**
	 * ログイン済みであるか判定します。<br>
	 * @return ログイン済みの場合にtrueを返却
	 */
	boolean isLogined() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return loginUser != null;
	}

	/**
	 * セッションが保持しているウィンドウセッションIDを取得します.<br>
	 * @return セッションが保持しているウィンドウセッションID配列
	 */
	String[] getWindwosSessionIds() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		List<String> list = new LinkedList<String>();
		if (windowSession != null) {
			for (String windowId : windowSession.keySet()) {
				list.add(windowId);
			}
		}
		return list.toArray(new String[0]);
	}

	/**
	 * セッションが保持しているウィンドウ単位のセッション数を取得します。<br>
	 * @return セッションが保持しているウィンドウ単位のセッション数
	 */
	int getWindowSessionCount() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return windowSession.size();
	}

	/**
	 * ウィンドウ毎のセッション情報を取得します。<br>
	 * セッション上に指定されたウィンドウIDのウィンドウセッションが存在しない場合はnullが返却されます。<br>
	 * 新規にウィンドウセッション生成が必要な場合は{@link SessionContainer#createNewWindowSession(String)}を利用します。<br>
	 * @param windowId ウィンドウID
	 * @return ウィンドウ毎のセッション情報
	 */
	WindowSessionContainer<L> getWindowSession(String windowId) {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		synchronized (windowSession) {
			if (windowSession == null) {
				return null;
			}
			return windowSession.get(windowId);
		}
	}

	void setWindowSession(WindowSessionContainer<L> windowSessionContainer) {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		synchronized (windowSession) {
			if (windowSession == null) {
				windowSession = new HashMap<String, WindowSessionContainer<L>>();
			}
			windowSession.put(windowSessionContainer.getWindowId(), windowSessionContainer);
		}
	}

	/**
	 * ウィンドウ毎のセッション情報を削除します。<br>
	 * @param windowId ウィンドウID
	 */
	void removeWindowSession(String windowId) {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		synchronized (windowSession) {
			if (windowSession == null) {
				return;
			}
			windowSession.remove(windowId);
		}
	}

	/**
	 * ウィンドウ毎のセッション情報が保持されているか判定します。<br>
	 * @param windowId ウィンドウID
	 * @return ウィンドウ毎のセッション情報が保持されている場合にtrueを返却
	 */
	boolean containsWindowSession(String windowId) {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		return getWindowSession(windowId) != null;
	}

	/**
	 * ウィンドウセッションタイムアウト監視を開始します。<br>
	 */
	void startWindowSessionMonitor() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		if (windowSessionMonitor == null) {
			windowSessionMonitor = new WindowSessionTimeoutMonitor<L>(this);
		}
		if (!windowSessionMonitor.isProcessing()) {
			new Thread(windowSessionMonitor).start();
		}
	}

	/**
	 * ウィンドウセッションタイムアウト監視を停止します。<br>
	 */
	void stopWindowSessionMonitor() {
		// セッション破棄済みの場合はタイムアウト例外スロー
		if (disposed) {
			throw new StrutsSessionTimeoutException();
		}
		if (windowSessionMonitor == null) {
			return;
		}
		windowSessionMonitor.halt();
	}

	/**
	 * クラス情報を文字列で取得します。<br>
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("{");
		builder.append("sessionId=");
		builder.append(sessionId);
		builder.append(", ");
		builder.append("loginUser=");
		builder.append(loginUser);
		builder.append(", ");
		builder.append("windowSession=");
		builder.append(windowSession);
		builder.append("}");
		return builder.toString();
	}
}
