package org.phosphoresce.webcore.ext.struts.action;

import java.io.Serializable;
import java.util.LinkedList;
import java.util.List;

import org.apache.struts.action.ActionMessage;
import org.apache.struts.action.ActionMessages;
import org.phosphoresce.webcore.core.ConfigName;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.config.StringResource;
import org.phosphoresce.webcore.core.config.StringResourceLevel;

/**
 * アプリケーションメッセージコンテナクラス<br>
 * <br>
 * フレームワークにて利用される各種業務メッセージは全て当クラスが管理します。<br>
 * アクションクラスやロジッククラスでのメッセージ情報は当クラスのインスタンスを受け渡すことでメッセージ情報が一元的に管理されます。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2009/12/18	Kitagawa		新規作成
 * 2012/07/06	Kitagawa		全体的に再構築
 * 2013/05/28	Kitagawa		先頭へのメッセージ追加インタフェースを追加
 * 2013/06/18	Kitagawa		メッセージレベル毎のインタフェースを削除(統合してコードによる分岐)
 *-->
 */
public final class MessageContainer implements Serializable {

	/** エラーメッセージ */
	private List<MessageSet> errorMessages = new LinkedList<MessageSet>();

	/** 情報メッセージ */
	private List<MessageSet> informationMessages = new LinkedList<MessageSet>();

	/** 警告メッセージ */
	private List<MessageSet> warningMessages = new LinkedList<MessageSet>();

	/** 連絡メッセージ */
	private List<MessageSet> noticeMessages = new LinkedList<MessageSet>();

	/** デバッグメッセージ */
	private List<MessageSet> debugMessages = new LinkedList<MessageSet>();

	/** トレースメッセージ */
	private List<MessageSet> traceMessages = new LinkedList<MessageSet>();

	/** その他メッセージ */
	private List<MessageSet> otherMessages = new LinkedList<MessageSet>();

	/**
	 * コンストラクタ<br>
	 */
	MessageContainer() {
		super();
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param set メッセージセットオブジェクト
	 */
	public final void addMessage(MessageSet set) {
		if (set == null) {
			return;
		}
		StringResourceLevel level = StringResourceLevel.getDefinitionLevel(set.getCode());
		if (StringResourceLevel.INFO.equals(level)) {
			informationMessages.add(set);
		} else if (StringResourceLevel.ERROR.equals(level)) {
			errorMessages.add(set);
		} else if (StringResourceLevel.WARN.equals(level)) {
			warningMessages.add(set);
		} else if (StringResourceLevel.NOTICE.equals(level)) {
			noticeMessages.add(set);
		} else if (StringResourceLevel.DEBUG.equals(level)) {
			debugMessages.add(set);
		} else if (StringResourceLevel.TRACE.equals(level)) {
			traceMessages.add(set);
		} else {
			otherMessages.add(set);
		}
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param set メッセージセットオブジェクト
	 */
	public final void addMessageOnHead(MessageSet set) {
		if (set == null) {
			return;
		}
		StringResourceLevel level = StringResourceLevel.getDefinitionLevel(set.getCode());
		if (StringResourceLevel.INFO.equals(level)) {
			informationMessages.add(0, set);
		} else if (StringResourceLevel.ERROR.equals(level)) {
			errorMessages.add(0, set);
		} else if (StringResourceLevel.WARN.equals(level)) {
			warningMessages.add(0, set);
		} else if (StringResourceLevel.NOTICE.equals(level)) {
			noticeMessages.add(0, set);
		} else if (StringResourceLevel.DEBUG.equals(level)) {
			debugMessages.add(0, set);
		} else if (StringResourceLevel.TRACE.equals(level)) {
			traceMessages.add(0, set);
		} else {
			otherMessages.add(0, set);
		}
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param row 明細行数
	 * @param code メッセージコード
	 * @param binds バインドオブジェクト
	 */
	public final void addMessage(Integer row, String code, Object... binds) {
		addMessage(new MessageSet(row, code, binds));
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param row 明細行数
	 * @param code メッセージコード
	 */
	public final void addMessage(Integer row, String code) {
		addMessage(new MessageSet(row, code, null));
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param code メッセージコード
	 * @param binds バインドオブジェクト
	 */
	public final void addMessage(String code, Object... binds) {
		addMessage(new MessageSet(null, code, binds));
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param code メッセージコード
	 */
	public final void addMessage(String code) {
		addMessage(new MessageSet(null, code, null));
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param row 明細行数
	 * @param code メッセージコード
	 * @param binds バインドオブジェクト
	 */
	public final void addMessageOnHead(Integer row, String code, Object... binds) {
		addMessageOnHead(new MessageSet(row, code, binds));
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param row 明細行数
	 * @param code メッセージコード
	 */
	public final void addMessageOnHead(int row, String code) {
		addMessageOnHead(new MessageSet(row, code, null));
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param code メッセージコード
	 * @param binds バインドオブジェクト
	 */
	public final void addMessageOnHead(String code, Object... binds) {
		addMessageOnHead(new MessageSet(null, code, binds));
	}

	/**
	 * アプリケーションメッセージを追加します。<br>
	 * @param code メッセージコード
	 */
	public final void addMessageOnHead(String code) {
		addMessageOnHead(new MessageSet(null, code, null));
	}

	/**
	 * 指定されたメッセージセットリストから名前空間通りのアクションメッセージを取得します。<br>
	 * @param targetMessages メッセージリスト
	 * @param namespace 名前空間文字列
	 * @return アクションメッセージオブジェクト
	 */
	private final ActionMessages getActionMessages(List<MessageSet> targetMessages, String namespace) {
		ActionMessages messages = new ActionMessages();
		if (targetMessages != null) {
			for (MessageSet set : targetMessages) {
				String string = "";
				if (set.getRowNumber() != null) {
					string = StringResource.getString("FCRV00001", set.getRowNumber(), StringResource.getString(set.getCode(), set.getBinds()));
				} else {
					string = StringResource.getString(set.getCode(), set.getBinds());
				}
				ActionMessage message = null;
				if (Config.getBoolean(ConfigName.COMMON_MESSAGE_SHOWCODE)) {
					message = new ActionMessage(namespace + ".general", "[" + set.getCode() + "] " + string);
				} else {
					message = new ActionMessage(namespace + ".general", string);
				}
				messages.add(namespace, message);
			}
		}
		return messages;
	}

	/**
	 * すべてのレベルのメッセージを取得します。<br>
	 * @return アクションメッセージ
	 */
	final ActionMessages getAllActionMessages() {
		final ActionMessages messages = new ActionMessages();
		messages.add(getInformationActionMessages());
		messages.add(getErrorActionMessages());
		messages.add(getWarningActionMessages());
		messages.add(getNoticeActionMessages());
		messages.add(getDebugActionMessages());
		messages.add(getTraceActionMessages());
		messages.add(getOtherActionMessages());
		return messages;
	}

	/**
	 * 情報アクションメッセージを取得します。<br>
	 * @return 情報アクションメッセージ
	 */
	final ActionMessages getInformationActionMessages() {
		return getActionMessages(informationMessages, "informations");
	}

	/**
	 * エラーアクションメッセージを取得します。<br>
	 * @return エラーアクションメッセージ
	 */
	final ActionMessages getErrorActionMessages() {
		return getActionMessages(errorMessages, "errors");
	}

	/**
	 * 警告アクションメッセージを取得します。<br>
	 * @return 警告アクションメッセージ
	 */
	final ActionMessages getWarningActionMessages() {
		return getActionMessages(warningMessages, "warnings");
	}

	/**
	 * 連絡アクションメッセージを取得します。<br>
	 * @return 警告アクションメッセージ
	 */
	final ActionMessages getNoticeActionMessages() {
		return getActionMessages(noticeMessages, "notices");
	}

	/**
	 * デバッグアクションメッセージを取得します。<br>
	 * @return デバッグアクションメッセージ
	 */
	final ActionMessages getDebugActionMessages() {
		return getActionMessages(debugMessages, "debugs");
	}

	/**
	 * トレースアクションメッセージを取得します。<br>
	 * @return トレースアクションメッセージ
	 */
	final ActionMessages getTraceActionMessages() {
		return getActionMessages(traceMessages, "traces");
	}

	/**
	 * その他アクションメッセージを取得します。<br>
	 * @return その他アクションメッセージ
	 */
	final ActionMessages getOtherActionMessages() {
		return getActionMessages(otherMessages, "others");
	}
}
