package org.phosphoresce.webcore.ext.struts.action;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.struts.action.ActionForward;
import org.phosphoresce.lib.commons.util.ReflectionUtil;
import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.ConfigName;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.transaction.TransactionUser;
import org.phosphoresce.webcore.ext.struts.StrutsConfigNames;
import org.phosphoresce.webcore.ext.struts.StrutsConstants;
import org.phosphoresce.webcore.ext.struts.exception.StrutsProcessInternalException;
import org.phosphoresce.webcore.ext.struts.util.StrutsUtil;

/**
 * サーブレット遷移先情報保持クラス<br>
 * <br>
 * 当クラスはStrutsのActionForwardを生成するための処理中の情報を保持するクラスとなります。<br>
 * フレームワーク側で遷移制御情報として必要な情報のみを管理します。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/19	Kitagawa		新規作成
 * 2012/07/09	Kitagawa		全体的に再構築
 *-->
 */
public final class Forward implements Serializable, StrutsConstants {

	/** 遷移先パス */
	private String path;

	/** リダイレクトフラグ */
	private boolean redirect;

	/** パラメータマップ */
	private Map<String, List<String>> parameters;

	/**
	 * コンストラクタ<br>
	 * @param path 遷移先パス
	 * @param redirect リダイレクトフラグ
	 */
	public Forward(String path, boolean redirect) {
		super();
		this.redirect = redirect;
		this.parameters = new HashMap<String, List<String>>();
		setPath(path);
	}

	/**
	 * コンストラクタ<br>
	 * @param path 遷移先パス
	 */
	public Forward(String path) {
		this(path, false);
	}

	/**
	 * コンストラクタ<br>
	 */
	public Forward() {
		this("/", false);
	}

	/**
	 * 指定されたリクエストパスから<br>
	 * クエリ文字列は除外されます。<br>
	 * @param path リクエストパス
	 * @return フォワードとして有効なリクエストパス
	 */
	private String createSimplePath(String path) {
		if (StringUtil.isEmpty(path)) {
			return "/";
		} else if (path.indexOf("?") >= 0) {
			path = path.substring(0, path.indexOf("?"));
			return path.length() <= 0 ? "/" : path;
		} else {
			return path;
		}
	}

	/**
	 * 指定されたパスから有効なパラメータマップオブジェクトを生成します。<br>
	 * @param path リクエストパス
	 * @return 指定されたパスのクエリ文字列から生成されたパラメータマップオブジェクト
	 */
	private Map<String, List<String>> createParameterFromReuqestPath(String path) {
		Map<String, List<String>> parameter = new HashMap<String, List<String>>();
		if (!StringUtil.isEmpty(path) && path.indexOf("?") >= 0) {
			path = path.substring(path.indexOf("?") + 1);
			String[] queries = path.split("&");
			for (String query : queries) {
				String[] querySet = query.split("=");
				if (querySet.length <= 0) {
					continue;
				}
				String key = querySet[0];
				String value = querySet.length <= 1 ? "" : querySet[1];
				if (!parameter.containsKey(key)) {
					parameter.put(key, new LinkedList<String>());
				}
				parameter.get(key).add(value);
			}
		}
		return parameter;
	}

	/**
	 * 指定されたオブジェクトからString型にキャストします。<br>
	 * @param value オブジェクト
	 * @return 文字列オブジェクト
	 */
	private String castObjectToString(Object value) {
		if (value != null) {
			return String.valueOf(value);
		} else {
			return EMPTY_STRING;
		}
	}

	/**
	 * 指定されたオブジェクト配列から文字列配列にキャストします。<br>
	 * @param values オブジェクト配列
	 * @return 文字列配列
	 */
	private String[] castObjectArrayToStringArray(Object[] values) {
		List<String> list = new LinkedList<String>();
		if (values != null) {
			for (Object value : values) {
				list.add(castObjectToString(value));
			}
		}
		return list.toArray(new String[0]);
	}

	/**
	 * 指定されたパラメータマップからリクエストクエリ文字列を生成します。<br>
	 * @param parameters パラメータマップ
	 * @return リクエストクエリ文字列
	 */
	private String createQueryString(Map<String, List<String>> parameters) {
		StringBuilder builder = new StringBuilder();
		if (parameters != null) {
			for (String name : parameters.keySet()) {
				List<String> list = parameters.get(name);
				for (String s : list) {
					String value = StringUtil.encodeURL(s, Config.getString(ConfigName.COMMON_CHARSET_DEFAULT));
					builder.append(name);
					builder.append("=");
					builder.append(value);
					builder.append("&");
				}
			}
		}
		if (builder.length() <= 0) {
			return EMPTY_STRING;
		} else {
			return builder.substring(0, builder.length() - 1);
		}
	}

	/**
	 * 遷移先パスを取得します。<br>
	 * @return 遷移先パス
	 */
	public String getPath() {
		return path;
	}

	/**
	 * 遷移先パスを設定します。<br>
	 * @param path 遷移先パス
	 * @return 自身のクラスインスタンス
	 */
	public Forward setPath(String path) {
		this.path = createSimplePath(path);
		this.parameters.putAll(createParameterFromReuqestPath(path));
		return this;
	}

	/**
	 * リダイレクトフラグを取得します。<br>
	 * @return リダイレクトフラグ
	 */
	public boolean isRedirect() {
		return redirect;
	}

	/**
	 * リダイレクトフラグを設定します。<br>
	 * @param redirect リダイレクトフラグ
	 * @return 自身のクラスインスタンス
	 */
	public Forward setRedirect(boolean redirect) {
		this.redirect = redirect;
		return this;
	}

	/**
	 * パラメータリストを取得します。<br>
	 * ここで返却されるリストに対する変更操作は行えません。<br>
	 * @param name パラメータキー
	 * @return パラメータリスト
	 */
	public List<String> getParameters(String name) {
		List<String> list = (List<String>) parameters.get(name);
		return Collections.unmodifiableList(list == null ? new LinkedList<String>() : list);
	}

	/**
	 * パラメータを取得します。<br>
	 * 同一キーで複数のパラメータが設定されている場合は先頭のパラメータが返却されます。<br>
	 * @param name パラメータキー
	 * @return パラメータ値
	 */
	public String getParameter(String name) {
		List<String> list = getParameters(name);
		return list.size() > 0 ? list.get(0) : EMPTY_STRING;
	}

	/**
	 * パラメータを追加します。<br>
	 * @param name パラメータキー
	 * @param value パラメータ値
	 * @return 自身のクラスインスタンス
	 */
	public Forward setParameter(String name, String value) {
		if (!parameters.containsKey(name)) {
			parameters.put(name, new LinkedList<String>());
		}
		parameters.get(name).add(StringUtil.nvl(value));
		return this;
	}

	/**
	 * パラメータを追加します。<br>
	 * @param name パラメータキー
	 * @param values パラメータ値
	 * @return 自身のクラスインスタンス
	 */
	public Forward setParameters(String name, String[] values) {
		if (!parameters.containsKey(name)) {
			parameters.put(name, new LinkedList<String>());
		}
		if (values != null) {
			for (String value : values) {
				setParameter(name, value);
			}
		}
		return this;
	}

	/**
	 * パラメータを追加します。<br>
	 * @param name パラメータキー
	 * @param value パラメータ値
	 * @return 自身のクラスインスタンス
	 */
	public Forward setParameter(String name, Integer value) {
		setParameter(name, castObjectToString(value));
		return this;
	}

	/**
	 * パラメータを追加します。<br>
	 * @param name パラメータキー
	 * @param values パラメータ値
	 * @return 自身のクラスインスタンス
	 */
	public Forward setParameters(String name, Integer[] values) {
		setParameters(name, castObjectArrayToStringArray(values));
		return this;
	}

	/**
	 * パラメータを追加します。<br>
	 * @param name パラメータキー
	 * @param value パラメータ値
	 * @return 自身のクラスインスタンス
	 */
	public Forward setParameter(String name, Long value) {
		setParameter(name, castObjectToString(value));
		return this;
	}

	/**
	 * パラメータを追加します。<br>
	 * @param name パラメータキー
	 * @param values パラメータ値
	 * @return 自身のクラスインスタンス
	 */
	public Forward setParameters(String name, Long[] values) {
		setParameters(name, castObjectArrayToStringArray(values));
		return this;
	}

	/**
	 * パラメータを追加します。<br>
	 * @param name パラメータキー
	 * @param value パラメータ値
	 * @return 自身のクラスインスタンス
	 */
	public Forward setParameter(String name, Boolean value) {
		setParameter(name, castObjectToString(value));
		return this;
	}

	/**
	 * パラメータを追加します。<br>
	 * @param name パラメータキー
	 * @param values パラメータ値
	 * @return 自身のクラスインスタンス
	 */
	public Forward setParameters(String name, Boolean[] values) {
		setParameters(name, castObjectArrayToStringArray(values));
		return this;
	}

	/**
	 * フォームオブジェクトの内容を全てパラメータとして設定します。<br>
	 * 尚、パラメータとして設定される対象のフィールドは以下の型に限定されます。<br>
	 * <ul>
	 * <li>String</li>
	 * <li>Integer</li>
	 * <li>Long</li>
	 * <li>Boolean</li>
	 * </ul>
	 * @param form フォームオブジェクト
	 * @return 自身のクラスインスタンス
	 */
	public Forward setParameter(AbstractForm<?> form) {
		try {
			if (form == null) {
				return this;
			} else {
				for (Field field : ReflectionUtil.getFields(form.getClass())) {
					field.setAccessible(true);
					Object object = field.get(form);
					if (object instanceof String) {
						setParameter(field.getName(), object == null ? null : object.toString());
					}
					if (object instanceof String[]) {
						setParameters(field.getName(), object == null ? null : (String[]) object);
					}
					if (object instanceof Integer) {
						setParameter(field.getName(), object == null ? null : object.toString());
					}
					if (object instanceof Integer[]) {
						setParameters(field.getName(), object == null ? null : (Integer[]) object);
					}
					if (object instanceof Long) {
						setParameter(field.getName(), object == null ? null : object.toString());
					}
					if (object instanceof Long[]) {
						setParameters(field.getName(), object == null ? null : (Long[]) object);
					}
					if (object instanceof Boolean) {
						setParameter(field.getName(), object == null ? null : object.toString());
					}
					if (object instanceof Boolean[]) {
						setParameters(field.getName(), object == null ? null : (Boolean[]) object);
					}
				}
				return this;
			}
		} catch (Throwable e) {
			throw new StrutsProcessInternalException("FSTR05001", e);
		}
	}

	/**
	 * リクエストクエリ文字列を取得します。<br>
	 * @return リクエストクエリ文字列
	 */
	public String getQueryString() {
		return createQueryString(parameters);
	}

	/**
	 * サーブレットパスを取得します。<br>
	 * @return サーブレットパス
	 */
	public String getServletPath() {
		String query = getQueryString();
		return path + (StringUtil.isEmpty(query) ? EMPTY_STRING : "?" + query);
	}

	/**
	 * Strutsアクションフォワードを生成します。<br>
	 * @param <L> ログインユーザオブジェクト
	 * @param <F> アクションフォームオブジェクト
	 * @param context コンテキストオブジェクト
	 * @return Strutsアクションフォワード
	 */
	public <L extends TransactionUser, F extends AbstractForm<L>> ActionForward createActionForward(ActionContext<L, F> context) {
		String actionPath = StrutsUtil.getRequestActionPath(context.getServletRequest(), context.getModuleConfig());

		// リクエストクエリ文字列生成
		String query = getQueryString();
		query = StringUtil.isEmpty(query) ? "" : "?" + query;

		// パスが未指定の場合はサーブレットパスとする
		if (StringUtil.isEmpty(path)) {
			String functionId = StrutsUtil.getRequestFunction(actionPath);
			ActionForward actionForward = new ActionForward();
			actionForward.setRedirect(redirect);
			actionForward.setPath(context.getServletRequest().getServletPath() + "/" + functionId + "/" + query);
			return actionForward;
		}

		// JSP基底パスから指定されている場合はそのままのパスで遷移
		if (path.startsWith(Config.getString(ConfigName.COMMON_JSP_PATH))) {
			ActionForward actionForward = new ActionForward();
			actionForward.setRedirect(redirect);
			actionForward.setPath(path);
			return actionForward;
		}

		// スラッシュから始まるパスの場合はパスに対応するアクションサーブレットに遷移
		if (path.startsWith("/")) {
			String function = StrutsUtil.getRequestFunction(actionPath);
			String action = StrutsUtil.getRequestFunctionMethod(getPath());
			ActionForward actionForward = new ActionForward();
			actionForward.setRedirect(redirect);
			actionForward.setPath(context.getServletRequest().getServletPath() + "/" + function + "/" + action + query);
			return actionForward;
		}

		// 上記までに該当しないパスはJSPパスとして遷移
		{
			String functionId = StrutsUtil.getRequestFunction(actionPath);
			ActionForward actionForward = new ActionForward();
			actionForward.setRedirect(redirect);
			actionForward.setPath(Config.getString(ConfigName.COMMON_JSP_PATH) + "/" + Config.getString(StrutsConfigNames.STRUTS_JSP_FUNCTION) + "/" + functionId + "/" + path);
			return actionForward;
		}
	}

	/**
	 * クラス情報を文字列で取得します。<br>
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("{");
		builder.append("path=");
		builder.append(path);
		builder.append(", ");
		builder.append("redirect=");
		builder.append(redirect);
		builder.append(", ");
		builder.append("parameters=");
		builder.append(parameters);
		builder.append("}");
		return builder.toString();
	}
}
