package org.phosphoresce.webcore.ext.struts.action;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * アクセス状況管理クラス<br>
 * <br>
 * アプリケーションに対するアクセス状況をリアルタイムに参照する為のクラスです。<br>
 * アクセス状況はアプリケーション起動中のみ保持され、アプリケーション終了のタイミングで情報は破棄されます。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2013/05/01	Kitagawa		新規作成
 *-->
 */
public final class AccessStatus implements Serializable {

	/** クラスインスタンス */
	private static AccessStatus instance;

	/** アクセス状況データ */
	private Map<String, AccessStatusEntity> data;

	/**
	 * アクセス状況管理エンティティ<br>
	 * 
	 * @author Kitagawa<br>
	 * 
	 *<!--
	 * 更新日		更新者			更新内容
	 * 2013/05/01	Kitagawa		新規作成
	 *-->
	 */
	public class AccessStatusEntity implements Serializable, Cloneable {

		/** ユーザーコード */
		private String userCode;

		/** リクエストカウンタ */
		private BigDecimal requestCount;

		/** 最終アクセスURL */
		private String lastAccessUrl;

		/** 最終アクセス時刻 */
		private Date lastAccessTime;

		/**
		 * コンストラクタ<br>
		 */
		public AccessStatusEntity() {
			super();
			this.userCode = "";
			this.requestCount = BigDecimal.ZERO;
			this.lastAccessUrl = null;
			this.lastAccessTime = null;
		}

		/**
		 * ユーザーコードを取得します。<br>
		 * @return ユーザーコード
		 */
		public String getUserCode() {
			return userCode;
		}

		/**
		 * ユーザーコードを設定します。<br>
		 * @param userCode ユーザーコード
		 */
		public void setUserCode(String userCode) {
			this.userCode = userCode;
		}

		/**
		 * リクエストカウンタを取得します。<br>
		 * @return リクエストカウンタ
		 */
		public BigDecimal getRequestCount() {
			return requestCount;
		}

		/**
		 * リクエストカウンタをインクリメントします。<br>
		 */
		public void incrementRequestCount() {
			this.requestCount = this.requestCount.add(new BigDecimal("1"));
		}

		/**
		 * 最終アクセスURLを取得します。<br>
		 * @return 最終アクセスURL
		 */
		public String getLastAccessUrl() {
			return lastAccessUrl;
		}

		/**
		 * 最終アクセスURLを設定します。<br>
		 * @param lastAccessUrl 最終アクセスURL
		 */
		public void setLastAccessUrl(String lastAccessUrl) {
			this.lastAccessUrl = lastAccessUrl;
		}

		/**
		 * 最終アクセス時刻を取得します。<br>
		 * @return 最終アクセス時刻
		 */
		public Date getLastAccessTime() {
			return lastAccessTime;
		}

		/**
		 * 最終アクセス時刻を設定します。<br>
		 * @param lastAccessTime 最終アクセス時刻
		 */
		public void setLastAccessTime(Date lastAccessTime) {
			this.lastAccessTime = lastAccessTime;
		}

		/**
		 * 最終アクセスからの経過時間(分)を取得します。<br>
		 * @return 最終アクセスからの経過時間(分)
		 */
		public long getElapsedMinuteOfLastAccess() {
			if (this.lastAccessTime == null) {
				return 0;
			} else {
				return (System.currentTimeMillis() - lastAccessTime.getTime()) / 1000 / 60;
			}
		}

		/**
		 * クローンオブジェクトを生成します。<br>
		 * @return クローンオブジェクト
		 */
		public AccessStatusEntity clone() {
			AccessStatusEntity clone = new AccessStatusEntity();
			clone.userCode = userCode;
			clone.requestCount = new BigDecimal(this.requestCount.toString());
			clone.lastAccessUrl = this.lastAccessUrl;
			clone.lastAccessTime = this.lastAccessTime == null ? null : (Date) this.lastAccessTime.clone();
			return clone;
		}
	}

	/**
	 * コンストラクタ<br>
	 */
	private AccessStatus() {
		super();
		this.data = new HashMap<String, AccessStatus.AccessStatusEntity>();
	}

	/**
	 * クラスインスタンスを取得します。<br>
	 * @return クラスインスタンス
	 */
	public static AccessStatus instance() {
		if (instance == null) {
			instance = new AccessStatus();
		}
		return instance;
	}

	/**
	 * 指定されたユーザーのステータスを更新します。<br>
	 * @param userCode ユーザーコード
	 * @param url アクセスURL
	 * @param time アクセス時刻
	 */
	public void update(String userCode, String url, Date time) {
		AccessStatusEntity entity = null;
		if (!data.containsKey(userCode)) {
			data.put(userCode, new AccessStatusEntity());
		}
		entity = data.get(userCode);
		entity.setUserCode(userCode);
		entity.setLastAccessUrl(url);
		entity.setLastAccessTime(time);
		entity.incrementRequestCount();
	}

	/**
	 * アクセスステータス情報を取得します。<br>
	 * ここで提供される情報は実際に管理される情報のクラスインスタンスとは異なるインスタンスとして提供されます。<br>
	 * ソート順序は最終アクセス時刻の降順となります。<br>
	 * @return アクセスステータス情報
	 */
	public List<AccessStatusEntity> getAccessInfo() {
		List<AccessStatusEntity> entities = new LinkedList<AccessStatusEntity>();
		for (AccessStatusEntity entity : data.values()) {
			entities.add(entity.clone());
		}
		Collections.sort(entities, new Comparator<AccessStatusEntity>() {
			@Override
			public int compare(AccessStatusEntity value1, AccessStatusEntity value2) {
				if (value1 == null && value2 == null) {
					return 0;
				} else if (value1 != null && value2 == null) {
					return -1;
				} else if (value1 == null && value2 != null) {
					return 1;
				} else {
					Date time1 = value1.lastAccessTime;
					Date time2 = value2.lastAccessTime;
					if (time1 == null && time2 == null) {
						return 0;
					} else if (time1 != null && time2 == null) {
						return -1;
					} else if (time1 == null && time2 != null) {
						return 1;
					} else {
						return time2.compareTo(time1);
					}
				}
			}
		});
		return entities;
	}
}
