package org.phosphoresce.webcore.ext.smartlayout.model;

import java.util.LinkedList;
import java.util.List;

import javax.servlet.jsp.PageContext;

import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.phosphoresce.lib.commons.util.AnnotationUtil;
import org.phosphoresce.lib.commons.util.FileUtil;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.phosphoresce.webcore.ext.smartlayout.SmartLayoutConstants;
import org.phosphoresce.webcore.ext.smartlayout.SmartLayoutEnvironment;
import org.phosphoresce.webcore.ext.smartlayout.exception.SmartLayoutTaglibProcessException;
import org.phosphoresce.webcore.ext.smartlayout.taglib.annotation.SmartTagModelConfig;
import org.phosphoresce.webcore.ext.smartlayout.taglib.util.SmartTagUtil;
import org.slf4j.LoggerFactory;

/**
 * スマートタグモデル抽象クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2013/03/13	Kitagawa		新規作成
 *-->
 */
public abstract class SmartTagModel implements SmartLayoutConstants {

	/** ロガーオブジェクト */
	protected CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(getClass()));

	/** アクティブページコンテキストオブジェクト */
	private PageContext activePageContext;

	/** 親モデルオブジェクト */
	private SmartTagModel parent;

	/** 子モデルオブジェクトリスト */
	private List<SmartTagModel> childs;

	/** テンプレート手動出力フラグ */
	private boolean manualOutput;

	/**
	 * コンストラクタ<br>
	 */
	public SmartTagModel() {
		super();
		this.activePageContext = null;
		this.parent = null;
		this.childs = new LinkedList<SmartTagModel>();
		this.manualOutput = false;
	}

	/**
	 * アクティブページコンテキストオブジェクトを取得します。<br>
	 * @return アクティブページコンテキストオブジェクト
	 */
	public final PageContext getActivePageContext() {
		return activePageContext;
	}

	/**
	 * 親モデルオブジェクトを取得します。<br>
	 * @return 親モデルオブジェクト
	 */
	public SmartTagModel getParent() {
		return parent;
	}

	/**
	 * 子モデルオブジェクトを取得します。<br>
	 * @return 子モデルオブジェクト
	 */
	public final List<SmartTagModel> getChilds() {
		return childs;
	}

	/**
	 * 子モデルオブジェクトを追加します。<br>
	 * @param model 子モデルオブジェクト
	 */
	public final void addChild(SmartTagModel model) {
		handleAddChild(model);
		model.parent = this;
		childs.add(model);
	}

	/**
	 * テンプレート手動出力フラグを取得します。<br>
	 * @return テンプレート手動出力フラグ
	 */
	public boolean isManualOutput() {
		return manualOutput;
	}

	/**
	 * テンプレート手動出力フラグを設定します。<br>
	 * @param manualOutput テンプレート手動出力フラグ
	 */
	public void setManualOutput(boolean manualOutput) {
		this.manualOutput = manualOutput;
	}

	/**
	 * モデル階層妥当性をチェックします。<br>
	 * @throws Throwable 不正な階層構造の場合にスローされます
	 */
	protected void checkModelHierarchy() throws Throwable {
		/* サブクラスで必要に応じて実装を行います */
	}

	/**
	 * モデルフィールドの補正処理を行います。<br>
	 * @param model 子モデルオブジェクト
	 */
	protected void validateFields() {
		/* サブクラスで必要に応じて実装を行います */
	}

	/**
	 * 子モデルオブジェクト追加時のハンドラ処理を行います。<br>
	 * @param model 子モデルオブジェクト
	 */
	protected void handleAddChild(SmartTagModel model) {
		/* サブクラスで必要に応じて実装を行います */
	}

	/**
	 * タグテンプレートパスを取得します。<br>
	 * @param name テンプレート名
	 * @return タグテンプレートパス
	 * @deprecated Velocityテンプレート内でのparseで利用することを目的としていてJavaからの利用は禁止します
	 */
	@Deprecated
	public final String getTemplatePath(String name) {
		return name == null ? null : FileUtil.connectPath(SMART_LAYOUT_VELOCITY_TEMPLATE_PATH, name);
	}

	/**
	 * 開始タグテンプレートパスを取得します。<br>
	 * @return 開始タグテンプレートパス
	 * @deprecated Velocityテンプレート内でのparseで利用することを目的としていてJavaからの利用は禁止します
	 */
	@Deprecated
	public final String getStartTemplatePath() {
		SmartTagModelConfig modelConfig = AnnotationUtil.getClassAnnotation(SmartTagModelConfig.class, this.getClass());
		String startTemplateName = modelConfig == null ? null : modelConfig.startTemplate();
		return getTemplatePath(startTemplateName);
	}

	/**
	 * 終了タグテンプレートパスを取得します。<br>
	 * @return 終了タグテンプレートパス
	 * @deprecated Velocityテンプレート内でのparseで利用することを目的としていてJavaからの利用は禁止します
	 */
	@Deprecated
	public final String getEndTemplatePath() {
		SmartTagModelConfig modelConfig = AnnotationUtil.getClassAnnotation(SmartTagModelConfig.class, this.getClass());
		String endTemplateName = modelConfig == null ? null : modelConfig.endTemplate();
		return getTemplatePath(endTemplateName);
	}

	/**
	 * ページコンテンツにモデル情報を出力します。<br>
	 * @param pageContext ページコンテンツオブジェクト
	 * @throws Throwable 正常に出力処理が行えなかった場合にスローされます
	 */
	public final void output(PageContext pageContext) throws Throwable {
		try {
			/*
			 * アクティブページコンテキストオブジェクト設定
			 */
			activePageContext = pageContext;

			/*
			 * 階層構造妥当性チェック
			 */
			checkModelHierarchy();

			/*
			 * モデルフィールド補正処理
			 */
			validateFields();

			/*
			 * テンプレートコンテキスト生成
			 */
			VelocityContext templateContext = new VelocityContext();
			templateContext.put(VELOCITY_KEY_UTIL, new SmartTagUtil());
			templateContext.put(VELOCITY_KEY_MODEL, this);
			templateContext.put(VELOCITY_KEY_FORM, pageContext.getRequest().getAttribute("form")); // TODO フォーム内容のアトリビュート名を要検討
			//context.put(VELOCITY_KEY_FORM, getPageContext().getAttribute("form")); // TODO フォーム内容の設定スコープを要検討

			/*
			 * モデル情報取得
			 */
			SmartTagModelConfig modelConfig = AnnotationUtil.getClassAnnotation(SmartTagModelConfig.class, this.getClass());
			String startTemplateName = modelConfig == null ? null : modelConfig.startTemplate();
			String endTemplateName = modelConfig == null ? null : modelConfig.endTemplate();
			Template startTemplate = startTemplateName == null ? null : SmartLayoutEnvironment.getTemplate(startTemplateName);
			Template endTemplate = endTemplateName == null ? null : SmartLayoutEnvironment.getTemplate(endTemplateName);
			if (modelConfig == null) {
				throw new SmartLayoutTaglibProcessException("FSML00006", new Object[] { this.getClass().getSimpleName(), SmartTagModelConfig.class.getSimpleName() });
			}
			if (startTemplate == null) {
				throw new SmartLayoutTaglibProcessException("FSML00015", new Object[] { startTemplateName });
			}
			if (endTemplate == null) {
				throw new SmartLayoutTaglibProcessException("FSML00015", new Object[] { endTemplateName });
			}

			/*
			 * 開始タグ出力処理
			 */
			try {
				startTemplate.merge(templateContext, pageContext.getOut());
			} catch (Throwable e) {
				throw new SmartLayoutTaglibProcessException("FSML00007", new Object[] { startTemplateName }, e);
			}

			/*
			 * 子タグ出力処理
			 */
			for (SmartTagModel child : childs) {
				if (!child.isManualOutput()) {
					child.output(pageContext);
				}
			}

			/*
			 * 終了タグ出力処理
			 */
			try {
				endTemplate.merge(templateContext, pageContext.getOut());
			} catch (Throwable e) {
				throw new SmartLayoutTaglibProcessException("FSML00008", new Object[] { endTemplateName }, e);
			}
		} finally {
			/*
			 * アクティブページコンテキストオブジェクト初期化
			 */
			activePageContext = null;
		}
	}
}
