package org.phosphoresce.webcore.ext.jdbc;

import java.sql.Driver;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Enumeration;

import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.phosphoresce.webcore.ext.jdbc.transaction.JDBCTransactionManager;
import org.slf4j.LoggerFactory;

/**
 * データベース接続アドオン環境初期化クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2011/12/24	Kitagawa		新規作成
 *-->
 */
public class JDBCConfiguration implements JDBCConstants {

	/** ロガーオブジェクト */
	private static CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(JDBCConfiguration.class));

	/**
	 * コンストラクタ<br>
	 */
	private JDBCConfiguration() {
		super();
	}

	/**
	 * データベース接続アドオンの初期化処理を行います。<br>
	 * 当初期化処理はアプリケーション共通初期化処理が完了されていることが前提となります。<br>
	 */
	public synchronized static void configure() {
		try {
			log.output("FJDB00001");

			// データベース接続可能
			JDBCEnvironment.setEnabledDatabase(true);

			// ディフォルト定義接続確認
			checkConnection(null);

			// 名前空間定義接続確認
			String[] namespaces = Config.getStrings(JDBCConfigName.JDBC_CONNECTION_NAMESPACES);
			for (String namespace : namespaces) {
				checkConnection(namespace);
			}

			log.output("FJDB00008");
		} catch (Throwable e) {
			log.output(e, "FJDB00018");

			// データベース接続不可
			JDBCEnvironment.setEnabledDatabase(false);
		}
	}

	/**
	 * データベース接続アドオンの破棄処理を実行します。<br>
	 * @throws Throwable 終了処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void destroy() throws Throwable {
		Enumeration<Driver> drivers = DriverManager.getDrivers();
		while (drivers.hasMoreElements()) {
			Driver driver = drivers.nextElement();
			try {
				DriverManager.deregisterDriver(driver);
				log.output("FJDB00025", driver == null ? "null" : driver.getClass().getName());
			} catch (SQLException e) {
				log.output(e, "FJDB00020", driver == null ? "null" : driver.getClass().getName());
			}
		}
		log.output("FJDB00009");
	}

	/**
	 * 指定された名前空間のJDBC接続をチェックします。<br>
	 * @param namespace 名前空間
	 */
	private static void checkConnection(String namespace) {
		try {
			boolean useDatasource = false;
			JDBCTransactionManager manager = new JDBCTransactionManager(namespace);
			manager.open();
			manager.close();

			if (StringUtil.isEmpty(namespace)) {
				if (useDatasource) {
					log.output("FJDB00002");
				} else {
					log.output("FJDB00003");
				}
			} else {
				if (useDatasource) {
					log.output("FJDB00006", namespace);
				} else {
					log.output("FJDB00007", namespace);
				}
			}
		} catch (Throwable e) {
			if (StringUtil.isEmpty(namespace)) {
				log.output(e, "FJDB00018");
			} else {
				log.output(e, "FJDB00019", namespace);
			}

			// データベース接続不可
			JDBCEnvironment.setEnabledDatabase(false);
		}
	}
}
