package org.phosphoresce.webcore.ext.jasper.export;

import java.util.LinkedList;
import java.util.List;

import net.sf.jasperreports.engine.JRDataSource;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRExporter;
import net.sf.jasperreports.engine.JRExporterParameter;
import net.sf.jasperreports.engine.JRPrintPage;
import net.sf.jasperreports.engine.JasperCompileManager;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRBeanCollectionDataSource;

import org.phosphoresce.webcore.ext.jasper.exception.JasperCriticalException;

/**
 * JasperReports複数ページ帳票出力クラス<br>
 * <br>
 * 複数のJasperReports定義体から帳票出力を行うクラスです。<br>
 * 同一定義体であっても結合処理を伴う場合は当クラスを利用します。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/07/06	Kitagawa		新規作成
 *-->
 */
public class JasperMultipageWriter extends JasperWriter {

	/** JasperReports帳票定義パス */
	private List<String> jrxmlPaths;

	/** JasperReports帳票データコンテナオブジェクト */
	private List<JasperContainer> containers;

	/**
	 * コンストラクタ<br>
	 */
	public JasperMultipageWriter() {
		super();
		this.jrxmlPaths = new LinkedList<String>();
		this.containers = new LinkedList<JasperContainer>();
	}

	/**
	 * JasperReports出力定義情報を追加します。<br>
	 * @param jrxmlPath JasperReports帳票定義パス
	 * @param container JasperReports帳票データコンテナオブジェクト
	 */
	public void addJasperReports(String jrxmlPath, JasperContainer container) {
		if (jrxmlPath == null) {
			throw new NullPointerException("jrxmlPaths");
		}
		if (container == null) {
			throw new NullPointerException("containers");
		}
		this.jrxmlPaths.add(jrxmlPath);
		this.containers.add(container);
	}

	/**
	 * コンパイル済みJasperReports帳票定義オブジェクトを生成します。<br>
	 * @return JasperReports帳票定義オブジェクト
	 * @throws JRException 正常にJasperReports帳票定義オブジェクトを生成できなかった場合にスローされます
	 */
	private List<JasperReport> createJasperReport() throws JRException {
		List<JasperReport> list = new LinkedList<JasperReport>();
		for (String jrxmlPath : jrxmlPaths) {
			if (jrxmlPath.endsWith(".jasper")) {
				list.add(null);
			} else {
				list.add(JasperCompileManager.compileReport(jrxmlPath));
			}
		}
		return list;
	}

	/**
	 * JasperReportsデータソースオブジェクトを生成します。<br>
	 * @return JasperReportsデータソースオブジェクト
	 */
	private List<JRDataSource> createJRDataSource() {
		List<JRDataSource> list = new LinkedList<JRDataSource>();
		for (JasperContainer container : containers) {
			list.add(new JRBeanCollectionDataSource(container.getRecords()));
		}
		return list;
	}

	/**
	 * JasperReports出力オブジェクトを生成します。<br>
	 * @return JasperReports出力オブジェクト
	 * @throws JRException 正常にJasperReports出力オブジェクトを生成できなかった場	合にスローされます
	 */
	private List<JasperPrint> createJasperPrint() throws JRException {
		List<JasperPrint> list = new LinkedList<JasperPrint>();
		List<JasperReport> reports = createJasperReport();
		List<JRDataSource> sources = createJRDataSource();
		int count = 0;
		for (JasperReport report : reports) {
			if (report == null) {
				list.add(JasperFillManager.fillReport(jrxmlPaths.get(count), containers.get(count).getParameters(), sources.get(count)));
			} else {
				list.add(JasperFillManager.fillReport(report, containers.get(count).getParameters(), sources.get(count)));
			}
			count++;
		}
		return list;
	}

	/**
	 * 自身が保持しているパラメータオブジェクトを指定されたJRExporterに対して設定します。<br>
	 * @param exporter JRExporterオブジェクト
	 */
	protected void fillParameter(JRExporter exporter) {
		try {
			List<JasperPrint> prints = createJasperPrint();
			if (prints.size() <= 0) {
				return;
			} else if (prints.size() == 1) {
				exporter.setParameter(JRExporterParameter.JASPER_PRINT, prints.get(0));
			} else {
				JasperPrint print = prints.get(0);
				for (int i = 1; i <= prints.size() - 1; i++) {
					for (JRPrintPage page : prints.get(i).getPages()) {
						print.addPage(page);
					}
				}
				exporter.setParameter(JRExporterParameter.JASPER_PRINT, print);
			}
		} catch (Throwable e) {
			throw new JasperCriticalException("FJPR00009", e);
		}
	}

	/**
	 * エクスポート処理を行います。<br>
	 * サブクラスにおいてエクスポートを行う際のラップ処理を追加する為に設けられました。<br>
	 * @param exporter エクスポーターオブジェクト
	 * @throws 処理時に予期せぬ例外が発生した場合にスローされます
	 * @see org.phosphoresce.webcore.core.jasper.export.JasperWriter#exportProcessWrapper(net.sf.jasperreports.engine.JRExporter)
	 */
	@Override
	protected void exportProcessWrapper(JRExporter exporter) throws Throwable {
		if (containers.size() <= 0) {
			throw new JasperCriticalException("FJPR00010");
		} else {
			super.exportProcessWrapper(exporter);
		}
	}
}
