package org.phosphoresce.webcore.ext.jasper.export;

import java.awt.Font;
import java.awt.FontFormatException;
import java.awt.GraphicsEnvironment;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;

import net.sf.jasperreports.engine.util.JRGraphEnvInitializer;

import org.phosphoresce.webcore.ext.jasper.JasperConstants;

import com.lowagie.text.FontFactory;

/**
 * JasperRerportsフォントユーティリティクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/07/11	Kitagawa		新規作成
 *-->
 */
public class JasperFontUtil implements JasperConstants {

	/**
	 * コンストラクタ<br>
	 */
	private JasperFontUtil() {
		super();
	}

	/**
	 * 指定されたTrueTypeフォントをJasperReportに対して登録します。<br>
	 * @param filename TrueTypeフォント
	 * @param fontname 登録フォント名
	 * @throws IOException 正常にフォントを読み込めなかった場合にスローされます
	 * @throws FontFormatException 指定されたフォント形式が不正な場合にスローされます
	 */
	public static void registerTrueTypeFont(String filename, String fontname) throws FontFormatException, IOException {
		try {
			Field field = JRGraphEnvInitializer.class.getDeclaredField("AVAILABLE_FONT_FACE_NAMES");
			field.setAccessible(true);
			field.set(null, null);
		} catch (Throwable e) {
			throw new RuntimeException(e);
		}
		GraphicsEnvironment.getLocalGraphicsEnvironment().registerFont(createTrueTypeFont(filename));
		JRGraphEnvInitializer.initializeGraphEnv();
		if (fontname == null) {
			FontFactory.register(filename);
		} else {
			FontFactory.register(filename, fontname);
		}
	}

	/**
	 * 指定されたTrueTypeフォントをJasperReportに対して登録します。<br>
	 * @param filename TrueTypeフォント
	 * @throws IOException 正常にフォントを読み込めなかった場合にスローされます
	 * @throws FontFormatException 指定されたフォント形式が不正な場合にスローされます
	 */
	public static void registerTrueTypeFont(String filename) throws FontFormatException, IOException {
		registerTrueTypeFont(filename, null);
	}

	/**
	 * 指定されたTrueTypeフォントのファイルパスからFontオブジェクトを生成します。<br>
	 * @param filename TrueTypeフォントのファイルパス
	 * @return Fontオブジェクト
	 * @throws IOException 正常にフォントを読み込めなかった場合にスローされます
	 * @throws FontFormatException 指定されたフォント形式が不正な場合にスローされます
	 */
	public static Font createTrueTypeFont(String filename) throws FontFormatException, IOException {
		return createTrueTypeFont(new File(filename));
	}

	/**
	 * 指定されたTrueTypeフォントのFileオブジェクトからFontオブジェクトを生成します。<br>
	 * @param file TrueTypeフォントのFileオブジェクト
	 * @return Fontオブジェクト
	 * @throws IOException 正常にフォントを読み込めなかった場合にスローされます
	 * @throws FontFormatException 指定されたフォント形式が不正な場合にスローされます
	 */
	public static Font createTrueTypeFont(File file) throws FontFormatException, IOException {
		FileInputStream stream = null;
		try {
			stream = new FileInputStream(file);
			Font font = createTrueTypeFont(stream);
			stream.close();
			return font;
		} finally {
			if (stream != null) {
				stream.close();
			}
		}
	}

	/**
	 * 指定されたTrueTypeフォントのInputStreamオブジェクトからFontオブジェクトを生成します。<br>
	 * @param stream TrueTypeフォントのInputStreamオブジェクト
	 * @return Fontオブジェクト
	 * @throws IOException 正常にフォントを読み込めなかった場合にスローされます
	 * @throws FontFormatException 指定されたフォント形式が不正な場合にスローされます
	 */
	public static Font createTrueTypeFont(InputStream stream) throws FontFormatException, IOException {
		Font font = Font.createFont(Font.TRUETYPE_FONT, stream);
		return font;
	}

	/**
	 * 指定されたTrueTypeフォントのファイルパスからFontオブジェクトを生成します。<br>
	 * @param filename TrueTypeフォントのファイルパス
	 * @param style フォントスタイル
	 * @param size フォントサイズ
	 * @return Fontオブジェクト
	 * @throws IOException 正常にフォントを読み込めなかった場合にスローされます
	 * @throws FontFormatException 指定されたフォント形式が不正な場合にスローされます
	 */
	public static Font createTrueTypeFont(String filename, int style, float size) throws FontFormatException, IOException {
		return createTrueTypeFont(new File(filename), style, size);
	}

	/**
	 * 指定されたTrueTypeフォントのFileオブジェクトからFontオブジェクトを生成します。<br>
	 * @param file TrueTypeフォントのFileオブジェクト
	 * @param style フォントスタイル
	 * @param size フォントサイズ
	 * @return Fontオブジェクト
	 * @throws IOException 正常にフォントを読み込めなかった場合にスローされます
	 * @throws FontFormatException 指定されたフォント形式が不正な場合にスローされます
	 */
	public static Font createTrueTypeFont(File file, int style, float size) throws FontFormatException, IOException {
		FileInputStream stream = null;
		try {
			stream = new FileInputStream(file);
			Font font = createTrueTypeFont(stream, style, size);
			stream.close();
			return font;
		} finally {
			if (stream != null) {
				stream.close();
			}
		}
	}

	/**
	 * 指定されたTrueTypeフォントのInputStreamオブジェクトからFontオブジェクトを生成します。<br>
	 * @param stream TrueTypeフォントのInputStreamオブジェクト
	 * @param style フォントスタイル
	 * @param size フォントサイズ
	 * @return Fontオブジェクト
	 * @throws IOException 正常にフォントを読み込めなかった場合にスローされます
	 * @throws FontFormatException 指定されたフォント形式が不正な場合にスローされます
	 */
	public static Font createTrueTypeFont(InputStream stream, int style, float size) throws FontFormatException, IOException {
		return createTrueTypeFont(stream).deriveFont(style, size);
	}
}
