package org.phosphoresce.webcore.ext.hibernate4.transaction;

import java.util.HashMap;
import java.util.Map;

import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.service.ServiceRegistry;
import org.hibernate.service.ServiceRegistryBuilder;
import org.phosphoresce.webcore.ext.hibernate4.exception.HibernateConfigureException;

/**
 * Hibernateトランザクションファクトリクラス<br>
 * <br>
 * Hibernateセッションファクトリインスタンスの初期化及び、生成を行うインタフェースが提供されます。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2013/02/07	Kitagawa		新規作成
 *-->
 */
public final class TransactionFactory {

	/** クラスインスタンス */
	private static TransactionFactory instance;

	/** Hibernate環境設定オブジェクトキャッシュ */
	private Map<String, Configuration> configurations;

	/** ディフォルトHibernate環境設定オブジェクトキャッシュ */
	private Configuration defaultConfiguration;

	/**
	 * コンストラクタ<br>
	 */
	private TransactionFactory() {
		super();
		this.configurations = new HashMap<String, Configuration>();
	}

	/**
	 * クラスインスタンスを取得します。<br>
	 * @return クラスインスタンス
	 */
	static TransactionFactory instance() {
		if (instance == null) {
			throw new HibernateConfigureException("FHBN00017");
		}
		return instance;
	}

	/**
	 * 指定された名前空間で環境設定オブジェクトを設定します。<br>
	 * @param namespace 名前空間
	 * @param configuration 初期化済み環境設定オブジェクト
	 */
	public synchronized static void initialize(String namespace, Configuration configuration) {
		if (instance == null) {
			instance = new TransactionFactory();
		}
		synchronized (instance) {
			if (instance.configurations.containsKey(namespace)) {
				throw new HibernateConfigureException("FHBN00019", new String[] { namespace });
			}
			instance.configurations.put(namespace, configuration);

			// Hibernateセッションテスト生成処理(スキーマエクスポートを実行する為)
			instance.createSessionFactory(namespace).close();
		}
	}

	/**
	 * ディフォルトの環境設定オブジェクトを設定します。<br>
	 * @param configuration 初期化済み環境設定オブジェクト
	 */
	public synchronized static void initialize(Configuration configuration) {
		if (instance == null) {
			instance = new TransactionFactory();
		}
		synchronized (instance) {
			if (instance.defaultConfiguration != null) {
				throw new HibernateConfigureException("FHBN00020");
			}
			instance.defaultConfiguration = configuration;

			// Hibernateセッションテスト生成処理(スキーマエクスポートを実行する為)
			instance.createSessionFactory().close();
		}
	}

	/**
	 * セッションファクトリクラスインスタンスを生成します。<br>
	 * @return セッションファクトリクラスインスタンス
	 */
	SessionFactory createSessionFactory() {
		if (defaultConfiguration == null) {
			throw new HibernateConfigureException("FHBN00018");
		}
		ServiceRegistry registry = new ServiceRegistryBuilder().applySettings(defaultConfiguration.getProperties()).buildServiceRegistry();
		return defaultConfiguration.buildSessionFactory(registry);
	}

	/**
	 * セッションファクトリクラスインスタンスを生成します。<br>
	 * @param namespace 名前空間
	 * @return セッションファクトリクラスインスタンス
	 */
	SessionFactory createSessionFactory(String namespace) {
		if (defaultConfiguration == null) {
			throw new HibernateConfigureException("FHBN00018");
		}
		if (!configurations.containsKey(namespace)) {
			throw new HibernateConfigureException("FHBN00021", new String[] { namespace });
		}
		ServiceRegistry registry = new ServiceRegistryBuilder().applySettings(defaultConfiguration.getProperties()).buildServiceRegistry();
		return defaultConfiguration.buildSessionFactory(registry);
	}
}
