package org.phosphoresce.webcore.ext.hibernate4.transaction;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.transaction.TransactionManager;
import org.phosphoresce.webcore.ext.hibernate4.HibernateConstants;
import org.phosphoresce.webcore.ext.hibernate4.exception.HibernateTransactionException;

/**
 * Hibernateトランザクション管理クラス<br>
 * <br>
 * Hibernateの接続確立、切断、トランザクションのコミット、ロールバック制御を提供するクラスです。<br>
 * 当クラスをフレームワーク外のモジュールからインスタンス化して利用することは想定されません。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2013/02/01	Kitagawa		新規作成
 *-->
 */
public final class HibernateTransactionManager extends TransactionManager implements HibernateConstants {

	/** セッションファクトリオブジェクト */
	private SessionFactory factory;

	/** セッションオブジェクト */
	private Session session;

	/** トランザクションオブジェクト */
	private Transaction transaction;

	/**
	 * コンストラクタ<br>
	 */
	public HibernateTransactionManager() {
		super();
	}

	/**
	 * コンストラクタ<br>
	 * @param namespace 名前空間
	 */
	public HibernateTransactionManager(String namespace) {
		super(namespace);
	}

	/**
	 * Hibernateセッションオブジェクトを取得します。<br>
	 * @return session Hibernateセッションオブジェクト
	 */
	Session getSession() {
		if (!isOpened()) {
			throw new HibernateTransactionException("FHBN00013");
		}
		return session;
	}

	/**
	 * トランザクション開始処理を実行します。<br>
	 * @see org.phosphoresce.webcore.core.transaction.TransactionManager#openProcess()
	 */
	@Override
	protected void openProcess() {
		try {
			if (!StringUtil.isEmpty(getNamespace())) {
				factory = TransactionFactory.instance().createSessionFactory(getNamespace());
			} else {
				factory = TransactionFactory.instance().createSessionFactory();
			}
			session = factory.openSession();
			transaction = session.beginTransaction();
		} catch (Throwable e) {
			throw new HibernateTransactionException("FHBN00015", e);
		}
	}

	/**
	 * トランザクションを終了処理を実行します。<br>
	 * @see org.phosphoresce.webcore.core.transaction.TransactionManager#closeProcess()
	 */
	@Override
	protected void closeProcess() {
		try {
			if (!(transaction.wasCommitted() || transaction.wasRolledBack())) {
				transaction.rollback();
			}
			session.close();
			factory.close();
			transaction = null;
			session = null;
			factory = null;
		} catch (Throwable e) {
			throw new HibernateTransactionException("FHBN00016", e);
		}
	}

	/**
	 * トランザクションをコミット処理を実行します。<br>
	 * @see org.phosphoresce.webcore.core.transaction.TransactionManager#commitProcess()
	 */
	@Override
	protected void commitProcess() {
		if (transaction.isActive()) {
			transaction.commit();
		}
	}

	/**
	 * トランザクションをロールバック処理を実行します。<br>
	 * @see org.phosphoresce.webcore.core.transaction.TransactionManager#rollbackProcess()
	 */
	@Override
	protected void rollbackProcess() {
		if (transaction.isActive()) {
			transaction.rollback();
		}
	}
}
