package org.phosphoresce.webcore.ext.hibernate4.transaction;

import java.util.Date;

import javax.persistence.Column;
import javax.persistence.MappedSuperclass;

/**
 * データベーステーブル有効期間条件保持エンティティ上位抽象クラス<br>
 * <br>
 * 当エンティティは有効期間情報を保持するエンティティクラスです。<br>
 * 有効期間は必ず設定される前提のエンティティで明示的にnullを指定された場合でも、内部で常に値が設定されます。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2013/02/01	Kitagawa		新規作成
 *-->
 */
@MappedSuperclass
public abstract class AbstractHibernateExpirableEntity extends AbstractHibernateEntity {

	/** 有効期間開始 */
	@Column(name = "expiry_start", nullable = false)
	private Date expiryStart = DAOUtil.getMinimumDate();

	/** 有効期間終了 */
	@Column(name = "expiry_end", nullable = false)
	private Date expiryEnd = DAOUtil.getMaximumDate();

	/**
	 * データベース登録処理前のエンティティ内容の整合処理を実施します。<br>
	 */
	void validate() {
		validateEntity();
	}

	/**
	 * データベース登録処理前のエンティティ内容の整合処理を実施します。<br>
	 * 当処理はエンティティ毎に処理が異なります。<br>
	 */
	protected abstract void validateEntity();

	/**
	 * 指定された日付がエンティティの有効期間内の日付であるか判定します。<br>
	 * エンティティ有効範囲が2000/01/01～2000/04/01の場合、2000/03/01はtrueが返却されます。<br>
	 * @param date 対象日付オブジェクト
	 * @return 有効期間内の日付である場合にtrueを返却
	 */
	public boolean isValidityPeriod(Date date) {
		Date target = date == null ? DAOUtil.getMinimumDate() : date;
		return getExpiryEnd().compareTo(target) >= 0 && target.compareTo(getExpiryStart()) >= 0;
	}

	/**
	 * 指定された日付範囲がエンティティの有効期間に一部でも含まれる期間であるか判定します。<br>
	 * エンティティ有効範囲が2000/01/01～2000/04/01の場合、1999/12/01～2000/03/01はtrueが返却されます。<br>
	 * @param date1 日付範囲開始
	 * @param date2 日付範囲終了
	 * @return 有効期間が一部でも含まれる期間である場合にtrueを返却
	 */
	public boolean isOverlapPeriod(Date date1, Date date2) {
		Date target1 = date1 == null ? DAOUtil.getMinimumDate() : date1;
		Date target2 = date2 == null ? DAOUtil.getMaximumDate() : date2;
		Date targetStart = target1.compareTo(target2) >= 0 ? target2 : target1;
		Date targetEnd = target1.compareTo(target2) >= 0 ? target1 : target2;
		return getExpiryEnd().compareTo(targetStart) >= 0 && targetEnd.compareTo(getExpiryStart()) >= 0;
	}

	/**
	 * 指定された日付範囲がエンティティの有効期間を全て含む期間であるか判定します。<br>
	 * エンティティ有効範囲が2000/01/01～2000/04/01の場合、1999/12/01～2000/05/01はtrueが返却されます。<br>
	 * @param date1 日付範囲開始
	 * @param date2 日付範囲終了
	 * @return 有効期間が全て含む期間である場合にtrueを返却
	 */
	public boolean isFullContainPeriod(Date date1, Date date2) {
		Date target1 = date1 == null ? DAOUtil.getMinimumDate() : date1;
		Date target2 = date2 == null ? DAOUtil.getMaximumDate() : date2;
		Date targetStart = target1.compareTo(target2) >= 0 ? target2 : target1;
		Date targetEnd = target1.compareTo(target2) >= 0 ? target1 : target2;
		return getExpiryStart().compareTo(targetStart) >= 0 && targetEnd.compareTo(getExpiryEnd()) >= 0;
	}

	/**
	 * 指定された日付範囲がエンティティの有効期間内の期間であるか判定します。<br>
	 * エンティティ有効範囲が2000/01/01～2000/04/01の場合、2000/02/01～2000/03/01はtrueが返却されます。<br>
	 * @param date1 日付範囲開始
	 * @param date2 日付範囲終了
	 * @return 有効期間が期間内である場合にtrueを返却
	 */
	public boolean isInContainPeriod(Date date1, Date date2) {
		Date target1 = date1 == null ? DAOUtil.getMinimumDate() : date1;
		Date target2 = date2 == null ? DAOUtil.getMaximumDate() : date2;
		Date targetStart = target1.compareTo(target2) >= 0 ? target2 : target1;
		Date targetEnd = target1.compareTo(target2) >= 0 ? target1 : target2;
		return getExpiryStart().compareTo(targetStart) >= 0 && targetEnd.compareTo(getExpiryEnd()) >= 0;
	}

	/**
	 * 指定された日付範囲がエンティティの有効期間に完全一致する期間であるか判定します。<br>
	 * エンティティ有効範囲が2000/01/01～2000/04/01の場合、2000/01/01～2000/04/01はtrueが返却されます。<br>
	 * @param date1 日付範囲開始
	 * @param date2 日付範囲終了
	 * @return 有効期間が完全一致する期間である場合にtrueを返却
	 */
	public boolean isEqualPeriod(Date date1, Date date2) {
		Date target1 = date1 == null ? DAOUtil.getMinimumDate() : date1;
		Date target2 = date2 == null ? DAOUtil.getMaximumDate() : date2;
		Date targetStart = target1.compareTo(target2) >= 0 ? target2 : target1;
		Date targetEnd = target1.compareTo(target2) >= 0 ? target1 : target2;
		return getExpiryEnd().compareTo(targetStart) == 0 && targetEnd.compareTo(getExpiryStart()) == 0;
	}

	/**
	 * 有効期間開始を取得します。<br>
	 * @return 有効期間開始
	 */
	public final Date getExpiryStart() {
		if (expiryStart == null) {
			expiryStart = DAOUtil.getMinimumDate();
		}
		return expiryStart;
	}

	/**
	 * 有効期間開始を設定します。<br>
	 * @param expiryStart 有効期間開始
	 */
	public final void setExpiryStart(Date expiryStart) {
		if (expiryStart == null) {
			this.expiryStart = DAOUtil.getMinimumDate();
		} else {
			this.expiryStart = expiryStart;
		}
	}

	/**
	 * 有効期間終了を取得します。<br>
	 * @return 有効期間終了
	 */
	public final Date getExpiryEnd() {
		if (expiryEnd == null) {
			expiryEnd = DAOUtil.getMaximumDate();
		}
		return expiryEnd;
	}

	/**
	 * 有効期間終了を設定します。<br>
	 * @param expiryEnd 有効期間終了
	 */
	public final void setExpiryEnd(Date expiryEnd) {
		if (expiryEnd == null) {
			this.expiryEnd = DAOUtil.getMaximumDate();
		} else {
			this.expiryEnd = expiryEnd;
		}
	}
}
