package org.phosphoresce.webcore.core.transaction;

import org.phosphoresce.webcore.core.exception.WebcoreCoreModuleException;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.slf4j.LoggerFactory;

/**
 * トランザクション管理上位抽象クラス<br>
 * <br>
 * データベースアクセスを利用する業務ロジックのトランザクション管理上位クラスです。<br>
 * JDBCデータベースアクセスやO/Rマッパーアクセス等の各種アドオンモジュールにおけるインタフェースを統一するために設けられました。<br>
 * 各トランザクション管理サブクラスは当クラスが提供する(void)及び、(String)のコンストラクタを必ずpublicで設けて作成します。<br>
 * トランザクションクラスを利用する任意のクラスはこのコンストラクタが存在する前提のクラス設計を行います。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/05	Kitagawa		新規作成
 *-->
 */
public abstract class TransactionManager {

	/** ロガーオブジェクト */
	protected final CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(getClass()));

	/** 接続定義名前空間 */
	private String namespace;

	/** オープン済みフラグ */
	private boolean opened;

	/** コンテナ管理下フラグ */
	private boolean followedContainer;

	/**
	 * コンストラクタ<br>
	 * @param namespace 接続定義名前空間
	 */
	public TransactionManager(String namespace) {
		super();
		this.namespace = namespace;
		this.opened = false;
		this.followedContainer = false;
	}

	/**
	 * コンストラクタ<br>
	 */
	public TransactionManager() {
		this(null);
	}

	/**
	 * コンテナ管理下フラグを取得します。<br>
	 * @return コンテナ管理下フラグ
	 */
	final boolean isFollowedContainer() {
		return followedContainer;
	}

	/**
	 * コンテナ管理下フラグを設定します。<br>
	 * @param followedContainer コンテナ管理下フラグ
	 */
	final void setFollowedContainer(boolean followedContainer) {
		this.followedContainer = followedContainer;
	}

	/**
	 * トランザクションをオープンします。<br>
	 * コンテナ管理下の場合は個別にこの操作を行うことはできません。<br>
	 */
	public final void open() {
		if (followedContainer) {
			throw new WebcoreCoreModuleException("FCRE00049");
		}
		openCore();
	}

	/**
	 * トランザクションをオープンします。<br>
	 */
	final void openCore() {
		if (isOpened()) {
			throw new WebcoreCoreModuleException("FCRE00050");
		}
		openProcess();
		opened = true;
	}

	/**
	 * トランザクションをクローズします。<br>
	 * コンテナ管理下の場合は個別にこの操作を行うことはできません。<br>
	 */
	public final void close() {
		if (followedContainer) {
			throw new WebcoreCoreModuleException("FCRE00049");
		}
		closeCore();
	}

	/**
	 * トランザクションをクローズします。<br>
	 */
	final void closeCore() {
		if (!isOpened()) {
			throw new WebcoreCoreModuleException("FCRE00051");
		}
		rollbackProcess();
		closeProcess();
		opened = false;
	}

	/**
	 * トランザクションをコミットします。<br>
	 * コンテナ管理下の場合は個別にこの操作を行うことはできません。<br>
	 */
	public final void commit() {
		if (followedContainer) {
			throw new WebcoreCoreModuleException("FCRE00049");
		}
		commitCore();
	}

	/**
	 * トランザクションをコミットします。<br>
	 */
	final void commitCore() {
		if (!isOpened()) {
			throw new WebcoreCoreModuleException("FCRE00051");
		}
		commitProcess();
	}

	/**
	 * トランザクションをロールバックします。<br>
	 * コンテナ管理下の場合は個別にこの操作を行うことはできません。<br>
	 */
	public final void rollback() {
		if (followedContainer) {
			throw new WebcoreCoreModuleException("FCRE00049");
		}
		rollbackCore();
	}

	/**
	 * トランザクションをロールバックします。<br>
	 */
	final void rollbackCore() {
		if (!isOpened()) {
			throw new WebcoreCoreModuleException("FCRE00051");
		}
		rollbackProcess();
	}

	/**
	 * クラスの終了処理を行います。<br>
	 * @see java.lang.Object#finalize()
	 */
	@Override
	protected final void finalize() throws Throwable {
		if (isOpened()) {
			closeCore();
			log.output("FCRE00055");
		}
	}

	/**
	 * 接続定義名前空間を取得します。<br>
	 * @return 接続定義名前空間
	 */
	public final String getNamespace() {
		return namespace;
	}

	/**
	 * トランザクションがオープンされているか判定します。<br>
	 * @return トランザクションがオープンされている場合にtrueを返却
	 */
	public final boolean isOpened() {
		return opened;
	}

	/**
	 * トランザクション開始処理を実行します。<br>
	 * サブクラス実装でprotectedメソッド以外のアクセスに変更することは推奨されません。<br>
	 */
	protected abstract void openProcess();

	/**
	 * トランザクションを終了処理を実行します。<br>
	 * サブクラス実装でprotectedメソッド以外のアクセスに変更することは推奨されません。<br>
	 */
	protected abstract void closeProcess();

	/**
	 * トランザクションをコミット処理を実行します。<br>
	 * サブクラス実装でprotectedメソッド以外のアクセスに変更することは推奨されません。<br>
	 */
	protected abstract void commitProcess();

	/**
	 * トランザクションをロールバック処理を実行します。<br>
	 * サブクラス実装でprotectedメソッド以外のアクセスに変更することは推奨されません。<br>
	 */
	protected abstract void rollbackProcess();
}
