package org.phosphoresce.webcore.core.transaction;

import java.util.HashMap;
import java.util.Map;

import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.exception.WebcoreCoreModuleException;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.slf4j.LoggerFactory;

/**
 * トランザクション管理コンテナクラス<br>
 * <br>
 * データベーストランザクションを一元的に管理するコンテナクラスです。<br>
 * サーブレット処理中などで複数のトランザクションオブジェクトを扱い、一元的にコミットやクローズを行う為に設けられました。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/05	Kitagawa		新規作成
 *-->
 */
public class TransactionContainer {

	/** ロガーオブジェクト */
	private CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(getClass()));

	/** トランザクションデータ */
	private Map<String, TransactionManager> transactions;

	/** オープン済みフラグ */
	private boolean opened;

	/**
	 * コンストラクタ<br>
	 */
	public TransactionContainer() {
		super();
		this.transactions = new HashMap<String, TransactionManager>();
		this.opened = false;
	}

	/**
	 * トランザクション管理オブジェクトを追加します。<br>
	 * @param name 管理キー
	 * @param transaction トランザクション管理オブジェクト
	 */
	public void add(String name, TransactionManager transaction) {
		if (StringUtil.isEmpty(name)) {
			throw new WebcoreCoreModuleException("FCRE00046");
		}
		if (transaction == null) {
			throw new WebcoreCoreModuleException("FCRE00047");
		}
		if (transaction.isOpened()) {
			throw new WebcoreCoreModuleException("FCRE00052");
		}
		if (transactions.containsKey(name)) {
			throw new WebcoreCoreModuleException("FCRE00044", new Object[] { name });
		}
		if (transaction.isFollowedContainer()) {
			throw new WebcoreCoreModuleException("FCRE00053");
		}
		if (opened) {
			throw new WebcoreCoreModuleException("FCRE00054");
		}
		transactions.put(name, transaction);
		transaction.setFollowedContainer(true);
		log.output("FCRE00045", name, transaction.getClass().getName());
	}

	/**
	 * 指定されたキーのトランザクション管理オブジェクトを取得します。<br>
	 * @param name 管理キー
	 * @return トランザクション管理オブジェクト
	 */
	@SuppressWarnings("unchecked")
	public <T extends TransactionManager> T get(String name) {
		synchronized (transactions) {
			if (!transactions.containsKey(name)) {
				throw new WebcoreCoreModuleException("FCRE00048", new Object[] { name });
			}
			return (T) transactions.get(name);
		}
	}

	/**
	 * 指定されたキーのトランザクション管理オブジェクトをコンテナ配下から削除します。<br>
	 * @param name 管理キー
	 * @return コンテナ配下から削除されたトランザクション管理オブジェクト
	 */
	@SuppressWarnings("unchecked")
	public <T extends TransactionManager> T remove(String name) {
		synchronized (transactions) {
			if (!transactions.containsKey(name)) {
				throw new WebcoreCoreModuleException("FCRE00048", new Object[] { name });
			}
			TransactionManager transaction = transactions.remove(name);
			transaction.setFollowedContainer(false);
			return (T) transaction;
		}
	}

	/**
	 * オープン済みフラグを取得します。<br>
	 * @return オープン済みフラグ
	 */
	public boolean isOpened() {
		return opened;
	}

	/**
	 * コンテナ上のトランザクションをオープンします。<br>
	 */
	public void open() {
		synchronized (transactions) {
			if (isOpened()) {
				throw new WebcoreCoreModuleException("FCRE00050");
			}
			for (String key : transactions.keySet()) {
				transactions.get(key).openCore();
			}
			opened = true;
		}
	}

	/**
	 * コンテナ上のトランザクションをクローズします。<br>
	 */
	public void close() {
		synchronized (transactions) {
			if (!isOpened()) {
				throw new WebcoreCoreModuleException("FCRE00051");
			}
			for (String key : transactions.keySet()) {
				transactions.get(key).closeCore();
			}
			opened = false;
		}
	}

	/**
	 * コンテナ上のトランザクションをコミットします。<br>
	 * コンテナ管理下の場合は個別にこの操作を行うことはできません。<br>
	 */
	public void commit() {
		synchronized (transactions) {
			if (!isOpened()) {
				throw new WebcoreCoreModuleException("FCRE00051");
			}
			for (String key : transactions.keySet()) {
				transactions.get(key).commitCore();
			}
		}
	}

	/**
	 * コンテナ上のトランザクションをロールバックします。<br>
	 */
	public void rollback() {
		synchronized (transactions) {
			if (!isOpened()) {
				throw new WebcoreCoreModuleException("FCRE00051");
			}
			for (String key : transactions.keySet()) {
				transactions.get(key).rollbackCore();
			}
		}
	}

	/**
	 * クラスの終了処理を行います。<br>
	 * @see java.lang.Object#ize()
	 */
	@Override
	protected void finalize() throws Throwable {
		if (isOpened()) {
			close();
			log.output("FCRE00055");
		}
	}
}
