package org.phosphoresce.webcore.core.servlet;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.phosphoresce.lib.commons.util.FileUtil;
import org.phosphoresce.lib.commons.util.StreamUtil;
import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.ConfigName;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.config.MIMETypes;

/**
 * リソースリクエストサーブレット<br>
 * <br>
 * 当クラスは指定されたサーブレットパスとwebcore.servlet.resourcerequest.pathに定義された基底パスをもとに
 * サーバ上に配置されている静的なリソースをレスポンスに返却します。<br>
 * スタイルシートやスクリプトファイル等を別サーバ上に配置して利用する場合やWEB-INF配下に隠蔽して利用する際に利用します。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2011/12/24	Kitagawa		新規作成
 *-->
 */
public class StaticResourceRequestServlet extends HttpServlet {

	/**
	 * コンストラクタ<br>
	 */
	public StaticResourceRequestServlet() {
		super();
	}

	/**
	 * GETリクエスト処理を実行します。<br>
	 * @param request リクエストオブジェクト
	 * @param response レスポンスオブジェクト
	 * @throws ServletException サーブレット処理に失敗した場合にスローされます
	 * @throws IOException 入出力例外が発生した場合にスローされます
	 * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		doService(request, response);
	}

	/**
	 * POSTリクエスト処理を実行します。<br>
	 * @param request リクエストオブジェクト
	 * @param response レスポンスオブジェクト
	 * @throws ServletException サーブレット処理に失敗した場合にスローされます
	 * @throws IOException 入出力例外が発生した場合にスローされます
	 * @see javax.servlet.http.HttpServlet#doPost(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		doService(request, response);
	}

	/**
	 * リクエスト処理を実行します。<br>
	 * @param request リクエストオブジェクト
	 * @param response レスポンスオブジェクト
	 * @throws ServletException サーブレット処理に失敗した場合にスローされます
	 * @throws IOException 入出力例外が発生した場合にスローされます
	 */
	private void doService(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		//		System.out.println("------------------------------------------------------------------------------------");
		//		for (@SuppressWarnings("unchecked")
		//		Enumeration<String> enumeration = request.getHeaderNames(); enumeration.hasMoreElements();) {
		//			String key = enumeration.nextElement();
		//			System.out.println(key + "=" + request.getHeader(key));
		//		}
		//		System.out.println("------------------------------------------------------------------------------------");

		File file = new File(FileUtil.connectPath(Config.getString(ConfigName.COMMON_STATIC_RESOURCE), StringUtil.nvl(request.getPathInfo())));
		if (file.exists() && file.isFile()) {
			if (file.lastModified() > request.getDateHeader("If-Modified-Since")) {
				// コンテントタイプ設定
				response.setContentType(getMIME(file.getAbsolutePath()) + "; charset=" + Config.getString(ConfigName.COMMON_CHARSET_DEFAULT));
				response.setContentLength((int) file.length());

				// ヘッダ設定
				response.addDateHeader("Date", System.currentTimeMillis());
				response.addDateHeader("Last-Modified", file.lastModified());
				response.addHeader("Connection", "close");

				// ステータス設定
				response.setStatus(HttpServletResponse.SC_OK);

				// レスポンス出力
				BufferedInputStream is = null;
				BufferedOutputStream os = null;
				try {
					is = new BufferedInputStream(new FileInputStream(file));
					os = new BufferedOutputStream(response.getOutputStream());
					StreamUtil.pipe(is, os);
					os.flush();
				} catch (Throwable e) {
					os.close();
					is.close();
				} finally {
					if (os != null) {
						os.close();
					}
					if (is != null) {
						is.close();
					}
				}
			} else {
				doIllegalRequest(request, response);
			}
		} else {
			// ステータス設定
			response.setStatus(HttpServletResponse.SC_NOT_MODIFIED);
		}
	}

	/**
	 * 不正なリクエストパスを受け取った際の処理を行います。<br>
	 * @param request リクエストオブジェクト
	 * @param response レスポンスオブジェクト
	 * @throws ServletException サーブレット処理に失敗した場合にスローされます
	 * @throws IOException 入出力例外が発生した場合にスローされます
	 */
	private void doIllegalRequest(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		response.sendError(HttpServletResponse.SC_NOT_FOUND, request.getPathInfo());
	}

	/**
	 * 指定されたファイル名に対応したMIMEタイプを取得します。<br>
	 * @param filename ファイル名
	 * @return MIMEタイプ文字列
	 */
	private String getMIME(String filename) {
		String extension = StringUtil.isEmpty(filename) || filename.indexOf(".") < 0 ? "" : filename.substring(filename.lastIndexOf(".") + 1);
		return MIMETypes.getMIMEType(extension);
	}
}
