package org.phosphoresce.webcore.core.servlet;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URLEncoder;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.ConfigName;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.exception.WebcoreConfigurationException;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.slf4j.LoggerFactory;

import com.sun.xml.internal.messaging.saaj.packaging.mime.internet.MimeUtility;

/**
 * レスポンス出力ストリームオブジェクト<br>
 * <br>
 * 通常の出力ストリームオブジェクトをラップして実際の出力インタフェース呼び出し時迄はレスポンスからストリームをオープンしません。<br>
 * 当クラスはストリームオープン後にアプリケーションエラー等でページコンテンツにリダイレクトすることを可能とする為のラップクラスです。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/07/10	Kitagawa		新規作成
 * 2012/12/14	Kitagawa		クライアント側のダウンロードキャンセル時のClientAbortExceptionを補足し例外ログ出力を行わないように修正(Tomcat依存してしまう為、IOExceptionを補足する)
 *-->
 */
public class ServletOutputStream extends OutputStream {

	/** ロガーオブジェクト */
	private static CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(ServletOutputStream.class));

	/** レスポンスオブジェクト */
	private HttpServletResponse response;

	/** 出力ストリームオブジェクト */
	private OutputStream stream;

	/** クライアントキャンセルフラグ */
	private boolean canceled;

	/**
	 * コンストラクタ<br>
	 * @param response レスポンスオブジェクト
	 * @param session セッションコンテナオブジェクト
	 */
	public ServletOutputStream(HttpServletResponse response) {
		super();
		this.response = response;
	}

	/**
	 * ストリームオブジェクトを生成します。<br>
	 * 当メソッドでは指定されたレスポンスに対して必要なヘッダ情報を付加します。<br>
	 * @param request サーブレットリクエストオブジェクト
	 * @param response サーブレットレスポンスオブジェクト
	 * @param charset キャラクタセット
	 * @param mimeType MIMEタイプ(不要な場合はnullを設定)
	 * @param contentLength コンテンツ長(不要な場合はnullを設定)
	 * @return ストリームオブジェクト
	 */
	public static ServletOutputStream create(HttpServletRequest request, HttpServletResponse response, String charset, String mimeType, Integer contentLength) {
		try {
			ServletOutputStream stream = new ServletOutputStream(response);

			// MIMEタイプ設定
			if (StringUtil.isEmpty(mimeType)) {
				response.setContentType(mimeType);
			}

			// キャラクタセット設定
			if (!StringUtil.isEmpty(charset)) {
				response.setCharacterEncoding(charset);
			} else {
				response.setCharacterEncoding(Config.getString(ConfigName.COMMON_CHARSET_DEFAULT));
			}

			// コンテンツ長設定
			if (contentLength != null) {
				response.setContentLength(contentLength);
			}

			log.output("FCRE00040", mimeType, charset, contentLength);
			return stream;
		} catch (Throwable e) {
			throw new WebcoreConfigurationException("FCRE00040", e);
		}
	}

	/**
	 * ダウンロード用ストリームオブジェクトを生成します。<br>
	 * 当メソッドでは指定されたレスポンスに対して必要なヘッダ情報を付加します。<br>
	 * @param request サーブレットリクエストオブジェクト
	 * @param response サーブレットレスポンスオブジェクト
	 * @param charset キャラクタセット
	 * @param mimeType MIMEタイプ(不要な場合はnullを設定)
	 * @param contentLength コンテンツ長(不要な場合はnullを設定)
	 * @param filename ダウンロードファイル名
	 * @return ストリームオブジェクト
	 */
	public static ServletOutputStream createDownloadStream(HttpServletRequest request, HttpServletResponse response, String charset, String mimeType, Integer contentLength, String filename) {
		try {
			// ダウンロードファイル名エンコード
			String encodedFilename = "";
			if (request.getHeader("User-Agent").indexOf("MSIE") == -1) {
				encodedFilename = MimeUtility.encodeWord(filename, "ISO-2022-JP", "B");
			} else {
				encodedFilename = URLEncoder.encode(filename, Config.getString(ConfigName.COMMON_CHARSET_DEFAULT));
			}

			// ダウンロードファイル名設定
			response.setHeader("Content-Disposition", "filename=\"" + encodedFilename + "\"");

			// ストリームオブジェクト生成
			ServletOutputStream stream = null;
			if (!StringUtil.isEmpty(mimeType)) {
				stream = create(request, response, charset, mimeType, contentLength);
			} else {
				stream = create(request, response, charset, "application/octet-stream", contentLength);
			}

			log.output("FCRE00042", filename);
			return stream;
		} catch (Throwable e) {
			throw new WebcoreConfigurationException("FCRE00043", e);
		}
	}

	/**
	 * JSONレスポンス用ストリームオブジェクトを生成します。<br>
	 * 当メソッドでは指定されたレスポンスに対して必要なヘッダ情報を付加します。<br>
	 * @param request サーブレットリクエストオブジェクト
	 * @param response サーブレットレスポンスオブジェクト
	 * @param charset キャラクタセット
	 * @param contentLength コンテンツ長(不要な場合はnullを設定)
	 * @return ストリームオブジェクト
	 */
	public static ServletOutputStream createJSONStream(HttpServletRequest request, HttpServletResponse response, String charset, Integer contentLength) {
		// ストリームオブジェクト生成
		//ServletOutputStream stream = create(request, response, charset, "application/json", contentLength);
		ServletOutputStream stream = create(request, response, charset, "text/javascript", contentLength);

		log.output("FCRE00044");
		return stream;
	}

	/**
	 * PDFレスポンス用ストリームオブジェクトを生成します。<br>
	 * 当メソッドでは指定されたレスポンスに対して必要なヘッダ情報を付加します。<br>
	 * @param request サーブレットリクエストオブジェクト
	 * @param response サーブレットレスポンスオブジェクト
	 * @param charset キャラクタセット
	 * @param contentLength コンテンツ長(不要な場合はnullを設定)
	 * @return ストリームオブジェクト
	 */
	public static ServletOutputStream createPDFStream(HttpServletRequest request, HttpServletResponse response, String charset, Integer contentLength) {
		// ストリームオブジェクト生成
		ServletOutputStream stream = create(request, response, charset, "application/pdf", contentLength);

		log.output("FCRE00044");
		return stream;
	}

	/**
	 * ストリームにデータを出力します。<br>
	 * @see java.io.OutputStream#write(int)
	 */
	@Override
	public void write(int b) throws IOException {
		if (stream == null) {
			stream = new BufferedOutputStream(response.getOutputStream());
			log.output("FSTR00025");
		}
		try {
			stream.write(b);
		} catch (IOException e) {
			// このタイミングで発生するIOExceptionはクライアントダウンロードキャンセルによるものとして無視する
			if (!canceled) {
				log.output("FSTR00026");
			}
			canceled = true;
		}
	}

	/**
	 * ストリームにデータを出力します。<br>
	 * @see java.io.OutputStream#write(byte[])
	 */
	@Override
	public void write(byte[] b) throws IOException {
		if (stream == null) {
			stream = new BufferedOutputStream(response.getOutputStream());
			log.output("FSTR00025");
		}
		try {
			stream.write(b);
		} catch (IOException e) {
			// このタイミングで発生するIOExceptionはクライアントダウンロードキャンセルによるものとして無視する
			if (!canceled) {
				log.output("FSTR00026");
			}
			canceled = true;
		}
	}

	/**
	 * ストリームにデータを出力します。<br>
	 * @see java.io.OutputStream#write(byte[], int, int)
	 */
	@Override
	public void write(byte[] b, int off, int len) throws IOException {
		if (stream == null) {
			stream = new BufferedOutputStream(response.getOutputStream());
			log.output("FSTR00025");
		}
		try {
			stream.write(b, off, len);
		} catch (IOException e) {
			// このタイミングで発生するIOExceptionはクライアントダウンロードキャンセルによるものとして無視する
			if (!canceled) {
				log.output("FSTR00026");
			}
			canceled = true;
		}
	}

	/**
	 * ストリームをフラッシュします。<br>
	 * @see java.io.OutputStream#flush()
	 */
	@Override
	public void flush() throws IOException {
		if (stream == null) {
			stream = new BufferedOutputStream(response.getOutputStream());
			log.output("FSTR00025");
		}
		try {
			stream.flush();
			log.output("FSTR00027");
		} catch (IOException e) {
			// このタイミングで発生するIOExceptionはクライアントダウンロードキャンセルによるものとして無視する
			if (!canceled) {
				log.output("FSTR00026");
			}
			canceled = true;
		}
	}

	/**
	 * ストリームをクローズします。<br>
	 * @see java.io.OutputStream#close()
	 */
	@Override
	public void close() throws IOException {
		if (stream == null) {
			stream = new BufferedOutputStream(response.getOutputStream());
			log.output("FSTR00025");
		}
		try {
			stream.close();
			log.output("FSTR00028");
		} catch (IOException e) {
			// このタイミングで発生するIOExceptionはクライアントダウンロードキャンセルによるものとして無視する
			if (!canceled) {
				log.output("FSTR00026");
			}
			canceled = true;
		}
	}

	/**
	 * ストリームオブジェクトを破棄します。<br>
	 * @see java.lang.Object#finalize()
	 */
	@Override
	protected void finalize() throws Throwable {
		super.finalize();
		try {
			close();
		} catch (Throwable e) {
			// ストリームクローズ時の例外は無視
		}
	}
}
