package org.phosphoresce.webcore.core.config;

import java.io.IOException;
import java.io.Serializable;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;

import org.phosphoresce.lib.commons.util.PropertiesUtil;
import org.phosphoresce.lib.commons.util.StringUtil;

/**
 * アプリケーション文字列定義クラス<br>
 * <br>
 * 当クラスはアプリケーション初期化時に外部定義されたリソースを元にインスタンスが生成され、それ以降はシングルトンとして扱われます。<br>
 * <br>
 * 当クラスがアクセスする文字列定義キーを定義スコープを限定してアクセスする場合は、インナークラスScopedStringResourceを特定のプレフィクスを持つ
 * 定義情報のみを扱うクラスインスタンスとして生成して利用します。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/06	Kitagawa		新規作成
 *-->
 */
public class StringResource implements Serializable {

	/** クラスインスタンス */
	private static StringResource instance;

	/** プロパティリソース */
	private Properties properties;

	/** プロパティリソース(ディフォルト) */
	private Properties propertiesDefault;

	/**
	 * コンストラクタ<br>
	 */
	private StringResource() {
		super();
		this.properties = new Properties();
		this.propertiesDefault = new Properties();
	}

	/**
	 * 指定されたメッセージプロパティからクラスを初期化します。<br>
	 * @param properties プロパティリソースパス
	 * @param append 追加読み込みを行う場合にtrueを指定
	 * @throws IOException プロパティリソースの読み込みに失敗した場合にスローされます
	 */
	public static synchronized void initialize(String properties, boolean append) throws IOException {
		if (instance == null) {
			instance = new StringResource();
		}
		if (append) {
			if (instance.properties == null || instance.propertiesDefault == null) {
				instance = new StringResource();
			}
			instance.properties.putAll(PropertiesUtil.load(properties, true));
			instance.propertiesDefault.putAll(PropertiesUtil.load(properties));
		} else {
			instance = new StringResource();
			instance.properties = PropertiesUtil.load(properties, true);
			instance.propertiesDefault = PropertiesUtil.load(properties);
		}
	}

	/**
	 * 指定されたメッセージプロパティからクラスを初期化します。<br>
	 * @param properties プロパティリソースパス
	 * @throws IOException プロパティリソースの読み込みに失敗した場合にスローされます
	 */
	public static synchronized void initialize(String properties) throws IOException {
		initialize(properties, false);
	}

	/**
	 * クラスを破棄します。<br>
	 */
	public static synchronized void destroy() {
		clear();
		instance = null;
	}

	/**
	 * 保持されているプロパティをすべてクリアします。<br>
	 */
	public static synchronized void clear() {
		if (instance != null) {
			if (instance.properties != null) {
				instance.properties.clear();
			}
			if (instance.propertiesDefault != null) {
				instance.propertiesDefault.clear();
			}
		}
	}

	/**
	 * プロパティオブジェクトを取得します。<br>
	 * @return プロパティオブジェクト
	 */
	static Properties getProperties() {
		if (instance != null) {
			return instance.properties;
		} else {
			return new Properties();
		}
	}

	/**
	 * スコープ限定アプリケーションメッセージアクセスクラス<br>
	 * <br>
	 * 当クラスはコンストラクタに対して指定された文字列をプレフィクス
	 * として持つ定義に限ってメッセージリソースにアクセスするクラスです。<br>
	 * 当クラスを利用したメッセージアクセス時のコード指定は、プレフィクス
	 * 以降のコードのみを指定します。<br>
	 * 
	 * @author Kitagawa<br>
	 * 
	 *<!--
	 * 更新日		更新者			更新内容
	 * 2010/09/07	Kitagawa		新規作成
	 *-->
	 */
	public static final class ScopedStringResource implements Serializable {

		/** スコープ限定プレフィックス */
		private String prefix;

		/**
		 * コンストラクタ<br>
		 */
		private ScopedStringResource() {
			super();
		}

		/**
		 * コンストラクタ<br>
		 * @param prefix スコープ限定プレフィックス文字列
		 */
		public ScopedStringResource(String prefix) {
			this.prefix = prefix;
		}

		/**
		 * 指定されたメッセージコードのメッセージを取得します。<br>
		 * @param code メッセージコード
		 * @param binds バインド文字列
		 * @return メッセージ
		 */
		public String get(String code, Object... binds) {
			return StringResource.getString(StringUtil.isEmpty(prefix) ? code : prefix + "." + code, binds);
		}

		/**
		 * 指定されたメッセージコードのメッセージを取得します。<br>
		 * @param code メッセージコード
		 * @return メッセージ
		 */
		public String get(String code) {
			return StringResource.getString(StringUtil.isEmpty(prefix) ? code : prefix + "." + code);
		}
	}

	/**
	 * 指定されたメッセージコードのメッセージを取得します。<br>
	 * @param code メッセージコード
	 * @param binds バインド文字列
	 * @return メッセージ
	 */
	public static String getString(String code, Object... binds) {
		if (instance == null) {
			return code;
		}
		String leveledCode = StringResourceLevel.getDefinitionCode(code);
		String message = instance.properties.getProperty(leveledCode);
		if (StringUtil.isEmpty(message)) {
			message = instance.propertiesDefault.getProperty(leveledCode);
		}
		if (StringUtil.isEmpty(message)) {
			return "";
		}
		if (binds != null && binds.length > 0) {
			List<String> bindList = new LinkedList<String>();
			for (Object bind : binds) {
				String value = bind == null ? "null" : bind.toString();
				bindList.add(value);
			}
			return StringUtil.bind(message, bindList.toArray(new String[0]));
		}
		return message;
	}

	/**
	 * 指定されたメッセージコードのメッセージを取得します。<br>
	 * @param code メッセージコード
	 * @return メッセージ
	 */
	public static String getString(String code) {
		return getString(code, new Object[0]);
	}

	/**
	 * 全てのメッセージコードキーを取得します。<br>
	 * @return メッセージコードキー配列
	 */
	public static String[] keys() {
		List<String> list = new LinkedList<String>();
		for (Object key : instance.properties.keySet()) {
			list.add((String) key);
		}
		return list.toArray(new String[0]);
	}
}
