package org.phosphoresce.webcore.core.config;

import java.io.File;
import java.io.IOException;

import org.phosphoresce.lib.commons.util.FileUtil;
import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.ConfigName;
import org.phosphoresce.webcore.core.Environment;

/**
 * サーバーリソース管理クラス<br>
 * <br>
 * 各種サーバー上のリソースに対するアクセスインタフェースが提供されます。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2011/10/17	Kitagawa		新規作成
 *-->
 */
public class ServerResource {

	/**
	 * コンストラクタ<br>
	 */
	private ServerResource() {
		super();
	}

	/**
	 * 指定されたパスのサーバーリソースを取得します。<br>
	 * 指定されたパスが相対パスの場合、アプリケーションコンテキスト、WEB-INFパスの順序で
	 * リソースを検索し、最初に見つかったリソースを返却します。<br>
	 * 絶対パスが指定された場合はそのパスに存在するリソースを返却します。<br>
	 * また、指定されたパスに関連するリソースが存在しない場合は指定されたパスのFileオブジェクトを返却します。<br>
	 * @param path 対象リソースパス
	 * @return サーバーリソースファイルオブジェクト
	 */
	public static File getFile(String path) {
		File file = null;
		file = new File(FileUtil.connectPath(Environment.getContentPath(), path));
		if (file.exists()) {
			return file;
		}
		file = new File(FileUtil.connectPath(Environment.getWebinfPath(), path));
		if (file.exists()) {
			return file;
		}
		file = new File(path);
		if (file.exists()) {
			return file;
		}
		return file;
	}

	/**
	 * 指定されたパスのサーバーリソースが存在するか判定します。<br>
	 * 指定されたパスが相対パスの場合、アプリケーションコンテキスト、WEB-INFパスの順序でリソースを検索します。<br>
	 * 絶対パスが指定された場合はそのパスに存在するリソースを検索します。<br>
	 * @param path 対象リソースパス
	 * @return 指定されたパスのリソースが存在する場合にtrueを返却
	 */
	public static boolean existsFile(String path) {
		File file = getFile(path);
		return file != null && file.exists();
	}

	/**
	 * 指定されたパスのダウンロードリソースを取得します。<br>
	 * @param path 対象リソースパス
	 * @return サーバーリソースファイルオブジェクト
	 */
	public static File getDownloadFile(String path) {
		String base = Config.getString(ConfigName.COMMON_GENERAL_DOWNLOAD_RESOURCE);
		return new File(FileUtil.connectPath(base, path));
	}

	/**
	 * 指定されたパスのダウンロードリソースが存在するか判定します。<br>
	 * @param path 対象リソースパス
	 * @return 指定されたパスのリソースが存在する場合にtrueを返却
	 */
	public static boolean existsDownloadFile(String path) {
		File file = getDownloadFile(path);
		return file != null && file.exists();
	}

	/**
	 * 指定されたパスのアップロードリソースを取得します。<br>
	 * @param path 対象リソースパス
	 * @return サーバーリソースファイルオブジェクト
	 */
	public static File getUploadFile(String path) {
		String base = Config.getString(ConfigName.COMMON_GENERAL_UPLOAD_RESOURCE);
		return new File(FileUtil.connectPath(base, path));
	}

	/**
	 * 指定されたパスのアップロードリソースが存在するか判定します。<br>
	 * @param path 対象リソースパス
	 * @return 指定されたパスのリソースが存在する場合にtrueを返却
	 */
	public static boolean existsUploadFile(String path) {
		File file = getUploadFile(path);
		return file != null && file.exists();
	}

	/**
	 * テンポラリファイル名を生成します。<br>
	 * ここで生成されたファイル名はパス名を含まないことに注意してください。<br>
	 * また、既にあるファイルであるかの判定も行いません。<br>
	 * @return テンポラリファイル名
	 */
	private static String createNewTemporaryFilename() {
		StringBuilder builder = new StringBuilder();
		builder.append("{TMP-");
		builder.append(StringUtil.toMD5Hash(String.valueOf(System.currentTimeMillis())));
		builder.append("-");
		builder.append(StringUtil.toMD5Hash(String.valueOf(Math.random())));
		builder.append("}");
		return builder.toString();
	}

	/**
	 * 新規にテンポラリファイルを生成します。<br>
	 * @return テンポラリファイルオブジェクト
	 * @throws IOException 正常にテンポラリファイルが生成できなかった場合にスローされます
	 */
	public static File createNewTemporaryFile() throws IOException {
		File file = null;
		while (file == null || file.exists()) {
			file = getTemporaryFile(createNewTemporaryFilename());
		}
		file.createNewFile();
		return file;
	}

	/**
	 * 指定されたパスのテンポラリリソースを取得します。<br>
	 * @param path 対象リソースパス
	 * @return サーバーリソースファイルオブジェクト
	 */
	public static File getTemporaryFile(String path) {
		String base = Config.getString(ConfigName.COMMON_GENERAL_TEMPORARY_RESOURCE);
		return new File(FileUtil.connectPath(base, path));
	}

	/**
	 * 指定されたパスのテンポラリリソースが存在するか判定します。<br>
	 * @param path 対象リソースパス
	 * @return 指定されたパスのリソースが存在する場合にtrueを返却
	 */
	public static boolean existsTemporaryFile(String path) {
		File file = getTemporaryFile(path);
		return file != null && file.exists();
	}
}
