package org.phosphoresce.webcore.core.config;

import java.io.Serializable;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.phosphoresce.lib.commons.util.PropertiesUtil;
import org.phosphoresce.lib.commons.util.ResourceUtil;
import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.ConfigName;
import org.phosphoresce.webcore.core.Environment;
import org.phosphoresce.webcore.core.exception.WebcoreConfigurationException;

/**
 * アプリケーション動作環境設定クラス<br>
 * <br>
 * 当クラスは初期化時に指定されたプロパティリソースへのアクセスインタフェースを提供します。<br>
 * また、指定されたプロパティリソース内で追加読み込みプロパティが指定されている場合はそれらを含めて読み込みます。<br>
 * 初期化時に厳密動作モードが指定された場合、プロパティ取得時に存在しないキーが指定されると例外がスローされるようになります。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/03	Kitagawa		新規作成
 * 2012/02/26	Kitagawa		厳密動作モード(strict)を追加
 * 2012/02/26	Kitagawa		破棄メソッド(destroy)を追加
 * 2012/02/26	Kitagawa		配列取得メソッドにおいて先頭に#がある場合は除外するインタフェースを追加
 * 2013/07/04	Kitagawa		プロパティキーを直接文字列で受け取らず、列挙形式クラスを受取るように修正
 *-->
 */
public class Config implements Serializable {

	/** ディフォルト環境設定リソース名 */
	public static final String DEFAULT_CONFIG = "config.properties";

	/** クラスインスタンス */
	private static Config instance;

	/** 環境設定プロパティオブジェクト */
	private Properties properties;

	/** 厳密動作モード */
	private boolean strict;

	/**
	 * コンストラクタ<br>
	 */
	private Config() {
		super();
	}

	/**
	 * 指定されたコンテンツパス及び、プロパティリソースからクラスを初期化します。<br>
	 * @param properties プロパティリソースパス
	 * @param strict 厳密動作モード(存在しないキーを指定した際に例外扱いとする場合にtrueを指定)
	 */
	public static synchronized void initialize(String properties, boolean strict) {
		try {
			instance = new Config();
			instance.properties = PropertiesUtil.load(Environment.bindEnv(properties));
			instance.strict = strict;
			for (String value : getStrings(ConfigName.COMMON_RESOURCE_CONFIG_EXTENDS, true)) {
				if (!StringUtil.isEmpty(value)) {
					if (ResourceUtil.isExistResource(value)) {
						instance.properties.putAll(PropertiesUtil.load(value));
					}
				}
			}
		} catch (Throwable e) {
			throw new WebcoreConfigurationException("FCRE00021", e);
		}
	}

	/**
	 * 指定されたコンテンツパス及び、プロパティリソースからクラスを初期化します。<br>
	 * 当メソッドで初期化されたインスタンスは厳密動作モードで動作します。<br>
	 * @param properties プロパティリソースパス
	 */
	public static synchronized void initialize(String properties) {
		initialize(properties, true);
	}

	/**
	 * クラスを破棄します。<br>
	 */
	public static synchronized void destroy() {
		if (instance != null) {
			if (instance.properties != null) {
				instance.properties.clear();
			}
		}
		instance = null;
	}

	/**
	 * 厳密動作モードを取得します。<br>
	 * @return 厳密動作モード
	 */
	public static boolean isStrict() {
		if (instance == null) {
			throw new WebcoreConfigurationException("FCRE00022", new Object[] { Config.class.getName() });
		}
		return instance.strict;
	}

	/**
	 * 初期化済みであるか判定します。<br>
	 * @return 初期化済みである場合にtrueを返却
	 */
	public static boolean isInitialized() {
		return instance != null && instance.properties != null;
	}

	/**
	 * プロパティリソースにおいて定義されているキー配列を全て取得します。<br>
	 * @return 定義キー配列
	 */
	private static String[] getPropertyKeys() {
		if (instance == null) {
			throw new WebcoreConfigurationException("FCRE00022", new Object[] { Config.class.getName() });
		}
		List<String> list = new LinkedList<String>();
		for (Enumeration<Object> enumeration = instance.properties.keys(); enumeration.hasMoreElements();) {
			String key = (String) enumeration.nextElement();
			list.add(key);
		}
		return list.toArray(new String[0]);
	}

	/**
	 * 保持内容をマップオブジェクトとして取得します。<br>
	 * @return マップオブジェクト
	 */
	public static Map<String, String> getMap() {
		Map<String, String> map = new HashMap<String, String>();
		for (String key : getPropertyKeys()) {
			map.put(key, getString(key));
		}
		return map;
	}

	/**
	 * 指定されたキーのプロパティが未指定であるか判定します。<br>
	 * @param key プロパティキー
	 * @return 指定されたキーのプロパティが未指定である場合にtrueを返却
	 */
	private static boolean isEmpty(String key) {
		return StringUtil.isEmpty(getString(key));
	}

	/**
	 * 指定されたキーのプロパティが未指定であるか判定します。<br>
	 * @param name プロパティ定義名
	 * @return 指定されたキーのプロパティが未指定である場合にtrueを返却
	 */
	public static boolean isEmpty(ConfigName name) {
		if (name == null) {
			throw new NullPointerException();
		}
		return isEmpty(name.key());
	}

	/**
	 * 指定キーのプロパティを取得します。<br>
	 * @param key プロパティキー
	 * @return プロパティ値
	 */
	private static String getString(String key) {
		if (key == null) {
			throw new NullPointerException();
		}
		if (instance == null) {
			throw new WebcoreConfigurationException("FCRE00022", new Object[] { Config.class.getName() });
		}
		if (instance.strict) {
			if (!instance.properties.containsKey(key)) {
				throw new WebcoreConfigurationException("FCRE00023", new Object[] { key });
			}
		}
		String value = instance.properties.getProperty(key);
		value = Environment.bindEnv(value);
		return value == null ? "" : value;
	}

	/**
	 * 指定キーのプロパティを取得します。<br>
	 * @param name プロパティ定義名
	 * @return プロパティ値
	 */
	public static String getString(ConfigName name) {
		if (name == null) {
			throw new NullPointerException();
		}
		return getString(name.key());
	}

	/**
	 * 指定キーのプロパティを取得します。<br>
	 * カンマ区切りされているプロパティを文字列配列として取得します。<br>
	 * @param key プロパティキー
	 * @param execludeComment トークン先頭に"#"がある場合は除外する場合にtrueを返却
	 * @return プロパティ値
	 */
	private static String[] getStrings(String key, boolean execludeComment) {
		String def = Config.getString(key);
		String[] values = StringUtil.isEmpty(def) ? new String[0] : Config.getString(key).split(",");
		if (execludeComment) {
			LinkedList<String> list = new LinkedList<String>();
			for (String value : values) {
				if (!value.startsWith("#")) {
					list.add(Environment.bindEnv(value));
				}
			}
			return (String[]) list.toArray(new String[0]);
		} else {
			return values;
		}
	}

	/**
	 * 指定キーのプロパティを取得します。<br>
	 * カンマ区切りされているプロパティを文字列配列として取得します。<br>
	 * @param name プロパティ定義名
	 * @param execludeComment トークン先頭に"#"がある場合は除外する場合にtrueを返却
	 * @return プロパティ値
	 */
	public static String[] getStrings(ConfigName name, boolean execludeComment) {
		if (name == null) {
			throw new NullPointerException();
		}
		return getStrings(name.key(), execludeComment);
	}

	/**
	 * 指定キーのプロパティを取得します。<br>
	 * カンマ区切りされているプロパティを文字列配列として取得します。<br>
	 * @param key プロパティキー
	 * @return プロパティ値
	 */
	private static String[] getStrings(String key) {
		return getStrings(key, true);
	}

	/**
	 * 指定キーのプロパティを取得します。<br>
	 * カンマ区切りされているプロパティを文字列配列として取得します。<br>
	 * @param name プロパティ定義名
	 * @return プロパティ値
	 */
	public static String[] getStrings(ConfigName name) {
		if (name == null) {
			throw new NullPointerException();
		}
		return getStrings(name.key());
	}

	/**
	 * 指定キーのプロパティを取得します。<br>
	 * @param key プロパティキー
	 * @return プロパティ値
	 */
	private static boolean getBoolean(String key) {
		String value = getString(key);
		return value == null ? false : Boolean.valueOf(value);
	}

	/**
	 * 指定キーのプロパティを取得します。<br>
	 * @param name プロパティ定義名
	 * @return プロパティ値
	 */
	public static boolean getBoolean(ConfigName name) {
		if (name == null) {
			throw new NullPointerException();
		}
		return getBoolean(name.key());
	}

	/**
	 * 指定キーのプロパティを取得します。<br>
	 * @param key プロパティキー
	 * @return プロパティ値
	 */
	private static long getLong(String key) {
		String value = getString(key);
		return value == null ? 0 : Long.parseLong(value);
	}

	/**
	 * 指定キーのプロパティを取得します。<br>
	 * @param name プロパティ定義名
	 * @return プロパティ値
	 */
	public static long getLong(ConfigName name) {
		if (name == null) {
			throw new NullPointerException();
		}
		return getLong(name.key());
	}

	/**
	 * 指定キーのプロパティを取得します。<br>
	 * @param key プロパティキー
	 * @return プロパティ値
	 */
	private static int getInteger(String key) {
		String value = getString(key);
		return value == null ? 0 : Integer.parseInt(value);
	}

	/**
	 * 指定キーのプロパティを取得します。<br>
	 * @param name プロパティ定義名
	 * @return プロパティ値
	 */
	public static int getInteger(ConfigName name) {
		if (name == null) {
			throw new NullPointerException();
		}
		return getInteger(name.key());
	}
}
