package org.phosphoresce.webcore.core;

import java.io.File;
import java.security.SecureRandom;

import org.phosphoresce.lib.commons.util.FileUtil;
import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.exception.WebcoreCoreModuleException;

/**
 * テンポラリリソース管理クラス<br>
 * <br>
 * 環境定義されたテンポラリリソースパスに対するアクセスインタフェースが提供されます。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/11/09	Kitagawa		新規作成
 * 2013/05/17	Kitagawa		全体的に再構築
 *-->
 */
public class TemporaryResource {

	/** テンポラリリソース基底パス */
	public static final String TEMPORARY_PATH = Config.getString(ConfigName.COMMON_GENERAL_TEMPORARY_RESOURCE);

	/**
	 * コンストラクタ<br>
	 */
	private TemporaryResource() {
		super();
	}

	/**
	 * 指定されたパスのファイルオブジェクトを取得します。<br>
	 * @param path テンポラリリソースパス(基底パスからの相対パス)
	 * @return テンポラリリソースファイルオブジェクト
	 */
	public static File getFile(String path) {
		File root = new File(TEMPORARY_PATH);
		if (!root.exists()) {
			FileUtil.mkdir(root);
		}
		return new File(FileUtil.connectPath(TEMPORARY_PATH, path));
	}

	/**
	 * 指定されたパスのリソースが存在するか判定します。<br>
	 * @param path テンポラリリソースパス(基底パスからの相対パス)
	 * @return リソースが存在する場合にtrueを返却します
	 */
	public static boolean isExist(String path) {
		return getFile(path).exists();
	}

	/**
	 * 指定されたパスのリソースがディレクトリであるか判定します。<br>
	 * @param path テンポラリリソースパス(基底パスからの相対パス)
	 * @return リソースがディレクトリである場合にtrueを返却します
	 */
	public static boolean isDirectory(String path) {
		return getFile(path).isDirectory();
	}

	/**
	 * 指定されたパスのリソースがファイルであるか判定します。<br>
	 * @param path テンポラリリソースパス(基底パスからの相対パス)
	 * @return リソースがファイルである場合にtrueを返却します
	 */
	public static boolean isFile(String path) {
		return getFile(path).isFile();
	}

	/**
	 * 指定されたパスのリソースを削除します。<br>
	 * @param path テンポラリリソースパス(基底パスからの相対パス)
	 */
	public static void delete(String path) {
		File file = getFile(path);
		if (file.exists()) {
			file.delete();
		}
	}

	/**
	 * テンポラリリソースファイルをすべて削除します。<br>
	 */
	public static void deleteAll() {
		File root = new File(TEMPORARY_PATH);
		if (root.exists()) {
			for (File resource : root.listFiles()) {
				FileUtil.delete(resource);
			}
		}
	}

	/**
	 * 新規にテンポラリリソースを生成します。<br>
	 * @return テンポラリリソースファイルオブジェクト
	 */
	private static File createNewTemporaryFile() {
		try {
			File file = null;
			SecureRandom random = new SecureRandom();
			while (true) {
				long n = random.nextLong();
				if (n == Long.MIN_VALUE) {
					n = 0;
				} else {
					n = Math.abs(n);
				}
				String filename = StringUtil.toMD5Hash(String.valueOf(System.currentTimeMillis()) + String.valueOf(n));
				file = getFile(filename);
				if (!file.exists()) {
					file.createNewFile();
					break;
				}
			}
			return file;
		} catch (Throwable e) {
			throw new WebcoreCoreModuleException("FCRE00038", e);
		}
	}

	/**
	 * 新規にテンポラリリソースを生成してリソース名を返却します。<br>
	 * @return テンポラリリソースファイル名
	 */
	public static String createNewTemporary() {
		return createNewTemporaryFile().getName();
	}
}
