package org.phosphoresce.webcore.core;

import java.io.IOException;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.phosphoresce.lib.commons.util.DOM4JUtil;
import org.phosphoresce.lib.commons.util.StringUtil;

/**
 * アプリケーション環境変数保持クラス<br>
 * <br>
 * 当クラスはアプリケーション起動時に初期化され、アプリケーション実行中は永続的に情報を保持し続けます。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/07/06	Kitagawa		新規作成
 *-->
 */
public class Environment implements Serializable, GlobalConstants {

	/** サーバーホスト */
	private static String serverHost;

	/** コンテンツパス */
	private static String contentPath;

	/** WEB-INFパス */
	private static String webinfPath;

	/** 環境設定リソースパス */
	private static String configPath;

	/** コンテキストパス */
	private static String contextPath;

	/** 開発モード */
	private static boolean developMode;

	/** アプリケーション起動時刻 */
	private static long startedTime;

	/**
	 * コンストラクタ<br>
	 */
	private Environment() {
		super();
	}

	/**
	 * 環境変数で管理されている値を動的に文字列にバインドします。<br>
	 * @param value バインド対象文字列
	 * @return 編集後文字列
	 */
	public static String bindEnv(String value) {
		Map<String, String> map = new HashMap<String, String>();
		map.put("webinf", webinfPath);
		map.put("content", contentPath);
		map.put("server", serverHost);
		map.put("context", contextPath);
		return StringUtil.bind(value, map);
	}

	/**
	 * 環境変数内容をMapオブジェクトとして取得します。<br>
	 * @return Mapオブジェクト
	 */
	public static Map<String, Object> getMap() {
		Map<String, Object> map = new HashMap<String, Object>();
		map.put("serverHost", serverHost);
		map.put("contentPath", contentPath);
		map.put("webinfPath", webinfPath);
		map.put("configPath", configPath);
		map.put("contextPath", contextPath);
		map.put("developMode", developMode);
		map.put("startedTime", startedTime);
		return map;
	}

	/**
	 * 指定されたドキュメント内の動的な文字列を設定し妥当な文字列バインドを行った上で提供します。<br>
	 * 当メソッドは前提として環境設定オブジェクトが初期化されている必要があります。<br>
	 * @param document 対象ドキュメント
	 * @return 編集後ドキュメント
	 * @throws IOException ドキュメントオブジェクトのXML文字列解析が行えなかった場合にスローされます
	 * @throws DocumentException 文字列編集後、XMLのパースが正常に行えなかった場合にスローされます
	 */
	public static org.dom4j.Document bindEnv(org.dom4j.Document document) throws DocumentException, IOException {
		return DocumentHelper.parseText(bindEnv(DOM4JUtil.toXMLString(document)));
	}

	/**
	 * 指定されたドキュメント内の動的な文字列を設定し妥当な文字列バインドを行った上で提供します。<br>
	 * 当メソッドは前提として環境設定オブジェクトが初期化されている必要があります。<br>
	 * @param document 対象ドキュメント
	 * @return 編集後ドキュメント
	 * @throws IOException ドキュメントオブジェクトのXML文字列解析が行えなかった場合にスローされます
	 * @throws DocumentException 文字列編集後、XMLのパースが正常に行えなかった場合にスローされます
	 */
	public static org.w3c.dom.Document bindEnv(org.w3c.dom.Document document) throws DocumentException, IOException {
		return DOM4JUtil.convertW3C(DocumentHelper.parseText(bindEnv(DOM4JUtil.toXMLString(document))));
	}

	/**
	 * 指定されたプロパティ内の動的な文字列を設定し妥当な文字列バインドを行った上で提供します。<br>
	 * 当メソッドは前提として環境設定オブジェクトが初期化されている必要があります。<br>
	 * @param properties 対象プロパティ
	 * @return 編集後プロパティ
	 */
	public static Properties bindEnv(Properties properties) {
		Properties result = new Properties();
		for (Object key : properties.keySet()) {
			String value = properties.getProperty((String) key);
			result.put(key, bindEnv(value));
		}
		return result;
	}

	/**
	 * サーバーホストを取得します。<br>
	 * @return サーバーホスト
	 */
	public static String getServerHost() {
		return serverHost;
	}

	/**
	 * サーバーホストを設定します。<br>
	 * @param serverHost サーバーホスト
	 */
	public static void setServerHost(String serverHost) {
		Environment.serverHost = serverHost;
	}

	/**
	 * コンテンツパスを取得します。<br>
	 * @return コンテンツパス
	 */
	public static String getContentPath() {
		return contentPath;
	}

	/**
	 * コンテンツパスを設定します。<br>
	 * @param contentPath コンテンツパス
	 */
	public static void setContentPath(String contentPath) {
		Environment.contentPath = contentPath;
	}

	/**
	 * WEB-INFパスを取得します。<br>
	 * @return WEB-INFパス
	 */
	public static String getWebinfPath() {
		return webinfPath;
	}

	/**
	 * WEB-INFパスを設定します。<br>
	 * @param webinfPath WEB-INFパス
	 */
	public static void setWebinfPath(String webinfPath) {
		Environment.webinfPath = webinfPath;
	}

	/**
	 * 環境設定リソースパスを取得します。<br>
	 * @return 環境設定リソースパス
	 */
	public static String getConfigPath() {
		return configPath;
	}

	/**
	 * 環境設定リソースパスを設定します。<br>
	 * @param configPath 環境設定リソースパス
	 */
	public static void setConfigPath(String configPath) {
		Environment.configPath = configPath;
	}

	/**
	 * コンテキストパスを取得します。<br>
	 * @return コンテキストパス
	 */
	public static String getContextPath() {
		return contextPath;
	}

	/**
	 * コンテキストパスを設定します。<br>
	 * @param contextPath コンテキストパス
	 */
	public static void setContextPath(String contextPath) {
		Environment.contextPath = contextPath;
	}

	/**
	 * 開発モードを取得します。<br>
	 * @return 開発モード
	 */
	public static boolean isDevelopMode() {
		return developMode;
	}

	/**
	 * 開発モードを設定します。<br>
	 * @param developMode 開発モード
	 */
	public static void setDevelopMode(boolean developMode) {
		Environment.developMode = developMode;
	}

	/**
	 * アプリケーション起動時刻を取得します。<br>
	 * @return アプリケーション起動時刻
	 */
	public static long getStartedTime() {
		return startedTime;
	}

	/**
	 * アプリケーション起動時刻を設定します。<br>
	 * @param startedTime アプリケーション起動時刻
	 */
	public static void setStartedTime(long startedTime) {
		Environment.startedTime = startedTime;
	}
}
