package org.phosphoresce.webcore.core;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.InputStream;

import org.phosphoresce.lib.commons.util.FileUtil;
import org.phosphoresce.lib.commons.util.ResourceUtil;
import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.webcore.core.config.Config;
import org.phosphoresce.webcore.core.config.ExntendsConfiguration;
import org.phosphoresce.webcore.core.config.MIMETypes;
import org.phosphoresce.webcore.core.config.StringResource;
import org.phosphoresce.webcore.core.daemon.GarbageCollectionDaemon;
import org.phosphoresce.webcore.core.exception.WebcoreConfigurationException;
import org.phosphoresce.webcore.core.logger.CodeConvertLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.joran.JoranConfigurator;

/**
 * アプリケーション環境初期化クラス<br>
 * <br>
 * 当クラスはアプリケーション起動時に初期化されている環境変数の情報を基に各種クラスの初期化を実施します。<br>
 * スタンドアローンモードでの実行の場合に限り、環境変数の初期化処理についても当クラス処理にて実行されます。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2010/11/09	Kitagawa		新規作成
 *-->
 */
public class Configuration implements GlobalConstants {

	/** ロガーオブジェクト */
	private static CodeConvertLogger log = new CodeConvertLogger(LoggerFactory.getLogger(Configuration.class));

	/** 絶対コンテンツパス定義フラグ */
	private static boolean absoluteContentPath = false;

	/** 絶対WEB-INFパス定義フラグ */
	private static boolean absoluteWebinfPath = false;

	/** スタンドアローンモードコンテンツパス */
	public static final String STANDALONE_CONTENT_PATH = FileUtil.connectPath(new String[] { new File("").getAbsolutePath(), "content" });

	/** スタンドアローンモードWEB-INFパス */
	public static final String STANDALONE_WEBINF_PATH = FileUtil.connectPath(new String[] { STANDALONE_CONTENT_PATH, "WEB-INF" });

	/** スタンドアローンモード環境設定リソースパス */
	public static final String STANDALONE_CONFIG_PATH = FileUtil.connectPath(new String[] { STANDALONE_WEBINF_PATH, "config", Config.DEFAULT_CONFIG });

	/** スタンドアローンモードホスト名 */
	public static final String STANDALONE_SERVER_HOST = "localhost";

	/**
	 * コンストラクタ<br>
	 */
	private Configuration() {
		super();
	}

	/**
	 * アプリケーションの初期化処理を行います。<br>
	 * @param standalone スタンドアローンモードでの初期化を行う場合にtrueを設定
	 */
	public synchronized static void configure(boolean standalone) {
		try {
			configreEvironment(standalone);
			configureConfig(standalone);
			configureSystemProperty();
			configureDevelopMode();
			configureString();
			configureLogger();
			configureMIMETypes();
			configureGCDaemon();
			configureTemporaryResource();
			configureExtendsConfigure();
			log.output("FCRE00001");

			if (standalone) {
				log.output("FCRE00004");
			}
			if (absoluteContentPath) {
				log.output("FCRE00005", Environment.getContentPath());
			}
			if (absoluteWebinfPath) {
				log.output("FCRE00006", Environment.getWebinfPath());
			}
		} catch (Throwable e) {
			throw new WebcoreConfigurationException("FCRE00021", e);
		}
	}

	/**
	 * アプリケーションの初期化処理を行います。<br>
	 * 当初期化処理はアプリケーションサーバでのアプリケーション実行用の処理が実行されます。<br>
	 */
	public synchronized static void configure() {
		configure(false);
	}

	/**
	 * アプリケーション実行環境変数の初期化を行います。<br>
	 * @param standalone スタンドアローンモードでの実行の場合はtrueを設定
	 * @throws Throwable 初期化処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void configreEvironment(boolean standalone) throws Throwable {
		if (standalone) {
			Environment.setContentPath(STANDALONE_CONTENT_PATH);
			Environment.setWebinfPath(STANDALONE_WEBINF_PATH);
			Environment.setConfigPath(STANDALONE_CONFIG_PATH);
			Environment.setServerHost(STANDALONE_SERVER_HOST);
		} else {
			// スタンドアローンモード以外の場合はアプリケーション起動時に初期化を行う
		}
	}

	/**
	 * 環境設定オブジェクトの初期化処理を行います。<br>
	 * @param standalone スタンドアローンモードでの実行の場合はtrueを設定
	 * @throws Throwable 初期化処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void configureConfig(boolean standalone) throws Throwable {
		if (standalone) {
			Config.initialize(STANDALONE_CONFIG_PATH);
		} else {
			Config.initialize(Environment.getConfigPath());
		}
	}

	/**
	 * システムプロパティの初期化処理を行います。<br>
	 * @throws Throwable 初期化処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void configureSystemProperty() throws Throwable {
		System.setProperty("file.encoding", Config.getString(ConfigName.COMMON_CHARSET_DEFAULT));
	}

	/**
	 * 開発モードの初期化処理を行います。<br>
	 * @throws Throwable 初期化処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void configureDevelopMode() throws Throwable {
		for (String value : Config.getStrings(ConfigName.COMMON_ABSOLUTE_CONTENT_PATH)) {
			if (!StringUtil.isEmpty(value)) {
				if (new File(value).exists()) {
					Environment.setContentPath(value);
					absoluteContentPath = true;
					break;
				}
			}
		}
		for (String value : Config.getStrings(ConfigName.COMMON_ABSOLUTE_WEBINF_PATH)) {
			if (!StringUtil.isEmpty(value)) {
				if (new File(value).exists()) {
					Environment.setWebinfPath(value);
					absoluteWebinfPath = true;
					break;
				}
			}
		}
	}

	/**
	 * 文字列情報の初期化処理を行います。<br>
	 * @throws Throwable 初期化処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void configureString() throws Throwable {
		StringResource.clear();
		for (String value : Config.getStrings(ConfigName.COMMON_RESOURCE_STRING)) {
			if (!StringUtil.isEmpty(value)) {
				try {
					StringResource.initialize(value, true);
					log.output("FCRE00034", value);
				} catch (FileNotFoundException e) {
					log.output("FCRE00028", value);
				}
			}
		}
	}

	/**
	 * ロガーエンジンの初期化処理を行います。<br>
	 * @throws Throwable 初期化処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void configureLogger() throws Throwable {
		LoggerContext context = (LoggerContext) LoggerFactory.getILoggerFactory();
		JoranConfigurator configurator = new JoranConfigurator();
		configurator.setContext(context);
		context.reset();

		String resourcePath = Config.getString(ConfigName.COMMON_RESOURCE_LOGBACK);
		String resourceText = Environment.bindEnv(ResourceUtil.getText(resourcePath));

		InputStream stream = new ByteArrayInputStream(resourceText.getBytes(Config.getString(ConfigName.COMMON_CHARSET_DEFAULT)));
		configurator.doConfigure(stream);
		log = new CodeConvertLogger(LoggerFactory.getLogger(Configuration.class));

		Logger nlog = LoggerFactory.getLogger(Configuration.class);
		nlog.info(StringUtil.padding("", 320, '*', true));

		log.output("FCRE00035", resourcePath);
		log.output("FCRE00002");
	}

	/**
	 * MIMEタイプ定義の初期化処理を行います。<br>
	 * @throws Throwable 初期化処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void configureMIMETypes() throws Throwable {
		MIMETypes.initialize();
	}

	/**
	 * ガーベージコレクションデーモンサービスを初期化します。<br>
	 * @throws Throwable 初期化処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void configureGCDaemon() throws Throwable {
		if (Config.getBoolean(ConfigName.COMMON_DAEMON_GC)) {
			log.output("FCRE00017", ConfigName.COMMON_DAEMON_GC);
			GarbageCollectionDaemon.execute();
			log.output("FCRE00019");
		} else {
			log.output("FCRE00018");
		}
	}

	/**
	 * テンポラリリソースを初期化します。<br>
	 * @throws Throwable 初期化処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void configureTemporaryResource() throws Throwable {
		TemporaryResource.deleteAll();
		log.output("FCRE00039");
	}

	/**
	 * アプリケーション起動時拡張初期化対象クラスの処理を行います。<br>
	 * @throws Throwable 初期化処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	@SuppressWarnings("unchecked")
	public synchronized static void configureExtendsConfigure() throws Throwable {
		for (String value : Config.getStrings(ConfigName.COMMON_INITIALIZE_CLASSES)) {
			if (!StringUtil.isEmpty(value) && !value.startsWith("#")) {
				log.output("FCRE00029", value);
				Class<ExntendsConfiguration> clazz = (Class<ExntendsConfiguration>) Class.forName(value);
				clazz.getMethod("configure", new Class[0]).invoke(null, new Object[0]);
				log.output("FCRE00030", value);
			}
		}
	}

	/**
	 * アプリケーション環境設定を破棄します。<br>
	 */
	public synchronized static void destroy() {
		try {
			destroyGCDaemon();
			destroyExtendsConfigure();

			MIMETypes.destroy();
			//StringResource.destroy(); // Bootstrapサーブレット側でのdestroyとする
			log.output("FCRE00013");

			long milisecond = System.currentTimeMillis() - Environment.getStartedTime();
			long second = milisecond / 1000;
			long minute = second / 60;
			long hour = minute / 60;
			minute = minute - (hour * 60);
			second = second - (hour * 60 * 60) - (minute * 60);
			milisecond = milisecond - (hour * 60 * 60 * 1000) - (minute * 60 * 1000) - (second * 1000);
			log.output("FCRE00014", String.valueOf(hour), String.valueOf(minute), String.valueOf(second));
		} catch (Throwable e) {
			throw new WebcoreConfigurationException("FCRE00026", e);
		}
	}

	/**
	 * ガーベージコレクションデーモンサービスを初期化します。<br>
	 * @throws Throwable 初期化処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void destroyGCDaemon() throws Throwable {
		if (Config.getBoolean(ConfigName.COMMON_DAEMON_GC)) {
			GarbageCollectionDaemon.stopDaemon();
			log.output("FCRE00020");
		}
	}

	/**
	 * アプリケーション停止時拡張対象クラスの処理を行います。<br>
	 * @throws Throwable 初期化処理中に予期せぬエラーが発生した場合にスローされます 
	 */
	public synchronized static void destroyExtendsConfigure() throws Throwable {
		for (String value : Config.getStrings(ConfigName.COMMON_INITIALIZE_CLASSES)) {
			if (!StringUtil.isEmpty(value) && !value.startsWith("#")) {
				log.output("FCRE00031", value);
				Class<?> clazz = (Class<?>) Class.forName(value);
				clazz.getMethod("destroy", new Class[0]).invoke(null, new Object[0]);
				log.output("FCRE00032", value);
			}
		}
	}
}
