package org.phosphoresce.lib.poi.dyna;

import org.phosphoresce.lib.poi.PoiGlobal;
import org.phosphoresce.lib.poi.adapter.PoiCell;
import org.phosphoresce.lib.poi.exception.PoiException;

/**
 * 動的Excel生成ユーティリティクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2005/11/21	Kitagawa		新規作成
 * 2006/03/09	Kitagawa		動的値セット処理の例外をPoiExceptionに変更
 * 2007/01/16	Kitagawa		動的セルのコメント記述に対応(コメントは動的セルにおいて「#」以降とする)
 * 2012/07/09	Kitagawa		バージョン2にアップデート
 *-->
 */
class PoiDynaUtil implements PoiGlobal {

	/**
	 * コンストラクタ<br>
	 */
	private PoiDynaUtil() {
		super();
	}

	/**
	 * 指定された動的セルの指定トークンインデックスの定義文字列を取得します。<br>
	 * 先頭インデックス(0)は動的セル識別プレフィクスとなります。<br>
	 * @param cell セルオブジェクト
	 * @param index トークンインデックス
	 * @return 指定されtトークンインデックスの定義文字列(第二階層定義文字列が含まれます)
	 */
	private static String getDynaDefinition(PoiCell cell, int index) {
		if (cell == null) {
			return "";
		}
		String[] tokens = cell.getString().split(DYNA_DELIM);
		if (index > tokens.length - 1) {
			return "";
		} else {
			return tokens[index];
		}
	}

	/**
	 * 指定された動的セルの指定トークンインデックスの定義文字列内に定義されている第二階層定義を取得します。<br>
	 * @param cell セルオブジェクト
	 * @param index トークンインデックス
	 * @param subindex サブトークンインデックス
	 * @return 指定されtトークンインデックスの第二階層定義文字列
	 */
	private static String getDynaSubDefinition(PoiCell cell, int index, int subindex) {
		String[] tokens = getDynaDefinition(cell, index).split(DYNA_SUBDELIM);
		if (subindex > tokens.length - 1) {
			return "";
		} else {
			return tokens[subindex];
		}
	}

	/**
	 * 指定された動的セルの指定トークンインデックスの定義文字列を取得します。<br>
	 * 先頭インデックス(0)は動的セル識別プレフィクスとなります。<br>
	 * @param cell セルオブジェクト
	 * @param index トークンインデックス
	 * @return 指定されtトークンインデックスのメイン定義文字列
	 */
	private static String getDynaMainDefinition(PoiCell cell, int index) {
		return getDynaSubDefinition(cell, index, 0);
	}

	/**
	 * 動的セルのキーを取得します。<br>
	 * @param cell セルオブジェクト
	 * @return セルオブジェクトに設定されている動的セルキー
	 */
	public static String getDynaKey(PoiCell cell) {
		return getDynaSubDefinition(cell, 2, 0);
	}

	/**
	 * 動的セルのキーを取得します。<br>
	 * @param cell セルオブジェクト
	 * @return セルオブジェクトに設定されている動的セルキー
	 */
	public static Object getDynaDefaultValue(PoiCell cell) {
		String type = getDynaSubDefinition(cell, 2, 1);
		String value = getDynaSubDefinition(cell, 2, 2);
		if (DYNA_DEFAULT_VALUE_TYPE_NUMERIC.equalsIgnoreCase(type)) {
			return Double.parseDouble(value);
		} else {
			return value;
		}
	}

	/**
	 * 指定されたセルオブジェクトが動的セルであるか判定します。<br>
	 * @param cell セルオブジェクト
	 * @return セルオブジェクトが動的セルである場合、trueを返却
	 */
	public static boolean isDynaCell(PoiCell cell) {
		return DYNA_PREFIX.equals(getDynaMainDefinition(cell, 0));
	}

	/**
	 * 指定されたセルが動的値設定セルであるか判定します。<br>
	 * @param cell セルオブジェクト
	 * @return 動的値設定セルの場合にtrueを返却
	 */
	public static boolean isDynaValueCell(PoiCell cell) {
		return DYNA_TYPE_VALUE.equals(getDynaMainDefinition(cell, 1));
	}

	/**
	 * 指定されたセルオブジェクトのタイプが動的リスト開始セルであるか判定します。<br>
	 * @param cell セルオブジェクト
	 * @return セルオブジェクトのタイプが動的値セルである場合trueを返却
	 */
	public static boolean isDynaListStartCell(PoiCell cell) {
		return DYNA_TYPE_LIST_START.equals(getDynaMainDefinition(cell, 1));
	}

	/**
	 * 指定されたセルオブジェクトのタイプが動的リスト終了セルであるか判定します。<br>
	 * @param cell セルオブジェクト
	 * @return セルオブジェクトのタイプが動的値セルである場合trueを返却
	 */
	public static boolean isDynaListEndCell(PoiCell cell) {
		return DYNA_TYPE_LIST_END.equals(getDynaMainDefinition(cell, 1));
	}

	/**
	 * 指定されたリスト開始セルに対応する終了位置を取得します。<br>
	 * @param cell セルオブジェクト
	 * @param range 検索範囲
	 * @return リスト終了行
	 */
	public static int getListEndIndex(PoiCell cell, PoiDynaRange range) {
		if (!isDynaListStartCell(cell)) {
			throw new PoiException(cell.getString() + " is not list start definition");
		}
		String dynaKey = getDynaMainDefinition(cell, 2);
		for (int rowIndex = cell.getPoiCell().getRowIndex(); rowIndex <= range.getRowIndexTo(); rowIndex++) {
			PoiCell check = cell.getSheet().getCell(rowIndex, 0);
			if (isDynaListEndCell(check) && dynaKey.equals(getDynaMainDefinition(check, 2))) {
				return rowIndex;
			}
		}
		throw new PoiException("not found list end definition of " + cell.getString());
	}
}
