package org.phosphoresce.lib.poi.dyna;

import java.io.Serializable;

import org.phosphoresce.lib.poi.adapter.PoiRange;

/**
 * 動的設定範囲情報クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/07/09	Kitagawa		新規作成(バージョン2にアップデート)
 *-->
 */
class PoiDynaRange implements Serializable {

	/** 親範囲オブジェクト */
	private PoiDynaRange parent;

	/** 行開始位置(0～) */
	private int rowIndexFrom;

	/** 列開始位置(0～) */
	private int colIndexFrom;

	/** 行終了位置(0～) */
	private int rowIndexTo;

	/** 列終了位置(0～) */
	private int colIndexTo;

	/** 処理位置 */
	private PoiDynaPoint processPoint;

	/**
	 * コンストラクタ<br>
	 * @param parent 親範囲オブジェクト
	 * @param rowIndexFrom 行開始位置(0～)
	 * @param colIndexFrom 列開始位置(0～)
	 * @param rowIndexTo 行終了位置(0～)
	 * @param colIndexTo 列終了位置(0～)
	 */
	public PoiDynaRange(PoiDynaRange parent, int rowIndexFrom, int colIndexFrom, int rowIndexTo, int colIndexTo) {
		super();
		this.parent = parent;
		this.rowIndexFrom = rowIndexFrom;
		this.colIndexFrom = colIndexFrom;
		this.rowIndexTo = rowIndexTo;
		this.colIndexTo = colIndexTo;
		if (parent != null) {
			this.processPoint = parent.processPoint;
		} else {
			this.processPoint = new PoiDynaPoint(rowIndexFrom, colIndexFrom);
		}
	}

	/**
	 * コンストラクタ<br>
	 * @param rowIndexFrom 行開始位置(0～)
	 * @param colIndexFrom 列開始位置(0～)
	 * @param rowIndexTo 行終了位置(0～)
	 * @param colIndexTo 列終了位置(0～)
	 */
	public PoiDynaRange(int rowIndexFrom, int colIndexFrom, int rowIndexTo, int colIndexTo) {
		this(null, rowIndexFrom, colIndexFrom, rowIndexTo, colIndexTo);
	}

	/**
	 * 通常範囲オブジェクトに変換します。<br>
	 * @return 通常範囲オブジェクト
	 */
	public PoiRange toRange() {
		return new PoiRange(rowIndexFrom, colIndexFrom, rowIndexTo, colIndexTo);
	}

	/**
	 * 範囲情報の高さを取得します(最小値=1)。<br>
	 * @return 範囲情報の高さ
	 */
	public int getHeight() {
		return rowIndexTo - rowIndexFrom + 1;
	}

	/**
	 * 範囲情報の幅を取得します(最小値=1)。<br>
	 * @return 範囲情報の幅
	 */
	public int getWidth() {
		return colIndexTo - colIndexFrom + 1;
	}

	/**
	 * 親範囲オブジェクトを取得します。<br>
	 * @return 親範囲オブジェクト
	 */
	public PoiDynaRange getParent() {
		return parent;
	}

	/**
	 * 行開始位置(0～)を取得します。<br>
	 * @return 行開始位置(0～)
	 */
	public int getRowIndexFrom() {
		return rowIndexFrom;
	}

	/**
	 * 列開始位置(0～)を取得します。<br>
	 * @return 列開始位置(0～)
	 */
	public int getColIndexFrom() {
		return colIndexFrom;
	}

	/**
	 * 行終了位置(0～)を取得します。<br>
	 * @return 行終了位置(0～)
	 */
	public int getRowIndexTo() {
		return rowIndexTo;
	}

	/**
	 * 列終了位置(0～)を取得します。<br>
	 * @return 列終了位置(0～)
	 */
	public int getColIndexTo() {
		return colIndexTo;
	}

	/**
	 * 処理位置を取得します。<br>
	 * @return 処理位置
	 */
	public PoiDynaPoint getProcessPoint() {
		return processPoint;
	}

	/**
	 * 処理行が存在するか判定します。<br>
	 * @return 処理行が存在する場合にtrueを返却
	 */
	public boolean hasMoreProcessRow() {
		return processPoint.getProcessRowIndex() < rowIndexTo + 1;
	}

	/**
	 * 処理列が存在するか判定します。<br>
	 * @return 処理列が存在する場合にtrueを返却
	 */
	public boolean hasMoreProcessCol() {
		return processPoint.getProcessColIndex() < colIndexTo + 1;
	}

	/**
	 * 処理行を次行に移動します。<br>
	 */
	public void nextProcessRow() {
		processPoint.setProcessRowIndex(processPoint.getProcessRowIndex() + 1);
		processPoint.setProcessColIndex(colIndexFrom);
	}

	/**
	 * 処理行を前行に移動します。<br>
	 */
	public void prevProcessRow() {
		processPoint.setProcessRowIndex(processPoint.getProcessRowIndex() - 1);
		processPoint.setProcessColIndex(colIndexFrom);
	}

	/**
	 * 処理列を次列に移動します。<br>
	 */
	public void nextProcessCol() {
		processPoint.setProcessColIndex(processPoint.getProcessColIndex() + 1);
	}

	/**
	 * 処理列を前列に移動します。<br>
	 */
	public void prevProcessCol() {
		processPoint.setProcessColIndex(processPoint.getProcessColIndex() - 1);
	}

	/**
	 * 指定されたサイズ分高さ範囲を下方向、右方向に拡張します。<br>
	 * 負数を指定した場合は縮小します。<br>
	 * @param rowSize 行拡張サイズ
	 * @param colSize 列拡張サイズ
	 * @param parentExpand 親範囲に対しても処理を行う場合にtrueを指定
	 */
	public void expand(int rowSize, int colSize, boolean parentExpand) {
		rowIndexTo += rowSize;
		colIndexTo += colSize;
		if (parentExpand && parent != null) {
			parent.expand(rowSize, colSize, parentExpand);
		}
	}

	/**
	 * 指定された行方向、列方向にシフト範囲のシフト処理を行います。<br>
	 * @param rowShift 行方向シフト値
	 * @param colShift 列方向シフト値
	 * @param parentExpand 親範囲に対しても処理を行う場合にtrueを指定
	 */
	public void shift(int rowShift, int colShift, boolean parentExpand) {
		colIndexFrom += colShift;
		colIndexTo += colShift;
		rowIndexFrom += rowShift;
		rowIndexTo += rowShift;
		if (parentExpand && parent != null) {
			parent.shift(rowShift, colShift, parentExpand);
		}
	}

	/**
	 * ハッシュコードを取得します。<br>
	 * @return ハッシュコード
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + colIndexFrom;
		result = prime * result + colIndexTo;
		result = prime * result + ((parent == null) ? 0 : parent.hashCode());
		result = prime * result + ((processPoint == null) ? 0 : processPoint.hashCode());
		result = prime * result + rowIndexFrom;
		result = prime * result + rowIndexTo;
		return result;
	}

	/**
	 * オブジェクト等価比較を行います。<br>
	 * @return 等価の場合にtrueを返却
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object object) {
		if (this == object) {
			return true;
		}
		if (object == null) {
			return false;
		}
		if (getClass() != object.getClass()) {
			return false;
		}
		PoiDynaRange other = (PoiDynaRange) object;
		if (colIndexFrom != other.colIndexFrom) {
			return false;
		}
		if (colIndexTo != other.colIndexTo) {
			return false;
		}
		if (parent == null) {
			if (other.parent != null) {
				return false;
			}
		} else if (!parent.equals(other.parent)) {
			return false;
		}
		if (processPoint == null) {
			if (other.processPoint != null) {
				return false;
			}
		} else if (!processPoint.equals(other.processPoint)) {
			return false;
		}
		if (rowIndexFrom != other.rowIndexFrom) {
			return false;
		}
		if (rowIndexTo != other.rowIndexTo) {
			return false;
		}
		return true;
	}

	/**
	 * クラス情報を文字列で取得します。<br>
	 * @return クラス情報文字列
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("{");
		builder.append("processPoint=");
		builder.append(processPoint);
		builder.append(", ");
		builder.append("rowIndexFrom=");
		builder.append(rowIndexFrom);
		builder.append(", ");
		builder.append("colIndexFrom=");
		builder.append(colIndexFrom);
		builder.append(", ");
		builder.append("rowIndexTo=");
		builder.append(rowIndexTo);
		builder.append(", ");
		builder.append("colIndexTo=");
		builder.append(colIndexTo);
		builder.append(", ");
		builder.append("height=");
		builder.append(getHeight());
		builder.append(", ");
		builder.append("width=");
		builder.append(getWidth());
		builder.append(", ");
		builder.append("parent=");
		builder.append(getParent());
		builder.append("}");
		return builder.toString();
	}
}
