package org.phosphoresce.lib.poi.dyna;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.LinkedList;
import java.util.List;

import org.apache.poi.ss.usermodel.Name;
import org.phosphoresce.lib.poi.PoiGlobal;
import org.phosphoresce.lib.poi.adapter.PoiSheet;
import org.phosphoresce.lib.poi.adapter.PoiWorkbook;
import org.phosphoresce.lib.poi.exception.PoiException;
import org.phosphoresce.lib.poi.print.PoiPrintConfig;
import org.phosphoresce.lib.poi.util.PoiStringUtil;
import org.phosphoresce.lib.poi.util.PoiUtil;

/**
 * 動的バインド出力クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2005/12/07	Kitagawa		新規作成
 * 2006/01/18	Kitagawa		シート生成時にシートが未選択となるバグを修正
 * 2006/01/25	Kitagawa		シート生成時に複数シートとしてのExcelファイルとして生成可能に変更(I/F変更)
 * 2012/07/09	Kitagawa		バージョン2にアップデート(PoiDynaSheetから移管)
 * 2012/11/20	Kitagawa		POI3.8にて既知のバグとしてシート削除時の名前定義インデックスの破損がある為、テンプレートシートは暫定的に非表示対応とする
 * 2012/11/20	Kitagawa		タイトル列/行定義複写に対応
 * 2013/04/23	Kitagawa		名前定義オブジェクトからシート名が取得できない場合(対応シートインデックスが0未満)は処理をスキップするように修正
 *-->
 */
public class PoiDynaExporter implements PoiGlobal {

	/** テンプレートワークブック */
	private PoiWorkbook template;

	/** 出力情報コンテナリスト */
	private List<PoiDynaExportContainer> containers;

	/** テンプレートファイルシート全削除フラグ */
	private boolean baseSheetDeleteOnExport;

	/**
	 * コンストラクタ<br>
	 * @param template テンプレートワークブック
	 */
	public PoiDynaExporter(PoiWorkbook template) {
		super();
		this.template = template;
		this.containers = new LinkedList<PoiDynaExportContainer>();
		this.baseSheetDeleteOnExport = false;
	}

	/**
	 * コンストラクタ<br>
	 * @param template テンプレートワークブック
	 * @param baseSheetDeleteOnExport テンプレートファイルシート全削除フラグ
	 */
	public PoiDynaExporter(PoiWorkbook template, boolean baseSheetDeleteOnExport) {
		super();
		this.template = template;
		this.containers = new LinkedList<PoiDynaExportContainer>();
		this.baseSheetDeleteOnExport = baseSheetDeleteOnExport;
	}

	/**
	 * 動的生成シート件数を取得します。<br>
	 * @return 動的生成シート件数
	 */
	public int getExportSheetCount() {
		return containers.size();
	}

	/**
	 * 動的生成シート情報を追加します。<br>
	 * @param templateSheetIndex テンプレートシートインデックス
	 * @param sheetName 生成対象シート名
	 * @param dynaMap 動的値マップオブジェクト
	 * @param printConfig 印刷環境設定オブジェクト
	 */
	public void addExportSheet(int templateSheetIndex, String sheetName, PoiDynaMap dynaMap, PoiPrintConfig printConfig) {
		if (sheetName == null || sheetName.length() == 0) {
			throw new PoiException("sheetName is empty");
		}
		for (PoiDynaExportContainer container : containers) {
			if (sheetName.equals(container.getSheetName())) {
				throw new PoiException(sheetName + " is duplicate sheet name");
			}
		}
		containers.add(new PoiDynaExportContainer(templateSheetIndex, sheetName, dynaMap, printConfig));
	}

	/**
	 * 動的生成シート情報を追加します。<br>
	 * @param templateSheetIndex テンプレートシートインデックス
	 * @param sheetName 生成対象シート名
	 * @param dynaMap 動的値マップオブジェクト
	 */
	public void addExportSheet(int templateSheetIndex, String sheetName, PoiDynaMap dynaMap) {
		addExportSheet(templateSheetIndex, sheetName, dynaMap, null);
	}

	/**
	 * 動的生成シート情報を削除します。<br>
	 */
	public void clearExportSheet() {
		containers.clear();
	}

	/**
	 * クラスに設定されている動的バインド情報を利用してテンプレートワークブックを基に動的バインド処理を行います。<br>
	 * テンプレートとして設定されたワークブックオブジェクトがディープコピーされて処理が行われる為、元のワークブックへの影響はありません。<br>
	 */
	private PoiWorkbook processDynamicBind() {
		/*
		 * ワークブックオブジェクトクローン
		 */
		PoiWorkbook workbook = this.template.clone();

		for (PoiDynaExportContainer container : containers) {
			/*
			 * テンプレートシートコピー
			 */
			PoiSheet template = workbook.getSheet(container.getTemplateSheetIndex());
			workbook.getPoiWorkbook().cloneSheet(container.getTemplateSheetIndex());
			workbook.getPoiWorkbook().setActiveSheet(workbook.getPoiWorkbook().getNumberOfSheets() - 1);
			workbook.getPoiWorkbook().setSelectedTab(workbook.getPoiWorkbook().getNumberOfSheets() - 1);
			PoiSheet sheet = workbook.getSheet(workbook.getPoiWorkbook().getNumberOfSheets() - 1);

			/*
			 * シート名の設定
			 */
			sheet.setSheetName(container.getSheetName());

			/*
			 * 再計算設定
			 */
			sheet.getPoiSheet().setForceFormulaRecalculation(true);
			//if (sheet.getPoiSheet() instanceof HSSFSheet) {
			//	// HSSF時
			//	((HSSFSheet) sheet.getPoiSheet()).setForceFormulaRecalculation(true);
			//} else {
			//	// XSSF時
			//	((XSSFSheet) sheet.getPoiSheet()).setForceFormulaRecalculation(true);
			//}

			/*
			 * 印刷環境設定オブジェクトの適用
			 */
			if (container.getPrintConfig() != null) {
				container.getPrintConfig().apply(sheet);
			} else {
				PoiPrintConfig.create(template.getPoiSheet()).apply(sheet);
			}

			/*
			 * 動的値バインド処理
			 */
			new PoiDynaValueSetter(sheet, container.getDynaMap()).execute();

			/*
			 * 印刷範囲の設定
			 */
			// POI3.8ではシートクローンのタイミングで印刷範囲もコピーされる
			//String printArea = workbook.getPoiWorkbook().getPrintArea(container.getTemplateSheetIndex());
			//if (printArea != null && printArea.indexOf(template.getPoiSheet().getSheetName() + "!") >= 0) {
			//	printArea = printArea.substring((template.getPoiSheet().getSheetName() + "!").length());
			//	workbook.getPoiWorkbook().setPrintArea(sheet.getSheetIndex(), printArea);
			//}

			/*
			 * ワークブック名前定義設定
			 */
			for (int i = 0; i <= workbook.getPoiWorkbook().getNumberOfNames() - 1; i++) {
				Name name = workbook.getPoiWorkbook().getNameAt(i);
				if (name == null) {
					continue;
				}
				if (name.getNameName().startsWith("_")) {
					continue;
				}
				if (name.getSheetIndex() < 0) {
					continue;
				}
				if (name.getSheetName().equals(template.getPoiSheet().getSheetName())) {
					String nameFormula = name.getRefersToFormula();
					if (!nameFormula.startsWith(template.getPoiSheet().getSheetName())) {
						continue;
					}
					nameFormula = nameFormula.substring((template.getPoiSheet().getSheetName() + "!").length());

					if ("Print_Titles".equals(name.getNameName())) {
						/*
						 * 印刷タイトル定義
						 */
						String titleRowFormula = "";
						String titleColFormula = "";

						if (nameFormula.indexOf(",") >= 0) {
							// 行列タイトル指定有
							titleColFormula = nameFormula.substring(0, nameFormula.indexOf(","));
							titleRowFormula = nameFormula.substring(nameFormula.indexOf(",") + 1);
						} else if (!nameFormula.matches(".*[0-9]+.*")) {
							// 列タイトル指定有
							titleColFormula = nameFormula;
						} else {
							// 列タイトル指定無
							titleRowFormula = nameFormula;
						}

						// シート名が指定されている場合は除去
						if (titleColFormula.indexOf("!") >= 0) {
							titleColFormula = titleColFormula.substring(titleColFormula.indexOf("!") + 1);
						}
						if (titleRowFormula.indexOf("!") >= 0) {
							titleRowFormula = titleRowFormula.substring(titleRowFormula.indexOf("!") + 1);
						}

						int titleFromRow = -1;
						int titleFromCol = -1;
						int titleToRow = -1;
						int titleToCol = -1;

						// 列タイトル側設定
						if (!PoiStringUtil.isEmpty(titleColFormula)) {
							String titleFrom = titleColFormula.substring(0, titleColFormula.indexOf(":"));
							String titleTo = titleColFormula.substring(titleColFormula.indexOf(":") + 1);
							titleFromCol = PoiUtil.convertCol(titleFrom);
							titleToCol = PoiUtil.convertCol(titleTo);
						}
						// 行タイトル側設定
						if (!PoiStringUtil.isEmpty(titleRowFormula)) {
							String titleFrom = titleRowFormula.substring(0, titleRowFormula.indexOf(":"));
							String titleTo = titleRowFormula.substring(titleRowFormula.indexOf(":") + 1);
							titleFromRow = PoiUtil.convertRow(titleFrom);
							titleToRow = PoiUtil.convertRow(titleTo);
						}
						sheet.setTitle(titleFromCol, titleToCol, titleFromRow, titleToRow);
					} else {
						/*
						 * その他定義
						 */
						Name newName = workbook.getPoiWorkbook().createName();
						newName.setSheetIndex(sheet.getSheetIndex());
						newName.setNameName(name.getNameName());
						newName.setRefersToFormula(container.getSheetName() + "!" + nameFormula);
						newName.setFunction(name.isFunctionName());
						newName.setComment(name.getComment());
					}
				}
			}
		}

		// ワークブック再計算
		workbook.evaluateAll();

		return workbook;
	}

	/**
	 * バインド処理を行ったワークブックオブジェクトを出力します。<br>
	 * @param afterProcess 動的出力後のワークブック処理オブジェクト
	 */
	public PoiWorkbook export(PoiDynaAfterProcess afterProcess) {
		PoiWorkbook workbook = processDynamicBind();

		if (baseSheetDeleteOnExport) {
			// 動的出力したシート以外を全て削除
			for (int i = 0; i <= template.getPoiWorkbook().getNumberOfSheets() - 1; i++) {
				// POI3.8にてシート削除を行うと名前定義のインデックスが破損する為、非表示にて暫定対応
				//workbook.getPoiWorkbook().removeSheetAt(0);
				workbook.getSheet(i).hide();
			}
		} else {
			// 変換したシートのみを削除(ディフォルト動作仕様)
			List<Integer> templateIndex = new LinkedList<Integer>();
			for (PoiDynaExportContainer container : containers) {
				int index = container.getTemplateSheetIndex();
				if (!templateIndex.contains(index)) {
					templateIndex.add(index);
				}
			}
			int offset = 0;
			for (int index : templateIndex) {
				// POI3.8にてシート削除を行うと名前定義のインデックスが破損する為、非表示にて暫定対応
				//workbook.getPoiWorkbook().removeSheetAt(index - offset);
				//offset++;
				workbook.getSheet(index).hide();
			}
		}

		if (afterProcess != null) {
			afterProcess.afterProcess(workbook);
		}

		// 先頭シート選択
		//workbook.getSheet(0).select();
		workbook.setSelectSheetAtFirst();

		return workbook;
	}

	/**
	 * バインド処理を行ったワークブックオブジェクトを出力します。<br>
	 */
	public PoiWorkbook export() {
		return export((PoiDynaAfterProcess) null);
	}

	/**
	 * 指定されたストリームオブジェクトに対してワークブックオブジェクトを出力します。<br>
	 * @param stream 出力対象ストリームオブジェクト
	 * @param afterProcess 動的出力後のワークブック処理オブジェクト
	 * @throws IOException 正常に出力処理が行えなかった場合にスローされます
	 */
	public void export(OutputStream stream, PoiDynaAfterProcess afterProcess) throws IOException {
		PoiWorkbook workbook = export(afterProcess);
		workbook.save(stream);
	}

	/**
	 * 指定されたストリームオブジェクトに対してワークブックオブジェクトを出力します。<br>
	 * @param stream 出力対象ストリームオブジェクト
	 * @throws IOException 正常に出力処理が行えなかった場合にスローされます
	 */
	public void export(OutputStream stream) throws IOException {
		export(stream, null);
	}

	/**
	 * 指定されたファイルパスにワークブックオブジェクトを保存します。<br>
	 * @param path ファイルパス
	 * @param afterProcess 動的出力後のワークブック処理オブジェクト
	 * @throws IOException 正常に保存が行えなかった場合にスローされます
	*/
	public void export(String path, PoiDynaAfterProcess afterProcess) throws IOException {
		FileOutputStream stream = null;
		try {
			stream = new FileOutputStream(new File(path));
			export(stream, afterProcess);
		} finally {
			if (stream != null) {
				stream.close();
			}
		}
	}

	/**
	 * 指定されたファイルパスにワークブックオブジェクトを保存します。<br>
	 * @param path ファイルパス
	 * @throws IOException 正常に保存が行えなかった場合にスローされます
	*/
	public void export(String path) throws IOException {
		export(path, null);
	}
}
