package org.phosphoresce.lib.poi.adapter;

import java.io.Serializable;

import org.phosphoresce.lib.poi.PoiGlobal;

/**
 * 範囲クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日       更新者          更新内容
 * 2005/11/21	Kitagawa		新規作成
 * 2012/07/09	Kitagawa		バージョン2にアップデート
 *-->
 */
public class PoiRange implements PoiGlobal, Serializable {

	/** 行開始位置(0～) */
	private int rowIndexFrom;

	/** 列開始位置(0～) */
	private int colIndexFrom;

	/** 行終了位置(0～) */
	private int rowIndexTo;

	/** 列終了位置(0～) */
	private int colIndexTo;

	/**
	 * コンストラクタ<br>
	 * 再左上端のセルを選択範囲としてクラスを初期化します。<br>
	 */
	public PoiRange() {
		this.rowIndexFrom = 0;
		this.colIndexFrom = 0;
		this.rowIndexTo = 0;
		this.colIndexTo = 0;
	}

	/**
	 * コンストラクタ<br>
	 * 指定されたPoiRangeオブジェクトと範囲のクラスとして初期化します。<br>
	 * @param range PoiRangeオブジェクト
	 */
	public PoiRange(PoiRange range) {
		this.rowIndexFrom = range.rowIndexFrom;
		this.colIndexFrom = range.colIndexFrom;
		this.rowIndexTo = range.rowIndexTo;
		this.colIndexTo = range.colIndexTo;
		validate();
	}

	/**
	 * コンストラクタ<br>
	 * 指定された範囲のクラスとして初期化します。<br>
	 * @param rowIndexFrom 行位置(開始)
	 * @param colIndexFrom 列位置(開始)
	 * @param rowIndexTo 行位置(終了)
	 * @param colIndexTo 列位置(終了)
	 */
	public PoiRange(int rowIndexFrom, int colIndexFrom, int rowIndexTo, int colIndexTo) {
		this.rowIndexFrom = rowIndexFrom;
		this.colIndexFrom = colIndexFrom;
		this.rowIndexTo = rowIndexTo;
		this.colIndexTo = colIndexTo;
		validate();
	}

	/**
	 * 範囲位置情報の整合性を取り、クラスオブジェクトを有効な形にします。<br>
	 * 行及び列の開始位置が終了位置より大きい場合には必ず終了位置よりも開始位置の
	 * 値が小さくなるように補正します。<br>
	 */
	private void validate() {
		if (rowIndexTo < rowIndexFrom) {
			int buffer = rowIndexTo;
			rowIndexTo = rowIndexFrom;
			rowIndexFrom = buffer;
		}
		if (colIndexTo < colIndexFrom) {
			int buffer = colIndexTo;
			colIndexTo = colIndexFrom;
			colIndexFrom = buffer;
		}
	}

	/**
	 * 列開始位置(0～)を取得します。<br>
	 * @return 列開始位置(0～)
	 */
	public int getColIndexFrom() {
		return colIndexFrom;
	}

	/**
	 * 列開始位置(0～)を設定します。<br>
	 * 設定したタイミングで整合性を取る為の処理が行われ、終了位置よりも開始位置値が小さくなるように補正されます。<br>
	 * @param colIndexFrom 列開始位置(0～)
	 */
	public void setColIndexFrom(int colIndexFrom) {
		this.colIndexFrom = colIndexFrom;
		validate();
	}

	/**
	 * 列終了位置(0～)を取得します。<br>
	 * @return 列終了位置(0～)
	 */
	public int getColIndexTo() {
		return colIndexTo;
	}

	/**
	 * 列終了位置(0～)を設定します。<br>
	 * 設定したタイミングで整合性を取る為の処理が行われ終了位置よりも開始位置値が小さくなるように補正されます。<br>
	 * @param colIndexTo 列終了位置(0～)
	 */
	public void setColIndexTo(int colIndexTo) {
		this.colIndexTo = colIndexTo;
		validate();
	}

	/**
	 * 行開始位置(0～)を取得します。<br>
	 * @return 行開始位置(0～)
	 */
	public int getRowIndexFrom() {
		return rowIndexFrom;
	}

	/**
	 * 行開始位置(0～)を設定します。<br>
	 * 位置を設定したタイミングで整合性を取る為の処理が行われ終了位置よりも開始位置値が小さくなるように補正されます。<br>
	 * @param rowIndexFrom 行開始位置(0～)
	 */
	public void setRowIndexFrom(int rowIndexFrom) {
		this.rowIndexFrom = rowIndexFrom;
		validate();
	}

	/**
	 * 行終了位置(0～)を取得します。<br>
	 * @return 行終了位置(0～)
	 */
	public int getRowIndexTo() {
		return rowIndexTo;
	}

	/**
	 * 行終了位置(0～)を設定します。<br>
	 * 設定したタイミングで整合性を取る為の処理が行われ終了位置よりも開始位置値が小さくなるように補正されます。<br>
	 * @param rowIndexTo 行終了位置(0～)
	 */
	public void setRowIndexTo(int rowIndexTo) {
		this.rowIndexTo = rowIndexTo;
		validate();
	}

	/**
	 * 範囲情報の高さを取得します(最小値=1)。<br>
	 * @return 範囲情報の高さ
	 */
	public int getHeight() {
		return rowIndexTo - rowIndexFrom + 1;
	}

	/**
	 * 範囲情報の幅を取得します(最小値=1)。<br>
	 * @return 範囲情報の幅
	 */
	public int getWidth() {
		return colIndexTo - colIndexFrom + 1;
	}

	/**
	 * 指定された行方向、列方向にシフト範囲のシフト処理を行います。<br>
	 * 設定したタイミングで整合性を取る為の処理が行われ終了位置よりも開始位置値が小さくなるように補正されます。<br>
	 * @param rowShift 行方向シフト値
	 * @param colShift 列方向シフト値
	 */
	public void shift(int rowShift, int colShift) {
		colIndexFrom += colShift;
		colIndexTo += colShift;
		rowIndexFrom += rowShift;
		rowIndexTo += rowShift;
		validate();
	}

	/**
	 * ハッシュコードを取得します。<br>
	 * @return ハッシュコード
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + colIndexFrom;
		result = prime * result + colIndexTo;
		result = prime * result + rowIndexFrom;
		result = prime * result + rowIndexTo;
		return result;
	}

	/**
	 * オブジェクト等価比較を行います。<br>
	 * @return 等価の場合にtrueを返却
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object object) {
		if (this == object) {
			return true;
		}
		if (object == null) {
			return false;
		}
		if (getClass() != object.getClass()) {
			return false;
		}
		PoiRange other = (PoiRange) object;
		if (colIndexFrom != other.colIndexFrom) {
			return false;
		}
		if (colIndexTo != other.colIndexTo) {
			return false;
		}
		if (rowIndexFrom != other.rowIndexFrom) {
			return false;
		}
		if (rowIndexTo != other.rowIndexTo) {
			return false;
		}
		return true;
	}

	/**
	 * クラス情報を文字列で取得します。<br>
	 * @return クラス情報文字列
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("{");
		builder.append("rowIndexFrom=");
		builder.append(rowIndexFrom);
		builder.append(", ");
		builder.append("colIndexFrom=");
		builder.append(colIndexFrom);
		builder.append(", ");
		builder.append("rowIndexTo=");
		builder.append(rowIndexTo);
		builder.append(", ");
		builder.append("colIndexTo=");
		builder.append(colIndexTo);
		builder.append(", ");
		builder.append("height=");
		builder.append(getHeight());
		builder.append(", ");
		builder.append("width=");
		builder.append(getWidth());
		builder.append("}");
		return builder.toString();
	}
}
