package org.phosphoresce.lib.poi.adapter;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.util.SheetUtil;
import org.phosphoresce.lib.poi.util.PoiStringUtil;

/**
 * セルクラス<br>
 * <br>
 * POIオブジェクトインタフェースの拡張及び、関連インスタンスへのアクセッサを提供します。<br>
 * また、Excel上でのオペレーションに類似したインタフェースを併せて提供します。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2005/11/21	Kitagawa		新規作成
 * 2012/07/09	Kitagawa		バージョン2にアップデート
 * 2012/11/20	Kitagawa		セル高さを自動調整する為に各種サイズアクセッサを追加
 *-->
 */
public class PoiCell extends PoiBaseCell {

	/**
	 * コンストラクタ<br>
	 * @param workbook ワークブックオブジェクト
	 * @param sheet シートオブジェクト
	 * @param row 行オブジェクト
	 * @param poiCell セルオブジェクト
	 */
	PoiCell(PoiWorkbook workbook, PoiSheet sheet, PoiRow row, Cell poiCell) {
		super(workbook, sheet, row, poiCell);
	}

	/**
	 * 現在のセル幅を取得します。<br>
	 * @return 現在のセル幅
	 */
	public int getWidth() {
		return getSheet().getPoiSheet().getColumnWidth(getPoiCell().getColumnIndex());
	}

	/**
	 * 現在のセル高さを取得します。<br>
	 * @return 現在のセル高さ
	 */
	public int getHeight() {
		return getRow().getPoiRow().getHeight();
	}

	/**
	 * セル値が表示可能なセル幅を取得します。<br>
	 * @return セル値が表示可能なセル幅
	 */
	public int getFixedWidth() {
		double width = SheetUtil.getColumnWidth(getSheet().getPoiSheet(), getPoiCell().getColumnIndex(), false, getRow().getPoiRow().getRowNum(), getRow().getPoiRow().getRowNum()) * 256D;
		return (int) width;
	}

	/**
	 * セル値が表示可能なセル高さを取得します。<br>
	 * @return セル値が表示可能なセル高さ
	 */
	public int getFixedHeight() {
		Object value = getValue();
		double width = getWidth();
		double fixedWidth = getFixedWidth();
		double fixedHeight = 0;

		// セル値が空の場合は処理をスキップ
		if (value == null || PoiStringUtil.isEmpty(value.toString())) {
			return getHeight();
		}

		// 改行が存在する場合は行毎の高さを加算
		if (value instanceof String) {
			String values[] = ((String) value).split("\r\n");
			for (String temp : values) {
				// 一時的に行毎の値を設定
				setValue(temp);
				fixedHeight += (Math.ceil(fixedWidth / width)) * ((getFont().getFontHeight() / 13.5D) * 18.0D);
			}
		}

		// 元の値を設定
		setValue(value);

		return (int) fixedHeight;
	}

	/**
	 * 現在のセルの内容が表示されるように行高さを調整します。<br>
	 * セル幅を基底として調整が行われます。<br>
	 * また、当セルに閉じた調整となる為、同一行内のセルの高さが考慮されないことに注意してください。<br>
	 * @param forceResize 現在のセル高さが十分な場合でも調整を行う場合にtrueを指定
	 */
	public void autoSizeRow(boolean forceResize) {
		if (!forceResize && getRow().getPoiRow().getHeight() > getFixedHeight()) {
			return;
		}
		if (getPoiCell().getCellStyle().getWrapText()) {
			getRow().getPoiRow().setHeight((short) getFixedHeight());
		}
	}

	/**
	 * 現在のセルの内容が表示されるように行高さを調整します。<br>
	 * セル幅を基底として調整が行われます。<br>
	 * また、当セルに閉じた調整となる為、同一行内のセルの高さが考慮されないことに注意してください。<br>
	 */
	public void autoSizeRow() {
		autoSizeRow(true);
	}

	/**
	 * セルに設定されているフォントを取得します。<br>
	 * @return セルに設定されているフォント
	 */
	public Font getFont() {
		return getWorkbook().getPoiWorkbook().getFontAt(getPoiCell().getCellStyle().getFontIndex());
	}

	/**
	 * セルのコピーを行います。<br>
	 * @param sheetIndex コピー先シートインデックス(0～)
	 * @param rowIndex コピー先行位置(0～)
	 * @param colIndex コピー先列位置(0～)
	 */
	public void copy(int sheetIndex, int rowIndex, int colIndex) {
		// コピー先オブジェクト取得
		PoiCell dest = workbook.getSheet(sheetIndex).getCell(rowIndex, colIndex);

		// 基本設定コピー
		dest.getPoiCell().setCellType(poiCell.getCellType());
		if (poiCell.getCellStyle() != null) {
			dest.getPoiCell().setCellStyle(poiCell.getCellStyle());
		}
		if (poiCell.getCellComment() != null) {
			dest.getPoiCell().setCellComment(poiCell.getCellComment());
		}
		if (poiCell.getHyperlink() != null) {
			dest.getPoiCell().setHyperlink(poiCell.getHyperlink());
		}

		// コピー処理
		if (Cell.CELL_TYPE_BLANK == poiCell.getCellType()) {
			// CELL_TYPE_BLANKセルタイプのみの設定とする
		} else if (Cell.CELL_TYPE_BOOLEAN == poiCell.getCellType()) {
			dest.getPoiCell().setCellValue(poiCell.getBooleanCellValue());
		} else if (Cell.CELL_TYPE_ERROR == poiCell.getCellType()) {
			dest.getPoiCell().setCellErrorValue(poiCell.getErrorCellValue());
		} else if (Cell.CELL_TYPE_FORMULA == poiCell.getCellType()) {
			dest.getPoiCell().setCellFormula(poiCell.getCellFormula());
		} else if (Cell.CELL_TYPE_NUMERIC == poiCell.getCellType()) {
			dest.getPoiCell().setCellValue(poiCell.getNumericCellValue());
		} else if (Cell.CELL_TYPE_STRING == poiCell.getCellType()) {
			dest.getPoiCell().setCellValue(poiCell.getStringCellValue());
		}
	}

	/**
	 * セルのコピーを行います。<br>
	 * @param sheetName コピー先シート名
	 * @param rowIndex コピー先行位置(0～)
	 * @param colIndex コピー先列位置(0～)
	 */
	public void copy(String sheetName, int rowIndex, int colIndex) {
		copy(workbook.getSheet(sheetName).getSheetIndex(), rowIndex, colIndex);
	}

	/**
	 * セルのコピーを行います。<br>
	 * @param rowIndex コピー先行位置(0～)
	 * @param colIndex コピー先列位置(0～)
	 */
	public void copy(int rowIndex, int colIndex) {
		copy(sheet.getSheetIndex(), rowIndex, colIndex);
	}
}
