package org.phosphoresce.lib.poi.adapter;

import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.phosphoresce.lib.poi.PoiGlobal;

/**
 * シート抽象インタフェースクラス<br>
 * <br>
 * 基本的なオブジェクト操作を提供する内部クラスです。<br>
 * 共通的な処理分割のみを目的としたクラスでユーザから直接インスタンス化されて利用されることはありません。<br>
 * 単純な共通処理部の分割クラスであり依存関係が分割された継承クラス関係ではありません。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2005/11/21	Kitagawa		新規作成
 * 2005/12/07	Kitagawa		PoiPrintConfigオブジェクトによる印刷設定インタフェース追加
 * 2012/07/09	Kitagawa		バージョン2にアップデート
 *-->
 */
abstract class PoiBaseSheet implements PoiGlobal {

	/** ワークブックオブジェクト */
	protected PoiWorkbook workbook;

	/** シートオブジェクト */
	protected Sheet poiSheet;

	/**
	 * コンストラクタ<br>
	 * @param workbook ワークブックオブジェクト
	 * @param poiSheet シートオブジェクト
	 */
	PoiBaseSheet(PoiWorkbook workbook, Sheet poiSheet) {
		super();
		this.workbook = workbook;
		this.poiSheet = poiSheet;
	}

	/**
	 * シートオブジェクトを取得します。<br>
	 * @return シートオブジェクト
	 */
	public Sheet getPoiSheet() {
		return poiSheet;
	}

	/**
	 * ワークブックオブジェクトを取得します。<br>
	 * @return ワークブックオブジェクト
	 */
	public PoiWorkbook getWorkbook() {
		return workbook;
	}

	/**
	 * 行オブジェクトを取得します。<br>
	 * @param rowIndex 行位置(0～)
	 * @return 行オブジェクト
	 */
	public PoiRow getRow(int rowIndex) {
		if (poiSheet.getLastRowNum() < rowIndex || poiSheet.getRow(rowIndex) == null) {
			poiSheet.createRow(rowIndex);
		}
		return new PoiRow((PoiWorkbook) workbook, (PoiSheet) this, poiSheet.getRow(rowIndex));
	}

	/**
	 * セルオブジェクトを取得します。<br>
	 * @param rowIndex 行位置(0～)
	 * @param colIndex 列位置(0～)
	 * @return セルオブジェクト
	 */
	public PoiCell getCell(int rowIndex, int colIndex) {
		return getRow(rowIndex).getCell(colIndex);
	}

	/**
	 * シート上の最大行位置を取得します。<br>
	 * @return シート上の最大行位置(0～)
	 */
	public int getLastRowNum() {
		return poiSheet.getLastRowNum();
	}

	/**
	 * シート上の最大列位置を取得します。<br>
	 * @return シート上の最大列位置(0～)
	 */
	public int getLastCellNum() {
		int col = 0;
		for (int i = 0; i <= getLastRowNum(); i++) {
			Row row = poiSheet.getRow(i);
			if (row != null) {
				if (col < row.getLastCellNum()) {
					col = row.getLastCellNum();
				}
			}
		}
		return col;
	}

	/**
	 * シートインデックスを取得します。<br>
	 * @return シートインデックス
	 */
	public int getSheetIndex() {
		return workbook.getPoiWorkbook().getSheetIndex(poiSheet);
	}

	/**
	 * クラス情報を文字列で提供します。<br>
	 * @return クラス情報文字列
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[");
		for (int i = poiSheet.getFirstRowNum(); i <= poiSheet.getLastRowNum(); i++) {
			builder.append(getRow(i).toString());
			if (i <= poiSheet.getLastRowNum() - 1) {
				builder.append(", ");
			}
		}
		builder.append("]");
		return builder.toString();
	}
}
