package org.phosphoresce.lib.jod;

import static org.phosphoresce.lib.jod.JODSOfficeProcessStatus.*;

import java.net.InetSocketAddress;
import java.net.Socket;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.phosphoresce.lib.jod.exception.JODConvertProcessException;
import org.phosphoresce.lib.jod.exception.JODProcessTooBusyException;

/**
 * LibreOfficeプロセス管理クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/12/19	Kitagawa		新規作成
 *-->
 */
public class JODSOfficeProcessManager {

	/** セルフインスタンス */
	private static JODSOfficeProcessManager instance;

	/** 環境設定オブジェクト */
	private JODSetting setting;

	/** プロセスインスタンスマップ */
	private Map<Integer, JODSOfficeProcess> processes;

	/** プロセス破棄処理スレッドオブジェクト */
	private JODSOfficeProcessDestroyThread destroyThread;

	/** インスタンス破棄フラグ */
	private boolean instanceDestroy;

	/**
	 * コンストラクタ<br>
	 * @param setting 環境設定オブジェクト
	 */
	private JODSOfficeProcessManager(JODSetting setting) {
		super();
		this.setting = setting;
		this.processes = new HashMap<Integer, JODSOfficeProcess>();
		this.destroyThread = new JODSOfficeProcessDestroyThread();
		this.instanceDestroy = false;
	}

	/**
	 * LibreOfficeプロセス破棄処理スレッドクラス<br>
	 * 
	 * @author Kitagawa<br>
	 * 
	 *<!--
	 * 更新日		更新者			更新内容
	 * 2012/12/19	Kitagawa		新規作成
	 *-->
	 */
	class JODSOfficeProcessDestroyThread extends Thread {

		/**
		 * コンストラクタ<br>
		 */
		public JODSOfficeProcessDestroyThread() {
			super(new Runnable() {
				public void run() {
					/*
					 * インスタンス破棄フラグが設定されるまでは無限ループ
					 */
					while (!instanceDestroy) {
						List<Integer> targets = new LinkedList<Integer>();
						try {
							synchronized (processes) {
								for (int port : processes.keySet()) {
									if (processes.get(port) == null) {
										targets.add(port);
										continue;
									}
									if (processes.get(port).getStatus() == WAIT_DESTROY) {
										targets.add(port);
										continue;
									}
								}
								for (int port : targets) {
									JODSOfficeProcess process = processes.remove(port);
									process.finalize();
									process = null;
								}
							}
						} catch (Throwable e) {
							// 例外が発生した場合もメモリリークを防止する為に監視処理を続行させる
							e.printStackTrace();
						}
						try {
							// 一定間隔で破棄対象を監視
							Thread.sleep(500);
						} catch (InterruptedException e) {
							// ウェイト処理における例外は無視
							e.printStackTrace();
						}
					}
				}
			});
		}
	}

	/**
	 * クラスを初期化します。<br>
	 * @param setting 環境設定オブジェクト
	 */
	public synchronized static void initialize(JODSetting setting) {
		if (instance != null) {
			throw new JODConvertProcessException("Instance is already initialized");
		}
		instance = new JODSOfficeProcessManager(setting);
		instance.destroyThread.start();
	}

	/**
	 * クラスインスタンスを取得します。<br>
	 * @return クラスインスタンス
	 */
	public synchronized static JODSOfficeProcessManager instance() {
		if (instance == null) {
			throw new JODConvertProcessException("Instance has not initialized");
		}
		return instance;
	}

	/**
	 * クラスインスタンスを破棄します。<br>
	 * クラスインスタンスへの破棄マークが行われた場合、当インスタンスはそれ以降利用不可となります。<br>
	 */
	public synchronized static void destroy() {
		if (instance != null) {
			instance.instanceDestroy = true;

			/*
			 * インスタンス破棄フラグが設定されたタイミングで保持されている全てのプロセスを破棄する
			 */
			synchronized (instance.processes) {
				for (int port : instance.processes.keySet()) {
					if (instance.processes.get(port) == null) {
						continue;
					}
					instance.processes.get(port).destroy();
				}
				instance.processes.clear();
			}
			instance = null;
		}
	}

	/**
	 * 新しいプロセスを生成します。<br>
	 * @return プロセスオブジェクト
	 */
	public synchronized JODSOfficeProcess createProcess() {
		if (instanceDestroy) {
			throw new JODConvertProcessException("Instance is already destroyed");
		}

		for (int i = setting.getSofficeReservePortStart(); i <= setting.getSofficeReservePortEnd(); i++) {
			JODSOfficeProcess process = createProcess(i);
			if (process != null) {
				return process;
			}
		}
		throw new JODProcessTooBusyException();
	}

	/**
	 * 新しいプロセスを生成します。<br>
	 * @param port 実行ポート
	 * @return プロセスオブジェクト
	 */
	public synchronized JODSOfficeProcess createProcess(int port) {
		if (instanceDestroy) {
			throw new JODConvertProcessException("Instance is already destroyed");
		}

		synchronized (processes) {
			if (processes.containsKey(port)) {
				if (processes.get(port).getStatus() == WAIT_PROCESS || processes.get(port).getStatus() == WAIT_CONNECT) {
					JODSOfficeProcess process = processes.get(port);
					process.open();
					return process;
				} else {
					return null;
				}
			}
			//if (!isValidatePort(i)) {
			//	return null;
			//}
			JODSOfficeProcess process = new JODSOfficeProcess(setting, port);
			processes.put(process.getPort(), process);
			return process;
		}
	}

	/**
	 * クラスインスタンスを破棄します。<br>
	 * @see java.lang.Object#finalize()
	 */
	@Override
	protected void finalize() throws Throwable {
		super.finalize();
		destroy();
	}

	/**
	 * 指定ポートが利用可能な状態であるか判定します。<br>
	 * @param port ポート番号
	 * @return 利用可能な場合にtrueを返却
	 */
	private boolean isValidatePort(int port) {
		Socket socket = null;
		try {
			socket = new Socket();
			socket.connect(new InetSocketAddress(setting.getSofficeHost(), port));
			return false;
		} catch (Throwable e) {
			return true;
		} finally {
			try {
				if (socket != null) {
					socket.close();
				}
			} catch (Throwable e) {
				// このタイミングでの例外は無視
			}
		}
	}
}
