package org.phosphoresce.lib.jod;

import static org.phosphoresce.lib.jod.JODSOfficeProcessStatus.*;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.phosphoresce.lib.commons.util.FileUtil;
import org.phosphoresce.lib.commons.util.PropertiesUtil;
import org.phosphoresce.lib.commons.util.StringUtil;
import org.phosphoresce.lib.jod.exception.JODConvertProcessException;

import com.artofsolving.jodconverter.openoffice.connection.OpenOfficeConnection;
import com.artofsolving.jodconverter.openoffice.connection.SocketOpenOfficeConnection;

/**
 * LibreOfficeソケットサービスプロセスクラス<br>
 * <br>
 * LibreOfficeソケットサービスのプロセス起動から接続確立までを担当するクラスです。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/12/19	Kitagawa		新規作成
 *-->
 */
public class JODSOfficeProcess implements JODConverterConstants {

	/** プロパティオブジェクト */
	private static Properties properties;

	/** 環境設定オブジェクト */
	private JODSetting setting;

	/** プロセスステータス */
	private JODSOfficeProcessStatus status;

	/** LibreOffice実行ポート */
	private int sofficePort;

	/** プロセスオブジェクト */
	private Process process;

	/** ソケットコネクションオブジェクト */
	private OpenOfficeConnection connection;

	/**
	 * コンストラクタ<br>
	 * @param setting 環境設定オブジェクト
	 * @param sofficePort ソケット実行ポート
	 */
	public JODSOfficeProcess(JODSetting setting, int sofficePort) {
		super();

		if (properties == null) {
			try {
				properties = PropertiesUtil.load("jodconverter.properties");
			} catch (IOException e) {
				throw new JODConvertProcessException("Failed to load default properties.");
			}
		}

		this.status = WAIT_CONNECT;
		this.setting = setting;
		this.sofficePort = sofficePort;
		this.process = null;
		this.connection = null;
	}

	/**
	 * ソケットサービスをオープンします。<br>
	 * ソケットへの接続試行を行い、接続が確立出来なかった場合はプロセスの起動を試みます。<br>
	 */
	public void open() {
		synchronized (status) {
			/*
			 * 既に処理開始待ち状態の場合はオープン処理をスキップ
			 */
			if (status == WAIT_PROCESS) {
				return;
			}

			/*
			 * 接続待ち状態ではない場合はエラー
			 */
			if (status != WAIT_CONNECT) {
				throw new JODConvertProcessException("Illegal status for open [" + status + "]");
			}

			try {
				// ステータス変更
				status = CONNECTING;

				if (test()) {
					/*
					 * 接続試行が成功した場合は接続
					 */
					// ソケットサービス接続
					connect();

					// ステータス変更
					status = WAIT_PROCESS;
				} else {
					/*
					 * 接続試行が行えなかった場合は新規プロセスを起動
					 */
					// 実行コマンド生成
					List<String> command = new LinkedList<String>();
					command.add(setting.getSofficePath());
					for (int i = 1; i <= EXECUTE_OPTIONS_DEFINITION_COUNT; i++) {
						Map<String, String> binds = new HashMap<String, String>();
						binds.put(EXECUTE_OPTIONS_BIND_KEY_HOST, setting.getSofficeHost());
						binds.put(EXECUTE_OPTIONS_BIND_KEY_PORT, String.valueOf(sofficePort));
						binds.put(EXECUTE_OPTIONS_BIND_KEY_USERENV, FileUtil.getSimpleURI(new File(setting.getTemporaryPath() + "_" + setting.getSofficeHost() + "_" + sofficePort)));

						String propKey = "";
						if (setting.getOsType() == JODOSType.WINDOWS) {
							propKey = CONFIG_KEY_WINDOWS_SOFFICE_OPTION + StringUtil.padding(String.valueOf(i), 2, '0', true);
						} else if (setting.getOsType() == JODOSType.LINUX) {
							propKey = CONFIG_KEY_LINUX_SOFFICE_OPTION + StringUtil.padding(String.valueOf(i), 2, '0', true);
						}
						String option = properties.getProperty(propKey);

						if (!StringUtil.isEmpty(option)) {
							option = StringUtil.bind(option, binds);
							command.add(option);
						}
					}

					// プロセス起動
					process = Runtime.getRuntime().exec(command.toArray(new String[0]));

					// ソケットサービス接続
					connect();

					// ステータス変更
					status = WAIT_PROCESS;
				}
			} catch (Throwable e) {
				try {
					destroy();
				} catch (Throwable e2) {
					// クローズ例外時の強制切断処理時の例外は無視
				}
				throw new JODConvertProcessException("Failed to startup soffice process", e);
			}
		}
	}

	/**
	 * 当インスタンスが提供するソケットサービスへの処理を排他します。<br>
	 * ステータスの設定を処理中状態とすることで、当インスタンスによるプロセスへの処理接続排他が行われます。<br>
	 * 明示的に排他が行われなかった場合、当インスタンスは常に複数スレッドから利用されることになります。<br>
	 * 尚、排他解除はソケット接続クローズ処理が行われた場合、又は、明示的に排他解除処理が行われた場合となります。<br>
	 */
	public void lock() {
		synchronized (status) {
			/*
			 * 接続待ちの場合は接続
			 */
			if (status == WAIT_CONNECT) {
				open();
			}

			/*
			 * 処理待ち状態ではない場合はエラー
			 */
			if (status != WAIT_PROCESS) {
				throw new JODConvertProcessException("Illegal status for lock [" + status + "]");
			}

			// ステータス設定
			status = PROCESSING;
		}
	}

	/**
	 * 当インスタンスが提供するソケットサービスへの排他状態を解除します。<br>
	 * ソケットサービスへのクローズ処理が呼ばれずに再利用待ちとなる為、利用側で接続維持状態で継続処理を行いたい場合にのみ利用します。<br>
	 * 通常は処理が完了したタイミングでクローズ処理又は、破棄処理を実行します。<br>
	 */
	public void unlock() {
		synchronized (status) {
			/*
			 * 既に処理待ち状態でロックされていない場合は処理スキップ
			 */
			if (status == WAIT_CONNECT || status == CONNECTING || status == WAIT_PROCESS || status == DISCONNECTING || status == FINALIZING || status == WAIT_DESTROY) {
				return;
			}

			// ステータス設定
			status = WAIT_PROCESS;
		}
	}

	/**
	 * ソケットサービスへの接続をクローズします。<br>
	 * ソケットプロセス自体の破棄は行なわれません。<br>
	 */
	public void close() {
		synchronized (status) {
			try {
				// ステータス変更
				status = DISCONNECTING;

				// ソケットサービス切断
				disconnect();

				// ステータス変更
				status = WAIT_CONNECT;
			} catch (Throwable e) {
				throw new JODConvertProcessException("Failed to disconnect soffice process", e);
			}
		}
	}

	/**
	 * ソケットプロセスを破棄します。<br>
	 * 当インスタンスから動的にプロセスが起動されていない場合は利用されたプロセスが破棄されるわけではありません。<br>
	 */
	public void destroy() {
		synchronized (status) {
			try {
				// ステータス変更
				status = FINALIZING;

				// ソケットサービス切断
				disconnect();

				// ソケットプロセス破棄
				if (process != null) {
					process.destroy();
					process = null;
				}

				// ステータス変更
				status = WAIT_DESTROY;
			} catch (Throwable e) {
				throw new JODConvertProcessException("Failed to destroy soffice process", e);
			}
		}
	}

	/**
	 * クラスインスタンスを破棄します。<br>
	 * @see java.lang.Object#finalize()
	 */
	@Override
	protected void finalize() throws Throwable {
		synchronized (status) {
			super.finalize();
			destroy();
		}
	}

	/**
	 * ソケットサービスへの接続試行を行います。
	 * @return ソケットサービスへの接続が行えた場合にtrueを返却
	 */
	private synchronized boolean test() {
		OpenOfficeConnection connection = null;
		try {
			connection = new SocketOpenOfficeConnection(setting.getSofficeHost(), sofficePort);
			connection.connect();
			return true;
		} catch (Throwable e) {
			return false;
		} finally {
			if (connection != null && connection.isConnected()) {
				connection.disconnect();
			}
		}
	}

	/**
	 * ソケットサービスプロセスへの接続を行います。<br>
	 * 接続が正常に行われなかった場合、定義された接続試行回数分の再接続処理が実施されます。<br>
	 * 接続試行回数を超えて接続が確立できなかった場合は例外がスローされます。<br>
	 * @throws 接続が正常に行われなかった場合にスローされます
	 */
	private synchronized void connect() throws Throwable {
		try {
			/*
			 * ソケット接続処理
			 */
			OpenOfficeConnection connection = new SocketOpenOfficeConnection(setting.getSofficeHost(), sofficePort);
			connection.connect();
			this.connection = connection;
		} catch (Throwable e) {
			/*
			 * 接続試行回数分ソケット接続処理
			 */
			try {
				for (int i = 1; i <= setting.getConnectTryCount() - 1; i++) {
					// 試行間隔ウェイト
					Thread.sleep(setting.getConnectTryDelay());

					try {
						// ソケットサーバーソケット接続試験実施
						OpenOfficeConnection connection = new SocketOpenOfficeConnection(setting.getSofficeHost(), sofficePort);
						connection.connect();
						this.connection = connection;
						break;
					} catch (Throwable ei) {
						if (i < setting.getConnectTryCount() - 1) {
							// 試行中の例外は無視
						} else {
							throw ei;
						}
					}
				}
			} catch (Throwable e2) {
				throw e2;
			}
		}
	}

	/**
	 * ソケットサービスプロセスへの切断を行います。<br>
	 * @throws ソケットサービスプロセスへの接続が正常に切断できなかった場合にスローされます
	 */
	private synchronized void disconnect() throws Throwable {
		try {
			if (connection == null) {
				return;
			}
			if (!connection.isConnected()) {
				return;
			}
		} catch (Throwable e) {
			throw e;
		} finally {
			if (connection != null) {
				connection.disconnect();
				connection = null;
			}
		}
	}

	/**
	 * プロセスステータスを取得します。<br>
	 * @return プロセスステータス
	 */
	public JODSOfficeProcessStatus getStatus() {
		return status;
	}

	/**
	 * LibreOffice実行ポートを取得します。<br>
	 * @return LibreOffice実行ポート
	 */
	public int getPort() {
		return sofficePort;
	}

	/**
	 * ソケットコネクションオブジェクトを取得します。<br>
	 * @return ソケットコネクションオブジェクト
	 */
	OpenOfficeConnection getConnection() {
		return connection;
	}
}
