package org.phosphoresce.lib.jod;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;

import org.phosphoresce.lib.jod.exception.JODConvertProcessException;

import com.artofsolving.jodconverter.DefaultDocumentFormatRegistry;
import com.artofsolving.jodconverter.DocumentConverter;
import com.artofsolving.jodconverter.DocumentFormat;
import com.artofsolving.jodconverter.DocumentFormatRegistry;
import com.artofsolving.jodconverter.openoffice.converter.OpenOfficeDocumentConverter;

/**
 * LibreOfficeドキュメントコンバートプロセッサクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/08/02	Kitagawa		新規作成
 *-->
 */
public class JODDocumentConverter {

	/** ドキュメントフォーマットレジストリオブジェクト */
	public static final DocumentFormatRegistry DEFAULT_FORMAT_REGISTORY = new DefaultDocumentFormatRegistry();

	/** ドキュメントフォーマット(Excel2003) */
	public static final DocumentFormat FORMAT_EXCEL2003 = createDocumentFormat("xls");

	/** ドキュメントフォーマット(Excel2007) */
	public static final DocumentFormat FORMAT_EXCEL2007_XLSM = createDocumentFormat("xlsm");

	/** ドキュメントフォーマット(Excel2007) */
	public static final DocumentFormat FORMAT_EXCEL2007_XLSX = createDocumentFormat("xlsx");

	/** ドキュメントフォーマット(OpenOffice.org Calc) */
	public static final DocumentFormat FORMAT_OPENOFFICE_CALC = createDocumentFormat("ods");

	/** ドキュメントフォーマット(PDF) */
	public static final DocumentFormat FORMAT_PDF = createDocumentFormat("pdf");

	/** 実行環境設定オブジェクト */
	private JODSetting setting;

	/** LibreOfficeプロセスオブジェクト */
	private JODSOfficeProcess process;

	/** 入力ドキュメントフォーマット */
	private DocumentFormat inputFormat;

	/** 出力ドキュメントフォーマット */
	private DocumentFormat outputFormat;

	/**
	 * コンストラクタ<br>
	 * @param inputFormat 入力ドキュメントフォーマット
	 * @param outputFormat 出力ドキュメントフォーマット
	 */
	public JODDocumentConverter(JODSetting setting, DocumentFormat inputFormat, DocumentFormat outputFormat) {
		super();
		this.setting = setting;
		this.inputFormat = inputFormat;
		this.outputFormat = outputFormat;
		this.process = JODSOfficeProcessManager.instance().createProcess();
	}

	/**
	 * コンストラクタ<br>
	 * @param inputFormatExtension 入力ドキュメントフォーマット識別拡張子
	 * @param outputFormatExtension 出力ドキュメントフォーマット識別拡張子
	 */
	public JODDocumentConverter(JODSetting setting, String inputFormatExtension, String outputFormatExtension) {
		super();
		this.setting = setting;
		this.inputFormat = createDocumentFormat(inputFormatExtension);
		this.outputFormat = createDocumentFormat(outputFormatExtension);
		this.process = JODSOfficeProcessManager.instance().createProcess();
	}

	/**
	 * 指定された拡張子のドキュメントフォーマットオブジェクトを生成します。<br>
	 * @param extension ドキュメント拡張子
	 * @return ドキュメントフォーマットオブジェクト
	 */
	private static DocumentFormat createDocumentFormat(String extension) {
		return DEFAULT_FORMAT_REGISTORY.getFormatByFileExtension(extension);
	}

	/**
	 * 接続をオープンします。<br>
	 */
	public void open() {
		process.lock();
	}

	/**
	 * 接続をクローズします。<br>
	 */
	public void close() {
		if (setting.isKeepConnect()) {
			process.close();
		} else {
			process.destroy();
		}
	}

	/**
	 * 指定された入力ファイルから出力ファイルにドキュメント変換を行います。<br>
	 * @param inputFilename 入力ファイル名
	 * @param outputFilename 出力ファイル名
	 */
	public void convert(String inputFilename, String outputFilename) {
		try {
			DocumentConverter converter = new OpenOfficeDocumentConverter(process.getConnection(), new DefaultDocumentFormatRegistry());
			converter.convert(new File(inputFilename), inputFormat, new File(outputFilename), outputFormat);
		} catch (Throwable e) {
			throw new JODConvertProcessException("Failed to convert " + inputFilename + " to " + outputFilename + ".", e);
		}
	}

	/**
	 * 指定された入力ファイルから出力ファイルにドキュメント変換を行います。<br>
	 * @param inputFile 入力ファイル名
	 * @param outputFile 出力ファイル名
	 */
	public void convert(File inputFile, File outputFile) {
		try {
			DocumentConverter converter = new OpenOfficeDocumentConverter(process.getConnection(), new DefaultDocumentFormatRegistry());
			converter.convert(inputFile, inputFormat, outputFile, outputFormat);
		} catch (Throwable e) {
			throw new JODConvertProcessException("Failed to convert " + inputFile + " to " + outputFile + ".", e);
		}
	}

	/**
	 * 指定された入力ストリームから出力ストリームにドキュメント変換を行います。<br>
	 * @param inputStream 入力ストリーム
	 * @param outputStream 出力ストリーム
	 */
	public void convert(InputStream inputStream, OutputStream outputStream) {
		try {
			DocumentConverter converter = new OpenOfficeDocumentConverter(process.getConnection(), new DefaultDocumentFormatRegistry());
			converter.convert(inputStream, inputFormat, outputStream, outputFormat);
		} catch (Throwable e) {
			throw new JODConvertProcessException("Failed to convert input stream to output stream.", e);
		}
	}

	/**
	 * 出力ストリーム内容からパイプして出力側ストリームにドキュメント変換を行います。<br>
	 * @param outputStream 入力側出力ストリーム
	 * @param outputProcess パイプ出力処理オブジェクト
	 */
	public void convert(OutputStream outputStream, final JODPipedOutputProcess outputProcess) {
		try {
			PipedInputStream pin = null;
			try {
				DocumentConverter converter = new OpenOfficeDocumentConverter(process.getConnection(), new DefaultDocumentFormatRegistry());

				final PipedOutputStream pout = new PipedOutputStream();
				pin = new PipedInputStream(pout);

				Thread thread = new Thread(new Runnable() {
					public void run() {
						try {
							outputProcess.process(pout);
						} catch (Throwable e) {
							outputProcess.setThrowable(e);
						} finally {
							if (pout != null) {
								try {
									pout.close();
								} catch (Throwable e) {
									outputProcess.setThrowable(e);
								}
							}
						}
					}
				});
				thread.start();
				converter.convert(pin, inputFormat, outputStream, outputFormat);

				if (outputProcess.hasThrowable()) {
					throw outputProcess.getThrowable();
				}
			} finally {
				if (pin != null) {
					pin.close();
				}
			}
		} catch (Throwable e) {
			throw new JODConvertProcessException("Failed to convert piping stream.", e);
		}
	}
}
