/**
 * 文字列操作ユーティリティクラス<br>
 * <br>
 * ブラウザに対する各種操作クラスです。<br>
 * ウィンドウサイズ操作等のブラウザページに対するインタフェースも提供されます。<br>
 *
 * @author Kitagawa<br>
 *
 *<!--
 * 更新日		更新者				更新内容
 * 2012/07/02	Kitagawa			新規作成
 *-->
 */

/**
 * コンストラクタ<br>
 */
var StringUtil = function() {
};

/** 空文字列 */
StringUtil.EMPTY = "";

/** 半角スペース */
StringUtil.ASCII_SPACE = " ";

/**
 * 指定された文字列がnull又は空文字列か判定します。<br>
 * @param {String} value 文字列
 * @returns {Boolean} 文字列がnull又は空文字列の場合にtrueを返却 
 */
StringUtil.isEmpty = function(value) {
	return value === undefined || value.length === 0;
};

/**
 * 指定された文字列が数値文字列(小数、負数含む)であるか判定します。<br>
 * @param {String} value 判定対象文字列
 * @returns {Boolean} 指定された文字列が数値文字列(小数、負数含む)である場合にtrueを返却
 */
StringUtil.isNumeric = function(value) {
	if (this.isEmpty(value)) {
		return false;
	}
	//if (!value.match("(^[+-]{0,1}[\.]{0,1}[0-9]+$)|(^[+-]{0,1}[0-9]+[\.]{1}[0-9]{0,}$)")) { // 非厳密チェック(+.999、-999.を許可)
	if (!value.match("(^[+-]{0,1}[0-9]+[\.]{1}[0-9]+$)|(^[+-]{0,1}[0-9]+$)")) {
		return false;
	}
	return true;
};

/**
 * 指定された文字列が整数値文字列(負数含む)であるか判定します。<br>
 * @param {String} value 判定対象文字列
 * @returns {Boolean} 指定された文字列が整数文字列(負数含む)である場合にtrueを返却
 */
StringUtil.isInteger = function(value) {
	if (this.isEmpty(value)) {
		return false;
	}
	if (!value.match("(^[+-]{0,1}[0-9]+$)")) {
		return false;
	}
	return true;
};

/**
 * 指定された文字列が数値文字列(小数、負数含まない)であるか判定します。<br>
 * @param {String} value 判定対象文字列
 * @returns {Boolean} 指定された文字列が数値文字列(小数、負数含まない)である場合にtrueを返却
 */
StringUtil.isDigit = function(value) {
	if (this.isEmpty(value)) {
		return false;
	}
	if (!value.match("(^[0-9]+$)")) {
		return false;
	}
	return true;
};

/**
 * 指定された文字列同士が同一であるか判定します。<br>
 * 文字列が両者ともnullの場合も同一として判定します。<br>
 * 当メソッドでは大文字小文字の判断を行います。<br>
 * @param {String} value1 対象文字列
 * @param {String} value2 対象文字列
 * @return {Boolean} 同一の文字列の場合にtrueを返却
 */
StringUtil.equals = function(value1, value2) {
	if (value1 === undefined && value2 == undefined) {
		return true;
	} else if (value1 === null && value2 == null) {
		return true;
	} else if (value1 !== undefined && value1 !== null) {
		return value1.equals(value2);
	} else if (value2 !== undefined && value2 !== null) {
		return value2.equals(value1);
	} else {
		return false;
	}
};

/**
 * 指定された文字列同士が同一であるか判定します。<br>
 * 文字列が両者ともnullの場合も同一として判定します。<br>
 * また、当メソッドでは大文字小文字の区別は行いません。<br>
 * @param {String} value1 対象文字列
 * @param {String} value2 対象文字列
 * @return {Boolean} 同一の文字列の場合にtrueを返却
 */
StringUtil.equalsIgnoreCase = function(value1, value2) {
	if (value1 === undefined && value2 == undefined) {
		return true;
	} else if (value1 === null && value2 == null) {
		return true;
	} else if (value1 !== undefined && value1 !== null) {
		return value1.equalsIgnoreCase(value2);
	} else if (value2 !== undefined && value2 !== null) {
		return value2.equalsIgnoreCase(value1);
	} else {
		return false;
	}
};

/**
 * 指定された文字列が空の場合にnullを返却します。<br>
 * @param {String} value 対象文字列
 * @return {String} 編集された文字列
 */
StringUtil.empty2null = function(value) {
	return this.isEmpty(value) ? null : value;
};

/**
 * 指定された文字列がnullの場合に別の文字列に置換して提供します。<br>
 * @param {String} value 対象文字列
 * @param {String} render null時置換文字列
 * @return {String} 編集された文字列
 */
StringUtil.nvl = function(value, render) {
	return value === null ? this.isEmpty(render) ? this.EMPTY : render : value;
};

/**
 * 指定された文字列がundefinedの場合に別の文字列に置換して提供します。<br>
 * @param {String} value 対象文字列
 * @param {String} render null時置換文字列
 * @return {String} 編集された文字列
 */
StringUtil.uvl = function(value, render) {
	return value === undefined ? this.isEmpty(render) ? this.EMPTY : render : value;
};

/**
 * 指定された文字列が空の場合に別の文字列に置換して提供します。<br>
 * @param {String} value 対象文字列
 * @param {String} render null時置換文字列
 * @return {String} 編集された文字列
 */
StringUtil.evl = function(value, render) {
	return StringUtil.isEmpty(value) ? this.isEmpty(render) ? this.EMPTY : render : value;
};

/**
 * 指定された数の半角スペース文字列を生成します。<br>
 * @param {Number} count 生成するスペース数
 * @return {String} 半角スペース文字列
 */
StringUtil.space = function(count) {
	var value = "";
	for ( var i = 0; i <= count - 1; i++) {
		value += this.ASCII_SPACE;
	}
	return value;
};

/**
 * 指定された文字列の文字長を取得します。<br>
 * @param {String} value 対象文字列
 * @return {Number} 文字長
 */
StringUtil.len = function(value) {
	return this.nvl(value).length;
};

/**
 * 指定された文字列のバイト長を取得します。<br>
 * @param {String} value 対象文字列
 * @param {String} charset キャラクタセット
 * @return {Number} 文字列バイト長
 */
StringUtil.lenb = function(value) {
	if (this.isEmpty(value)) {
		return 0;
	}
	var encodedValue = encodeURIComponent(value);
	var percentMatches = encodedValue.match(new RegExp("%", "g"));
	var percentCount = percentMatches ? percentMatches.length : 0;
	var length = encodedValue.length - percentCount * 2;
	return length;
};

/**
 * 指定した文字数範囲で文字列をカットして提供します。<br>
 * @param {String} value 対象文字列
 * @param {Number} position 開始位置
 * @param {Number} length 取得文字数
 * @return {String} 編集を行った文字列
 */
StringUtil.substr = function(value, position, length) {
	if (this.isEmpty(value) || position > this.len(value)) {
		return this.EMPTY;
	}
	if (position < 0) {
		position = 0;
	}
	if (length < 0) {
		length = 0;
	}
	if (this.len(value) < position + length) {
		length = this.len(value) - position;
	}
	return value.substring(position, position + length);
};

/**
 * 指定された長さ分の文字列を左側から取得します。<br>
 * @param {String} value 対象文字列
 * @param {Number} length 取得文字数
 * @returns {String} 文字列
 */
StringUtil.left = function(value, length) {
	if (this.isEmpty(value)) {
		return this.EMPTY;
	}
	return value.substring(0, length);
};

/**
 * 指定された長さ分の文字列を右側から取得します。<br>
 * @param {String} value 対象文字列
 * @param {Number} length 取得文字数
 * @returns {String} 文字列
 */
StringUtil.right = function(value, length) {
	if (this.isEmpty(value)) {
		return this.EMPTY;
	}
	return value.substring(value.length - length, value.length);
};

/**
 * 指定された長さ分の文字列を指定位置から取得します。<br>
 * @param {String} value 対象文字列
 * @param {Numebr} start 取得位置
 * @param {Numebr} length 取得文字数
 * @returns {String} 文字列
 */
StringUtil.mid = function(value, start, length) {
	if (this.isEmpty(value)) {
		return this.EMPTY;
	}
	return value.substring(start, start + length);
};

/**
 * 指定された文字列の置換を行い、その結果を取得します。<br>
 * 当メソッドは正規表現は使用しません。<br>
 * @param {String} source 処理の対象の文字列
 * @param {String} before 置換前の文字列
 * @param {String} after 置換後の文字列
 * @return {String} 置換処理後の文字列
 */
StringUtil.replace = function(source, before, after) {
	if (this.isEmpty(source)) {
		return source;
	}
	if (source.indexOf(before) < 0) {
		return source;
	}

	var result = "";
	var index = source.indexOf(before);
	result += source.substring(0, index) + after;
	if (index + before.length < source.length) {
		var rest = source.substring(index + before.length, source.length);
		result += this.replace(rest, before, after);
	}
	return result;
};

/**
 * 指定された文字列を全て逆順に並べなおして提供します。<br>
 * @param {String} source 並べ替える対象の文字列
 * @return {String} 指定された文字列を全て逆順に並べなおした文字列
 */
StringUtil.reverse = function(source) {
	if (this.isEmpty(source)) {
		return source;
	}
	var result = "";
	for ( var i = 0; i <= source.length - 1; i++) {
		result += source.charAt(source.length - 1 - i);
	}
	return result;
};

/**
 * 指定された文字列内に含まれる指定文字の数を取得します。<br>
 * @param {String} source 検索対象文字列
 * @param {String} c 検索文字
 * @return {Number} 文字列内に含まれる指定文字の数
 */
StringUtil.count = function(source, c) {
	if (this.isEmpty(source)) {
		return 0;
	}
	var count = 0;
	for ( var i = 0; i <= source.length - 1; i++) {
		if (source.charAt(i) == c) {
			count++;
		}
	}
	return count;
};

/**
 * 指定された文字列内の{#}に対してパラメータStringオブジェクト配列順に文字列を挿入して提供します。<br>
 * パラメータStringオブジェクトがnullの場合は空文字を挿入します。<br>
 * @param {String} source バインド対象ベース文字列
 * @param {Array} params バインドパラメータStringオブジェクト配列
 * @return {String} バインド編集後文字列
 */
StringUtil.bind = function(source, params) {
	if (this.isEmpty(source)) {
		return source;
	}
	if (params === undefined || params === null || params.length === 0) {
		return this.nvl(source);
	}
	var render = source.toString();
	for ( var i = 0; i <= params.length - 1; i++) {
		var value = this.nvl(params[i]);
		render = this.replace(render, "{" + i + "}", value);
	}
	return render;
};

/**
 * 文字列をHTML用にサニタイズして提供します。<br>
 * @param {String} value 対象文字列
 * @returns {String} HTML用にサニタイズした文字列
 */
StringUtil.sanitizeForHTML = function(value) {
	if (this.isEmpty(value)) {
		return this.EMPTY;
	}
	return value //
	.replace(new RegExp("&", "gi"), "&amp;") //
	.replace(new RegExp("<", "gi"), "&lt;") //
	.replace(new RegExp(">", "gi"), "&gt;") //
	.replace(new RegExp("\"", "gi"), "&quot;") //
	.replace(new RegExp("'", "gi"), "&apos;") //
	.replace(new RegExp("\\\\", "gi"), "\\") //
	.replace(new RegExp("\n", "gi"), "<br>") //
	.replace(new RegExp("\t", "gi"), "&nbsp;") //
	.replace(new RegExp("\v", "gi"), "<br>") //
	.replace(new RegExp("\f", "gi"), "<br>") //
	.replace(new RegExp("\0", "gi"), "") //
	.replace(new RegExp("\r", "gi"), "") //
	.replace(new RegExp("\b", "gi"), "") //
	;
};
