/**
 * イベントユーティリティクラス<br>
 * <br>
 * ウィンドウイベントに対する操作インタフェースを提供します。<br>
 *
 * @author Kitagawa<br>
 *
 *<!--
 * 更新日		更新者				更新内容
 * 2012/07/02	Kitagawa			新規作成
 *-->
 */

/**
 * コンストラクタ<br>
 */
var EventUtil = function() {
};

/** ドキュメントページ全体要素 */
EventUtil.PAGE_ELEMENTS = $([window, document, $("body")]);

/** 全イベント名 */
EventUtil.EVENT_NAMES = new Array( //
"blur", //
"focus", //
"change", //
"select", //
"selectstart", //
"submit", //
"reset", //
"abort", //
"error", //
"load", //
"unload", //
"click", //
"dblclick", //
"keyup", //
"keydown", //
"keypress", //
"mouseout", //
"mouseover", //
"mouseup", //
"mousedown", //
"mousemove", //
"dragdrop" //
);

/**
 * 指定されたイベントのキーコードを取得します。<br>
 * @param {Event} event イベントオブジェクト(Window.Event又はjQuery.Event)
 * @returns {Number} キーコード
 */
EventUtil.getKeyCode = function(event) {
	// jQuery.Event又はWindow.Event以外の場合はコード0を返却
	if (!(event instanceof jQuery.Event) && !(event instanceof Event)) {
		return 0;
	}

	// Window.Eventを取得
	var originalEvent = event;
	if (event instanceof jQuery.Event) {
		originalEvent = event.originalEvent;
	}

	if (originalEvent.which) {
		return originalEvent.which;
	} else {
		return originalEvent.keyCode;
	}
};

/**
 * 指定されたイベントキーがCtrlキーを持つか判定します。<br>
 * @param {Event} event イベントオブジェクト(Window.Event又はjQuery.Event)
 * @returns {Boolean} 指定されたイベントキーがCtrlキーを持つ場合にtrueを返却
 */
EventUtil.isCtrlKey = function(event) {
	// jQuery.Event又はWindow.Event以外の場合は処理スキップ
	if (!(event instanceof jQuery.Event) && !(event instanceof Event)) {
		return false;
	}

	// Window.Eventを取得
	var originalEvent = event;
	if (event instanceof jQuery.Event) {
		originalEvent = event.originalEvent;
	}

	return typeof originalEvent.modifiers === "undefined" ? originalEvent.ctrlKey : originalEvent.modifiers & Event.CONTROL_MASK;
};

/**
 * 指定されたイベントキーがShiftキーを持つか判定します。<br>
 * @param {Event} event イベントオブジェクト(Window.Event又はjQuery.Event)
 * @returns {Boolean} 指定されたイベントキーがCtrlキーを持つ場合にtrueを返却
 */
EventUtil.isShiftKey = function(event) {
	// jQuery.Event又はWindow.Event以外の場合は処理スキップ
	if (!(event instanceof jQuery.Event) && !(event instanceof Event)) {
		return false;
	}

	// Window.Eventを取得
	var originalEvent = event;
	if (event instanceof jQuery.Event) {
		originalEvent = event.originalEvent;
	}

	return typeof originalEvent.modifiers === "undefined" ? originalEvent.shiftKey : originalEvent.modifiers & Event.SHIFT_MASK;
};

/**
 * 指定されたイベントキーがAltキーを持つか判定します。<br>
 * @param {Event} event イベントオブジェクト(Window.Event又はjQuery.Event)
 * @returns {Boolean} 指定されたイベントキーがCtrlキーを持つ場合にtrueを返却
 */
EventUtil.isAltKey = function(event) {
	// jQuery.Event又はWindow.Event以外の場合は処理スキップ
	if (!(event instanceof jQuery.Event) && !(event instanceof Event)) {
		return false;
	}

	// Window.Eventを取得
	var originalEvent = event;
	if (event instanceof jQuery.Event) {
		originalEvent = event.originalEvent;
	}

	return typeof originalEvent.modifiers === "undefined" ? originalEvent.altKey : originalEvent.modifiers & Event.ALT_MASK;
};

/**
 * 指定されたイベントオブジェクトをキャンセルします。<br>
 * @param {Event} event イベントオブジェクト(Window.Event又はjQuery.Event)
 * @returns {Boolean} 常にfalseを返却します 
 */
EventUtil.cancelEvent = function(event) {
	// jQuery.Event又はWindow.Event以外の場合は処理スキップ
	if (!(event instanceof jQuery.Event) && !(event instanceof Event)) {
		return false;
	}

	// Window.Eventを取得
	var jqueryEvent = null;
	var originalEvent = event;
	if (event instanceof jQuery.Event) {
		jqueryEvent = event;
		originalEvent = event.originalEvent;
	}

	if (originalEvent) {
		if (originalEvent.type === "focus") {
			if (originalEvent.srcElement && originalEvent.srcElement.blur) {
				originalEvent.srcElement.blur();
			}
		}
		if (originalEvent.cancelBubble) {
			originalEvent.cancelBubble = true;
		}
		if (originalEvent.cancel) {
			originalEvent.cancel = true;
		}
		if (originalEvent.preventDefault) {
			originalEvent.preventDefault();
		}
		if (originalEvent.stopPropagation) {
			originalEvent.stopPropagation();
		}
		if (originalEvent.stopImmediatePropagation) {
			originalEvent.stopImmediatePropagation();
		}
		originalEvent.returnValue = true;
		try {
			originalEvent.keyCode = null;
		} catch (e) {
			// Ctrl、Shiftキー等の特殊キーの場合は設定不可で例外となるが処理不要である為、スルー
		}
	}

	if (jqueryEvent) {
		if (jqueryEvent.cancelBubble) {
			jqueryEvent.cancelBubble = true;
		}
		if (jqueryEvent.cancel) {
			jqueryEvent.cancel = true;
		}
		if (jqueryEvent.preventDefault) {
			jqueryEvent.preventDefault();
		}
		if (jqueryEvent.stopPropagation) {
			jqueryEvent.stopPropagation();
		}
		if (jqueryEvent.stopImmediatePropagation) {
			jqueryEvent.stopImmediatePropagation();
		}
		jqueryEvent.returnValue = true;
	}

	return false;
};

/**
 * ブラウザのコンテキストメニュー操作を無効化します。<br>
 * @param {Object} target 対象要素又はセレクタ文字列(未指定時はドキュメント全体に対して適用します)
 * @returns {Boolean} 正常終了時にtrueを返却
 */
EventUtil.limitContextMenu = function(target) {
	// 対象要素取得
	target = !target ? this.PAGE_ELEMENTS : target;

	// イベントキャンセル
	$(target).bind("contextmenu.EventUtil.limitContextMenu", function(e) {
		return EventUtil.cancelEvent(e);
	});

	return true;
};

/**
 * ブラウザのヘルプ操作を無効化します。<br>
 * @param {Object} target 対象要素又はセレクタ文字列(未指定時はドキュメント全体に対して適用します)
 * @returns {Boolean} 正常終了時にtrueを返却
 */
EventUtil.limitHelp = function(target) {
	// 対象要素取得
	target = !target ? this.PAGE_ELEMENTS : target;

	// イベントキャンセル
	$(target).bind("help.EventUtil.limitContextMenu", function(e) {
		return EventUtil.cancelEvent(e);
	});

	return true;
};

/**
 * ブラウザの履歴操作に伴うキー操作イベントを無効化します。<br>
 * @param {Object} target 対象要素又はセレクタ文字列(未指定時はドキュメント全体に対して適用します)
 * @returns {Boolean} 正常終了時にtrueを返却
 */
EventUtil.limitHistoryControl = function(target) {
	// 対象要素取得
	target = !target ? this.PAGE_ELEMENTS : target;

	// イベントキャンセル
	$(target).bind("keydown.EventUtil.limitHistoryControl", function(e) {
		// [Alt] + [←]
		if (EventUtil.isAltKey(e) && EventUtil.getKeyCode(e) === 37) {
			return EventUtil.cancelEvent(e);
		}
		// [Alt] + [→]
		if (EventUtil.isAltKey(e) && EventUtil.getKeyCode(e) === 39) {
			return EventUtil.cancelEvent(e);
		}
		// [Backspace]
		if (EventUtil.getKeyCode(e) === 8) {
			var src = e.srcElement;
			var srcType = src.type;
			if (srcType === "text" || srcType === "textarea" || srcType === "password") {
				if (src.readOnly) {
					return EventUtil.cancelEvent(e);
				}
			} else {
				return EventUtil.cancelEvent(e);
			}
		}
		return true;
	});

	return true;
};

/**
 * ブラウザのページ更新操作に伴うキー操作イベントを無効化します。<br>
 * @param {Object} target 対象要素又はセレクタ文字列(未指定時はドキュメント全体に対して適用します)
 * @returns {Boolean} 正常終了時にtrueを返却
 */
EventUtil.limitReloadControl = function(target) {
	// 対象要素取得
	target = !target ? this.PAGE_ELEMENTS : target;

	// イベントキャンセル
	$(target).bind("keydown.EventUtil.limitReloadControl", function(e) {
		// [F5]
		if (EventUtil.getKeyCode(e) === 116) {
			return EventUtil.cancelEvent(e);
		}
		// [Ctrl] + [R]
		if (EventUtil.isCtrlKey(e) && EventUtil.getKeyCode(e) === 82) {
			return EventUtil.cancelEvent(e);
		}
		return true;
	});

	return true;
};

/**
 * ブラウザのファンクションキー操作イベントを無効化します。<br>
 * @param {Object} target 対象要素又はセレクタ文字列(未指定時はドキュメント全体に対して適用します)
 * @returns {Boolean} 正常終了時にtrueを返却
 */
EventUtil.limitFunctionControl = function(target) {
	// 対象要素取得
	target = !target ? this.PAGE_ELEMENTS : target;

	// イベントキャンセル
	$(target).bind("keydown.EventUtil.limitFunctionControl", function(e) {
		// [F1～F12]
		if (EventUtil.getKeyCode(e) >= 112 && EventUtil.getKeyCode(e) <= 123) {
			return EventUtil.cancelEvent(e);
		}
		return true;
	});

	return true;
};

/**
 * ブラウザのファンクションキー操作イベントを無効化します。<br>
 * @param {Object} target 対象要素又はセレクタ文字列(未指定時はドキュメント全体に対して適用します)
 * @returns {Boolean} 正常終了時にtrueを返却
 */
EventUtil.lock = function(target) {
	// 対象要素取得
	target = !target ? this.PAGE_ELEMENTS : target;

	// イベントキャンセル
	$(target).bind("keydown.EventUtil.limitFunctionControl", function(e) {
		// [F1～F12]
		if (EventUtil.getKeyCode(e) >= 112 && EventUtil.getKeyCode(e) <= 123) {
			return EventUtil.cancelEvent(e);
		}
		return true;
	});

	return true;
};

/**
 * オブジェクトに対してハンドラを先頭に追加します。<br>
 * @param {Object} target 対象要素又はセレクタ文字列
 * @param {String} name イベント名(名前空間指定可能)
 * @param {Function} fn ハンドラファンクション
 * @returns {Boolean} 正常終了時にtrueを返却
 */
EventUtil.bindAtFirst = function(target, name, fn) {
	//	$(target).bindAtFirst(name, fn);
	//	return true;

	var type = name.contains(".") ? name.substring(0, name.indexOf(".")) : "";
	if (!$.data($(target)[0], "events") || !($.data($(target)[0], "events")[type] || {}).length) {
		$(target).bind(name, fn);
		return false;
	}

	/*
	 * バインド済みハンドラを退避
	 */
	var bindedBuffer = {};
	$.each(($.data($(target)[0], "events") || {})[type], function(key, element) {
		var bindName = "";
		if (StringUtil.isEmpty(element.namespace)) {
			bindName = element.type;
		} else {
			bindName = element.type + "." + element.namespace;
		}
		bindedBuffer[bindName] = element.handler;
	});

	/*
	 * 退避したハンドラを一時的にアンバインド
	 */
	$.each(bindedBuffer, function(key, element) {
		if (ObjectUtil.isPrototypeFunction(bindedBuffer, key)) {
			return true;
		}
		$(target).unbind(key);
	});

	/*
	 * 指定されたファンクションを先頭に追加
	 */
	$(target).bind(name, fn);

	/*
	 * 退避したハンドラを後続に復元
	 */
	$.each(bindedBuffer, function(key, element) {
		if (ObjectUtil.isPrototypeFunction(bindedBuffer, key)) {
			return true;
		}
		$(target).bind(key, element);
	});

	return true;
};
