package org.phosphoresce.lib.commons.util;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

/**
 * Zipファイル操作ユーティリティクラス。<br>
 *
 * @author Kitagawa<br>
 *
 *<!--
 * 更新日		更新者			更新内容
 * 2007/01/16	Kitagawa		新規作成
 * 2007/05/24	Kitagawa		Zipファイル解凍インタフェースの追加
 * 2012/08/28	Kitagawa		JDK1.5+対応
 *-->
 */
public final class ZipUtil {

	/** 入出力バイト長 */
	public static int BYTE_LENGTH = 1024;

	/** ディフォルトキャラクタセット */
	public static final String DEFAULT_CHARSET = "MS932";

	/**
	 * コンストラクタ<br>
	 */
	private ZipUtil() {
		super();
	}

	/**
	 * 指定されたファイルに対してZipContainerオブジェクトの内容をエントリしてZipファイルを生成する。<br>
	 * @param file 作成対象Zipファイルオブジェクト
	 * @param container エントリ内容保持オブジェクト
	 * @throws IOException 正常にZipファイル作成が行えなかった場合に発生
	 */
	public static void create(File file, ZipEntryContainer container) throws IOException {
		create(file, container, DEFAULT_CHARSET);
	}

	/**
	 * 指定されたファイルに対してZipContainerオブジェクトの内容をエントリしてZipファイルを生成する。<br>
	 * @param file 作成対象Zipファイルオブジェクト
	 * @param container エントリ内容保持オブジェクト
	 * @param charset エントリファイルのキャラクタセット(ZipApacheUtil.javaとの互換性APIで実際は動作に関係しません)
	 * @throws IOException 正常にZipファイル作成が行えなかった場合に発生
	 */
	public static void create(File file, ZipEntryContainer container, String charset) throws IOException {
		FileOutputStream stream = new FileOutputStream(file);
		create(stream, container, charset);
		stream.close();
	}

	/**
	 * 指定されたファイルに対してZipContainerオブジェクトの内容をエントリしてZipファイルを生成する。<br>
	 * @param stream 出力ストリームオブジェクト
	 * @param container エントリ内容保持オブジェクト
	 * @param charset エントリファイルのキャラクタセット(ZipApacheUtil.javaとの互換性APIで実際は動作に関係しません)
	 * @throws IOException 正常にZipファイル作成が行えなかった場合に発生
	 */
	public static void create(OutputStream stream, ZipEntryContainer container, String charset) throws IOException {
		ZipOutputStream zos = new ZipOutputStream(stream);

		for (String name : container.keySet()) {
			InputStream eis = container.get(name);

			ZipEntry entry = new ZipEntry(name);
			zos.putNextEntry(entry);

			int count;
			byte[] bytes = new byte[BYTE_LENGTH];

			BufferedInputStream bis = new BufferedInputStream(eis);
			while ((count = bis.read(bytes, 0, BYTE_LENGTH)) != -1) {
				zos.write(bytes, 0, count);
			}
			bis.close();
			zos.closeEntry();
			container.closeStream(name);
		}

		zos.close();
	}

	/**
	 * 指定されたZipファイルを指定されたディレクトリに対して解凍します。<br>
	 * @param file Zipファイル
	 * @param directory 解凍先ディレクトリ
	 * @throws IOException 正常にZipファイルの解凍が行えなかった場合に発生
	 */
	public static void unzip(File file, File directory) throws IOException {
		String target = directory == null ? "" : //
				directory.getAbsolutePath().endsWith(File.separator) ? directory.getAbsolutePath() : directory.getAbsolutePath() + File.separator;
		ZipFile zipFile = new ZipFile(file);
		Enumeration<? extends ZipEntry> enumeration = zipFile.entries();
		while (enumeration.hasMoreElements()) {
			ZipEntry entry = (ZipEntry) enumeration.nextElement();
			if (entry.isDirectory()) {
				new File(target + entry.getName()).mkdirs();
			} else {
				File parent = new File(target + entry.getName()).getParentFile();
				parent.mkdirs();
				FileOutputStream out = new FileOutputStream(target + entry.getName());
				InputStream in = zipFile.getInputStream(entry);
				byte[] buf = new byte[1024];
				int size = 0;
				while ((size = in.read(buf)) != -1) {
					out.write(buf, 0, size);
				}
				out.close();
			}
		}
	}

	/**
	 * 指定されたZipファイルをカレントディレクトリに対して解凍します。<br>
	 * @param file Zipファイル
	 * @throws IOException 正常にZipファイルの解凍が行えなかった場合に発生
	 */
	public static void unzip(File file) throws IOException {
		unzip(file, null);
	}
}