package org.phosphoresce.lib.commons.util;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

/**
 * 日付ユーティリティ<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2012/07/26	Kitagawa		新規作成
 *-->
 */
public class DateUtil {

	/**
	 * コンストラクタ<br>
	 */
	private DateUtil() {
		super();
	}

	/**
	 * 日付をフォーマットして提供します。<br>
	 * @param pattern フォーマットパターン
	 * @param date 日付オブジェクト
	 * @return フォーマットされた文字列
	 */
	public static String format(String pattern, Date date) {
		if (date == null) {
			return "";
		}
		return new SimpleDateFormat(pattern).format(date);
	}

	/**
	 * 指定された日付文字列をフォーマットして日付オブジェクトを提供します。<br>
	 * @param pattern フォーマットパターン
	 * @param date 日付文字列
	 * @return 日付オブジェクト
	 */
	public static Date parse(String pattern, String date) {
		if (StringUtil.isEmpty(date)) {
			return null;
		}
		try {
			return new SimpleDateFormat(pattern).parse(date);
		} catch (Throwable e) {
			throw new IllegalArgumentException(pattern, e);
		}
	}

	/**
	 * 指定された日付の一昨日日付を取得します。<br>
	 * @param date 日付オブジェクト
	 * @return 一昨日日付オブジェクト
	 */
	public static Date yesterday(Date date) {
		GregorianCalendar calendar = new GregorianCalendar();
		calendar.setTime(date);
		calendar.add(Calendar.DATE, -1);
		return calendar.getTime();
	}

	/**
	 * 指定された日付の一翌日日付を取得します。<br>
	 * @param date 日付オブジェクト
	 * @return 一昨日翌日オブジェクト
	 */
	public static Date tommorrow(Date date) {
		GregorianCalendar calendar = new GregorianCalendar();
		calendar.setTime(date);
		calendar.add(Calendar.DATE, +1);
		return calendar.getTime();
	}

	/**
	 * 指定された年号開始日を基に西暦年から和暦年を算出します。<br>
	 * @param startDate 年号開始日
	 * @param targetYear 西暦年
	 * @return 和暦年
	 */
	public static int getJapaneseYear(Date startDate, int targetYear) {
		if (startDate == null) {
			return 0;
		}
		return targetYear - Integer.parseInt(format("yyyy", startDate)) + 1;
	}

	/**
	 * 指定された日付の年を取得します。<br>
	 * @param date 日付オブジェクト
	 * @return 年
	 */
	public static int getYear(Date date) {
		return date == null ? 0 : Integer.parseInt(new SimpleDateFormat("yyyy").format(date));
	}

	/**
	 * 指定された日付の月を取得します。<br>
	 * @param date 日付オブジェクト
	 * @return 月
	 */
	public static int getMonth(Date date) {
		return date == null ? 0 : Integer.parseInt(format("MM", date));
	}

	/**
	 * 指定された日付の日を取得します。<br>
	 * @param date 日付オブジェクト
	 * @return 日
	 */
	public static int getDay(Date date) {
		return date == null ? 0 : Integer.parseInt(format("dd", date));
	}

	/**
	 * 指定された日付の時を取得します。<br>
	 * @param date 日付
	 * @return 指定された日付の時
	 */
	public static int getHour(Date date) {
		return date == null ? 0 : Integer.parseInt(format("HH", date));
	}

	/**
	 * 指定された日付の時(12時間表記)を取得します。<br>
	 * @param date 日付
	 * @return 指定された日付の時(12時間表記)
	 */
	public static int getHour12(Date date) {
		return date == null ? 0 : Integer.parseInt(format("hh", date));
	}

	/**
	 * 指定された日付の分を取得します。<br>
	 * @param date 日付
	 * @return 指定された日付の分
	 */
	public static int getMinute(Date date) {
		return date == null ? 0 : Integer.parseInt(format("mm", date));
	}

	/**
	 * 指定された日付の秒を取得します。<br>
	 * @param date 日付
	 * @return 指定された日付の秒
	 */
	public static int getSecond(Date date) {
		return date == null ? 0 : Integer.parseInt(format("ss", date));
	}

	/**
	 * 指定された日付のミリ秒を取得します。<br>
	 * @param date 日付
	 * @return 指定された日付のミリ秒
	 */
	public static int getMilliSecond(Date date) {
		return date == null ? 0 : Integer.parseInt(format("SSS", date));
	}

	/**
	 * 指定された日付の年(yyyy)を取得します。<br>
	 * @param date 日付オブジェクト
	 * @return 年(yyyy)
	 */
	public static String getYYYY(Date date) {
		return format("yyyy", date);
	}

	/**
	 * 指定された日付の月(MM)を取得します。<br>
	 * @param date 日付オブジェクト
	 * @return 月(MM)
	 */
	public static String getMM(Date date) {
		return format("MM", date);
	}

	/**
	 * 指定された日付の日(dd)を取得します。<br>
	 * @param date 日付オブジェクト
	 * @return 日(dd)
	 */
	public static String getDD(Date date) {
		return format("dd", date);
	}

	/**
	 * 指定された年月の月末日を取得します。<br>
	 * @param year 年
	 * @param month 月
	 * @return 月末日
	 */
	public static int getLastDay(int year, int month) {
		Calendar calendar = Calendar.getInstance();
		calendar.set(year, month - 1, 1);
		return calendar.getActualMaximum(Calendar.DATE);
	}

	/**
	 * 指定された年月の月末日付を取得します。<br>
	 * @param year 年
	 * @param month 月
	 * @return 月末日付オブジェクト
	 * @throws ParseException 正常に日付変換が行えなかった場合にスローされます
	 */
	public static Date getLastDate(int year, int month) throws ParseException {
		return new SimpleDateFormat("yyyy/M/d").parse(year + "/" + month + "/" + getLastDay(year, month));
	}

	/**
	 * 指定された年月日の日付オブジェクトを取得します。<br>
	 * @param year 年
	 * @param month 月
	 * @param day 日
	 * @return 日付オブジェクト
	 * @throws ParseException 正常に日付変換が行えなかった場合にスローされます
	 */
	public static Date getDate(int year, int month, int day) throws ParseException {
		return new SimpleDateFormat("yyyy/M/d").parse(year + "/" + month + "/" + day);
	}

	/**
	 * 指定日付が月初日か判定します。<br>
	 * @param date 日付オブジェクト
	 * @return 月初日の場合にtrueを返却
	 */
	public static boolean isStartOfMonth(Date date) {
		if (date == null) {
			return false;
		}
		return getDay(date) == 1;
	}

	/**
	 * 指定日付が月末日か判定します。<br>
	 * @param date 日付オブジェクト
	 * @return 月末日の場合にtrueを返却
	 */
	public static boolean isEndOfMonth(Date date) {
		if (date == null) {
			return false;
		}
		return getDay(date) == getLastDay(getYear(date), getMonth(date));
	}

	/**
	 * 指定された年がうるう年であるか判定します。<br>
	 * @param year 判定年
	 * @return 指定された年がうるう年であるである場合にtrueを返却
	 */
	public static boolean isLeapYear(int year) {
		return getLastDay(year, 2) == 29;
	}

	/**
	 * 指定された2つの期間が重なり合うか判定します。<br>
	 * @param date1St 比較元期間開始
	 * @param date1End 比較元期間終了
	 * @param date2St 比較先期間開始
	 * @param date2End 比較先期間終了
	 * @return 2つの期間が重なりあう場合にtrueを返却
	 */
	public static final boolean isOverlapPeriod(Date date1St, Date date1End, Date date2St, Date date2End) {
		long date1Stl = date1St == null ? Long.MIN_VALUE : date1St.getTime();
		long date1Endl = date1End == null ? Long.MAX_VALUE : date1End.getTime();
		long date2Stl = date2St == null ? Long.MIN_VALUE : date2St.getTime();
		long date2Endl = date2End == null ? Long.MAX_VALUE : date2End.getTime();
		return date1Stl <= date2Endl && date2Stl <= date1Endl;
	}
}
