package org.phosphoresce.commons.wpoi.adapter;

import java.util.List;

import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.util.CellRangeAddress;

/**
 * シート抽象クラスクラス<br>
 * <br>
 * POIオブジェクトインタフェースの拡張及び、関連インスタンスへのアクセッサを提供します。<br>
 * また、Excel上でのオペレーションに類似したインタフェースを併せて提供します。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2005/11/21	Kitagawa		新規作成
 * 2005/12/07	Kitagawa		PoiPrintConfigオブジェクトによる印刷設定インタフェース追加
 * 2012/07/09	Kitagawa		バージョン2にアップデート
 *-->
 */
public class PoiSheet extends PoiBaseSheet {

	/**
	 * コンストラクタ<br>
	 * @param workbook ワークブックオブジェクト
	 * @param poiSheet シートオブジェクト
	 */
	PoiSheet(PoiWorkbook workbook, Sheet poiSheet) {
		super(workbook, poiSheet);
	}

	/**
	 * シート名を設定します。<br>
	 * @param name シート名
	 */
	public void setSheetName(String name) {
		workbook.getPoiWorkbook().setSheetName(getSheetIndex(), name);
	}

	/**
	 * タイトル行列を設定します。<br>
	 * @param startColIndex タイトル列開始位置(0～)
	 * @param endColIndex タイトル列終了位置(0～)
	 * @param startRowIndex タイトル行開始位置(0～)
	 * @param endRowIndex タイトル行終了位置(0～)
	 */
	public void setTitle(int startColIndex, int endColIndex, int startRowIndex, int endRowIndex) {
		workbook.getPoiWorkbook().setRepeatingRowsAndColumns(getSheetIndex(), startColIndex, endColIndex, startRowIndex, endRowIndex);
	}

	/**
	 * タイトル列を設定します。<br>
	 * @param startColIndex タイトル列開始位置(0～)
	 * @param endColIndex タイトル列終了位置(0～)
	 */
	public void setTitleColumn(int startColIndex, int endColIndex) {
		workbook.getPoiWorkbook().setRepeatingRowsAndColumns(getSheetIndex(), startColIndex, endColIndex, -1, -1);
	}

	/**
	 * タイトル行を設定します。<br>
	 * @param startRowIndex タイトル行開始位置(0～)
	 * @param endRowIndex タイトル行終了位置(0～)
	 */
	public void setTitleRow(int startRowIndex, int endRowIndex) {
		workbook.getPoiWorkbook().setRepeatingRowsAndColumns(getSheetIndex(), -1, -1, startRowIndex, endRowIndex);
	}

	/**
	 * 指定された位置を含む結合セルの結合を解除します。<br>
	 * 指定した位置にいづれの結合セルの一部が含まれない場合は何も行いません。<br>
	 * @param rowIndex 行位置(0～)
	 * @param colIndex 列位置(0～)
	 */
	public void clearMargedRegion(int rowIndex, int colIndex) {
		for (int i = 0; i <= poiSheet.getNumMergedRegions() - 1; i++) {
			CellRangeAddress region = poiSheet.getMergedRegion(i);
			if (rowIndex >= region.getFirstRow() //
					&& rowIndex <= region.getLastRow() //
					&& colIndex >= region.getFirstColumn() //
					&& colIndex <= region.getLastColumn()) {
				poiSheet.removeMergedRegion(i);
			}
		}
	}

	/**
	 * 指定された範囲に含まれる結合セルの結合を全て解除します。<br>
	 * 範囲に結合セルの一部でも含まれる場合はその結合セルも解除対象とします。<br>
	 * @param range 結合解除範囲PoiRangeオブジェクト
	 */
	public void clearMargedRegions(PoiRange range) {
		if (range == null) {
			return;
		}
		for (int i = range.getRowIndexFrom(); i <= range.getRowIndexTo(); i++) {
			for (int j = range.getColIndexFrom(); j <= range.getColIndexTo(); j++) {
				clearMargedRegion(i, j);
			}
		}
	}

	/**
	 * 指定された位置に行を追加します。<br>
	 * @param rowIndex 挿入対象行位置(0～)
	 */
	public void insertRow(int rowIndex) {
		if (rowIndex <= getLastRowNum()) {
			poiSheet.shiftRows(rowIndex, getLastRowNum(), 1, true, true);
		}
		poiSheet.createRow(rowIndex);
	}

	/**
	 * 指定された位置に行を追加します。<br>
	 * @param rowIndex 挿入対象行位置(0～)
	 * @param number 挿入行数
	 */
	public void insertRow(int rowIndex, int number) {
		for (int i = 0; i <= number - 1; i++) {
			insertRow(rowIndex);
		}
	}

	/**
	 * 指定された位置の行を削除して上部に行を詰める処理を実施します。<br>
	 * @param rowIndex 行位置(0～)
	 */
	public void deleteRow(int rowIndex) {
		// 行オブジェクト削除
		if (poiSheet.getRow(rowIndex) != null) {
			poiSheet.removeRow(poiSheet.getRow(rowIndex));
		}

		// 結合セル補完
		for (int i = 0; i <= poiSheet.getNumMergedRegions() - 1; i++) {
			CellRangeAddress region = poiSheet.getMergedRegion(i);
			if (region.getFirstRow() == rowIndex && rowIndex == region.getLastRow()) {
				poiSheet.removeMergedRegion(i);
				i = 0;
			} else if (rowIndex >= region.getFirstRow() && region.getLastRow() >= rowIndex) {
				poiSheet.removeMergedRegion(i);
				region.setLastRow(region.getLastRow() - 1);
				poiSheet.addMergedRegion(region);
				i = 0;
			}
		}

		// 行詰処理
		if (rowIndex + 1 <= poiSheet.getLastRowNum()) {
			poiSheet.shiftRows(rowIndex + 1, poiSheet.getLastRowNum(), -1, true, true);
		}
	}

	/**
	 * 指定された位置の行を削除して上部に行を詰める処理を実施します。<br>
	 * @param rowIndex 行位置(0～)
	 * @param number 削除行数
	 */
	public void deleteRow(int rowIndex, int number) {
		for (int i = 0; i <= number - 1; i++) {
			deleteRow(rowIndex);
		}
	}

	/**
	 * 指定範囲の内容を指定された位置に複写します。<br>
	 * @param range 複写対象範囲オブジェクト
	 * @param rowIndex 複写先行位置(0～)
	 * @param colIndex 複写先列位置(0～)
	 */
	public void copy(PoiRange range, int rowIndex, int colIndex) {
		// 指定レンジに位置する結合範囲情報を取得
		List<CellRangeAddress> regions = PoiBaseUtil.getMargedRegions(poiSheet, range);

		// コピー先レンジ生成
		PoiRange dest = new PoiRange(range);
		int shiftRow = rowIndex - range.getRowIndexFrom();
		int shiftCol = colIndex - range.getColIndexFrom();
		dest.shift(shiftRow, shiftCol);

		clearMargedRegions(dest);
		for (CellRangeAddress region : regions) {
			poiSheet.addMergedRegion(new CellRangeAddress( //
					region.getFirstRow() + shiftRow, //
					region.getLastRow() + shiftRow, //
					region.getFirstColumn() + shiftCol, //
					region.getLastColumn() + shiftCol //
			));
		}

		for (int i = range.getRowIndexFrom(); i <= range.getRowIndexTo(); i++) {
			for (int j = range.getColIndexFrom(); j <= range.getColIndexTo(); j++) {
				getCell(i, j).copy(i + shiftRow, j + shiftCol);
			}
		}
	}
}
