package org.phosphoresce.commons.wpoi.adapter;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.phosphoresce.commons.wpoi.PoiGlobal;
import org.phosphoresce.commons.wpoi.PoiWorkbookType;

/**
 * シート抽象基底クラス<br>
 * <br>
 * 基本的なオブジェクト操作を提供する内部クラスです。<br>
 * 共通的な処理分割のみを目的としたクラスでユーザから直接インスタンス化されて利用されることはありません。<br>
 * 単純な共通処理部の分割クラスであり依存関係が分割された継承クラス関係ではありません。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2005/11/21	Kitagawa		新規作成
 * 2006/03/09	Kitagawa		IO関連の例外をPoiIOExceptionに変更
 * 2012/07/09	Kitagawa		バージョン2にアップデート
 *-->
 */
abstract class PoiBaseWorkbook implements PoiGlobal, Cloneable {

	/** ワークブックタイプ */
	protected PoiWorkbookType type;

	/** ワークブックオブジェクト */
	protected Workbook poiWorkbook;

	/**
	 * コンストラクタ<br>
	 * @param type ワークブックタイプ
	 * @param stream ストリームオブジェクト
	 * @throws IOException 正常にクラスを初期化できなかった場合にスローされます
	 */
	PoiBaseWorkbook(PoiWorkbookType type, InputStream stream) throws IOException {
		super();
		initialize(type, stream);
	}

	/**
	 * コンストラクタ<br>
	 * @param type ワークブックタイプ
	 * @param path ワークブックファイルパス
	 * @throws IOException 正常にクラスを初期化できなかった場合にスローされます
	 */
	PoiBaseWorkbook(PoiWorkbookType type, String path) throws IOException {
		super();
		initialize(type, path);
	}

	/**
	 * コンストラクタ<br>
	 * @param type ワークブックタイプ
	 * @throws IOException 正常にクラスを初期化できなかった場合にスローされます
	 */
	PoiBaseWorkbook(PoiWorkbookType type) throws IOException {
		super();
		initialize(type, (InputStream) null);
	}

	/**
	 * ワークブッククラスを初期化します。<br>
	 * @param type ワークブックタイプ
	 * @param stream ストリームオブジェクト
	 * @throws IOException 正常にワークブックを読み込めなかった場合にスローされます
	 */
	private void initialize(PoiWorkbookType type, InputStream stream) throws IOException {
		this.type = type;
		if (stream != null) {
			if (PoiWorkbookType.EXCEL2003.equals(type)) {
				this.poiWorkbook = new HSSFWorkbook(stream);
			} else if (PoiWorkbookType.EXCEL2007.equals(type)) {
				this.poiWorkbook = new XSSFWorkbook(stream);
			} else {
				this.poiWorkbook = new XSSFWorkbook(stream);
			}
		} else {
			if (PoiWorkbookType.EXCEL2003.equals(type)) {
				this.poiWorkbook = new HSSFWorkbook();
			} else if (PoiWorkbookType.EXCEL2007.equals(type)) {
				this.poiWorkbook = new XSSFWorkbook();
			} else {
				this.poiWorkbook = new XSSFWorkbook();
			}
		}
	}

	/**
	 * ワークブッククラスを初期化します。<br>
	 * @param type ワークブックタイプ
	 * @param path ファイルパス
	 * @throws IOException 正常にワークブックを読み込めなかった場合にスローされます
	 */
	private void initialize(PoiWorkbookType type, String path) throws IOException {
		InputStream stream = null;
		try {
			stream = new FileInputStream(path);
			initialize(type, stream);
		} catch (IOException e) {
			throw e;
		} finally {
			if (stream != null) {
				stream.close();
			}
		}
	}

	/**
	 * ワークブッククラスを初期化します。<br>
	 * @param type ワークブックタイプ
	 * @throws IOException 正常にワークブックを読み込めなかった場合にスローされます
	 */
	private void initialize(PoiWorkbookType type) throws IOException {
		initialize(type, (InputStream) null);
	}

	/**
	 * ワークブックオブジェクトをディープコピー	して複写します。<br>
	 * @return ワークブックオブジェクト
	 * @see java.lang.Object#clone()
	 */
	public PoiWorkbook clone() {
		try {
			ByteArrayOutputStream bos = new ByteArrayOutputStream();
			poiWorkbook.write(bos);
			ByteArrayInputStream bis = new ByteArrayInputStream(bos.toByteArray());
			return new PoiWorkbook(type, bis);
		} catch (Throwable e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * ワークブックオブジェクトを取得します。<br>
	 * @return ワークブックオブジェクト
	 */
	public Workbook getPoiWorkbook() {
		return poiWorkbook;
	}

	/**
	 * シートオブジェクトを取得します。<br>
	 * @param sheetName シート名
	 * @return シートオブジェクト
	 */
	public PoiSheet getSheet(String sheetName) {
		return new PoiSheet((PoiWorkbook) this, poiWorkbook.getSheet(sheetName));
	}

	/**
	 * シートオブジェクトを取得します。<br>
	 * @param sheetIndex シートインデックス
	 * @return シートオブジェクト
	 */
	public PoiSheet getSheet(int sheetIndex) {
		while (poiWorkbook.getNumberOfSheets() <= sheetIndex) {
			poiWorkbook.createSheet(DEFAULT_SHEET_PREFIX + (poiWorkbook.getNumberOfSheets() + 1));
		}
		return new PoiSheet((PoiWorkbook) this, poiWorkbook.getSheetAt(sheetIndex));
	}
}
