package org.phosphoresce.commons.wpoi.adapter;

import org.apache.poi.ss.usermodel.Row;
import org.phosphoresce.commons.wpoi.PoiGlobal;

/**
 * 行抽象インタフェースクラス<br>
 * <br>
 * 基本的なオブジェクト操作を提供する内部クラスです。<br>
 * 共通的な処理分割のみを目的としたクラスでユーザから直接インスタンス化されて利用されることはありません。<br>
 * 単純な共通処理部の分割クラスであり依存関係が分割された継承クラス関係ではありません。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2005/11/21	Kitagawa		新規作成
 * 2006/02/09	Kitagawa		setValueにてnullオブジェクトを判定して空文字列を入れるように修正
 * 2006/03/24	Kitagawa		setValueの文字列設定時に数値に変換できる文字列は数値としてセットするように修正(Excel仕様)
 * 2006/04/19	Kitagawa		getForceStringValueのインタフェースとして計算後の値を取得する為のパラメタを持つメソッドを追加
 * 2012/07/09	Kitagawa		バージョン2にアップデート
 *-->
 */
abstract class PoiBaseRow implements PoiGlobal {

	/** ワークブックオブジェクト */
	protected PoiWorkbook workbook;

	/** シートオブジェクト */
	protected PoiSheet sheet;

	/** 行オブジェクト */
	protected Row poiRow;

	/**
	 * コンストラクタ<br>
	 * @param workbook ワークブックオブジェクト
	 * @param sheet シートオブジェクト
	 * @param poiRow 行オブジェクト
	 */
	PoiBaseRow(PoiWorkbook workbook, PoiSheet sheet, Row poiRow) {
		super();
		this.workbook = workbook;
		this.sheet = sheet;
		this.poiRow = poiRow;
	}

	/**
	 * 行オブジェクトを取得します。<br>
	 * @return 行オブジェクト
	 */
	public Row getPoiRow() {
		return poiRow;
	}

	/**
	 * ワークブックオブジェクトを取得します。<br>
	 * @return ワークブックオブジェクト
	 */
	public PoiWorkbook getWorkbook() {
		return workbook;
	}

	/**
	 * シートオブジェクトを取得します。<br>
	 * @return シートオブジェクト
	 */
	public PoiSheet getSheet() {
		return sheet;
	}

	/**
	 * セルオブジェクトを取得します。<br>
	 * @param colIndex 列位置(0～)
	 * @return セルオブジェクト
	 */
	public PoiCell getCell(int colIndex) {
		return new PoiCell((PoiWorkbook) workbook, (PoiSheet) sheet, (PoiRow) this, poiRow.getCell(colIndex, Row.CREATE_NULL_AS_BLANK));
	}

	/**
	 * クラス情報を文字列で提供します。<br>
	 * @return クラス情報文字列
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("[");
		for (int i = poiRow.getFirstCellNum(); i <= poiRow.getLastCellNum(); i++) {
			builder.append(getCell(i).toString());
			if (i <= poiRow.getLastCellNum() - 1) {
				builder.append(", ");
			}
		}
		builder.append("]");
		return builder.toString();
	}
}
