package org.phosphoresce.commons.wpoi.adapter;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;

import org.apache.poi.ss.usermodel.Cell;
import org.phosphoresce.commons.wpoi.PoiGlobal;

/**
 * セル抽象インタフェースクラス<br>
 * <br>
 * 基本的なオブジェクト操作を提供する内部クラスです。<br>
 * 共通的な処理分割のみを目的としたクラスでユーザから直接インスタンス化されて利用されることはありません。<br>
 * 単純な共通処理部の分割クラスであり依存関係が分割された継承クラス関係ではありません。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2005/11/21	Kitagawa		新規作成
 * 2012/07/09	Kitagawa		バージョン2にアップデート
 *-->
 */
abstract class PoiBaseCell implements PoiGlobal {

	/** ワークブックオブジェクト */
	protected PoiWorkbook workbook;

	/** シートオブジェクト */
	protected PoiSheet sheet;

	/** 行オブジェクト */
	protected PoiRow row;

	/** セルオブジェクト */
	protected Cell poiCell;

	/**
	 * コンストラクタ<br>
	 * @param workbook ワークブックオブジェクト
	 * @param sheet シートオブジェクト
	 * @param row 行オブジェクト
	 * @param poiCell セルオブジェクト
	 */
	PoiBaseCell(PoiWorkbook workbook, PoiSheet sheet, PoiRow row, Cell poiCell) {
		super();
		this.workbook = workbook;
		this.sheet = sheet;
		this.row = row;
		this.poiCell = poiCell;
	}

	/**
	 * セルオブジェクトを取得します。<br>
	 * @return セルオブジェクト
	 */
	public Cell getPoiCell() {
		return poiCell;
	}

	/**
	 * ワークブックオブジェクトを取得します。<br>
	 * @return ワークブックオブジェクト
	 */
	public PoiWorkbook getWorkbook() {
		return workbook;
	}

	/**
	 * シートオブジェクトを取得します。<br>
	 * @return シートオブジェクト
	 */
	public PoiSheet getSheet() {
		return sheet;
	}

	/**
	 * 行オブジェクトを取得します。<br>
	 * @return 行オブジェクト
	 */
	public PoiRow getRow() {
		return row;
	}

	/**
	 * セル値をセットします。<br>
	 * @param value セル値
	 */
	public void setValue(Object value) {
		if (value == null) {
			poiCell.setCellType(Cell.CELL_TYPE_BLANK);
		} else if (value instanceof Boolean) {
			poiCell.setCellValue((Boolean) value);
		} else if (value instanceof Date) {
			poiCell.setCellValue((Date) value);
		} else if (value instanceof Calendar) {
			poiCell.setCellValue((Calendar) value);
		} else if (value instanceof Short) {
			poiCell.setCellValue((Short) value);
		} else if (value instanceof Integer) {
			poiCell.setCellValue((Integer) value);
		} else if (value instanceof Float) {
			poiCell.setCellValue((Float) value);
		} else if (value instanceof Double) {
			poiCell.setCellValue((Double) value);
		} else if (value instanceof Long) {
			poiCell.setCellValue((Long) value);
		} else if (value instanceof BigDecimal) {
			poiCell.setCellValue(((BigDecimal) value).doubleValue());
		} else if (value instanceof String) {
			if ("".equals(value)) {
				poiCell.setCellType(Cell.CELL_TYPE_BLANK);
			} else {
				poiCell.setCellValue((String) value);
			}
		} else {
			poiCell.setCellValue(value.toString());
		}
	}

	/**
	 * セルに設定されている計算式を取得します。<br>
	 * @return 計算式文字列
	 */
	public String getFormula() {
		if (poiCell.getCellType() == Cell.CELL_TYPE_FORMULA) {
			return poiCell.getCellFormula();
		} else {
			return EMPTY;
		}
	}

	/**
	 * セルに計算式を設定します。<br>
	 * @param formula 計算式文字列
	 */
	public void setFormula(String formula) {
		poiCell.setCellType(Cell.CELL_TYPE_FORMULA);
		poiCell.setCellFormula(formula);
	}

	/**
	 * セル値を取得します。<br>
	 * @param calc 該当セルの再計算実施後の値を利用する場合にtrueを設定
	 * @return 文字列として変換したセル値
	 */
	public Object getValue(boolean calc) {
		if (poiCell.getCellType() == Cell.CELL_TYPE_BLANK) {
			return EMPTY;
		} else if (poiCell.getCellType() == Cell.CELL_TYPE_BOOLEAN) {
			return poiCell.getBooleanCellValue();
		} else if (poiCell.getCellType() == Cell.CELL_TYPE_NUMERIC) {
			return poiCell.getNumericCellValue();
		} else if (poiCell.getCellType() == Cell.CELL_TYPE_STRING) {
			return poiCell.getStringCellValue();
		} else if (poiCell.getCellType() == Cell.CELL_TYPE_ERROR) {
			return poiCell.getErrorCellValue();
		} else if (poiCell.getCellType() == Cell.CELL_TYPE_FORMULA) {
			if (calc) {
				PoiBaseUtil.evaluateCell(workbook.getPoiWorkbook(), poiCell);
				return getValue(false);
			} else {
				return PoiBaseUtil.getBaseValue(poiCell);
			}
		} else {
			return EMPTY;
		}
	}

	/**
	 * セル値を取得します。<br>
	 * 計算式が指定されている場合は再計算を行わずに提供します。<br>
	 * @return 文字列として変換したセル値
	 */
	public Object getValue() {
		return getValue(false);
	}

	/**
	 * セル値を取得します。<br>
	 * @param calc 該当セルの再計算実施後の値を利用する場合にtrueを設定
	 * @return 文字列として変換したセル値
	 */
	public String getString(boolean calc) {
		if (poiCell.getCellType() == Cell.CELL_TYPE_BLANK) {
			return EMPTY;
		} else if (poiCell.getCellType() == Cell.CELL_TYPE_FORMULA) {
			if (calc) {
				PoiBaseUtil.evaluateCell(workbook.getPoiWorkbook(), poiCell);
				return getString(false);
			} else {
				return String.valueOf(PoiBaseUtil.getBaseValue(poiCell));
			}
		} else if (poiCell.getCellType() == Cell.CELL_TYPE_BOOLEAN) {
			return String.valueOf(poiCell.getBooleanCellValue());
		} else if (poiCell.getCellType() == Cell.CELL_TYPE_ERROR) {
			return String.valueOf(PoiBaseUtil.getBaseValue(poiCell));
		} else if (poiCell.getCellType() == Cell.CELL_TYPE_NUMERIC) {
			return String.valueOf(PoiBaseUtil.getBaseValue(poiCell));
		} else if (poiCell.getCellType() == Cell.CELL_TYPE_STRING) {
			return String.valueOf(poiCell.getStringCellValue());
		} else {
			return EMPTY;
		}
	}

	/**
	 * セル値を取得します。<br>
	 * 計算式が指定されている場合は再計算を行わずに提供します。<br>
	 * @return 文字列として変換したセル値
	 */
	public String getString() {
		return getString(false);
	}

	/**
	 * クラス情報を文字列で提供します。<br>
	 * @return クラス情報文字列
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return getString();
	}
}
