package org.phosphoresce.commons.socket.http.io;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;

import org.phosphoresce.commons.socket.core.SocketGlobal;
import org.phosphoresce.commons.util.StreamUtil;

/**
 * HTTPコンテンツデータ出力ストリームクラス<br>
 * <br>
 * 当クラスはインスタンス初期化時に指定されたコンテンツデータ形式によって、
 * ソケットストリームへのコンテンツボディ書き込みの方式を適正な形で実行する
 * コンテンツ出力アダプタストリームクラスです。<br>
 * <br>
 * 但し、Chunked形式の場合、データが末端であることを当クラスが認識することは
 * 不可能であるため、ユーザーが明示的にwriteEOCをコールする必要があります。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/11/10	Kitagawa		新規作成
 *-->
 */
public class HttpSocketContentOutputStream extends OutputStream {

	/** ソケットストリーム */
	private OutputStream stream = null;

	/** Chunkedコンテンツデータフラグ */
	private boolean chunked = false;

	/** データストリームバッファ */
	private ByteArrayOutputStream buffer = new ByteArrayOutputStream();

	/** データストリームバッファサイズ */
	private int bufferSize = SocketGlobal.DEFAULT_STREAM_BUFFER_SIZE;

	/** バッファ済みトータルサイズ */
	private long totalSize = 0;

	/**
	 * コンストラクタ<br>
	 */
	private HttpSocketContentOutputStream() {
		super();
	}

	/**
	 * コンストラクタ<br>
	 * @param stream ソケットストリーム
	 * @param chunked Chunkedコンテンツデータフラグ
	 * @param bufferSize ストリームデータバッファサイズ
	 * @throws IOException クラス初期化時にストリームからのバッファ取得が行えなかった場合に発生
	 */
	public HttpSocketContentOutputStream(OutputStream stream, boolean chunked, int bufferSize) throws IOException {
		super();
		this.stream = stream;
		this.chunked = chunked;
		this.bufferSize = bufferSize < 0 ? 1 : bufferSize;
		this.totalSize = 0;
	}

	/**
	 * コンストラクタ<br>
	 * @param stream ソケットストリーム
	 * @param chunked Chunkedコンテンツデータフラグ
	 * @throws IOException クラス初期化時にストリームからのバッファ取得が行えなかった場合に発生
	 */
	public HttpSocketContentOutputStream(OutputStream stream, boolean chunked) throws IOException {
		super();
		this.stream = stream;
		this.chunked = chunked;
		this.bufferSize = SocketGlobal.DEFAULT_STREAM_BUFFER_SIZE;
		this.totalSize = 0;
	}

	/**
	 * 扱うストリームがChunked形式であるか判定します。<br>
	 * @return 扱うストリームがChunked形式である場合にtrueを返却
	 */
	public boolean isChunked() {
		return chunked;
	}

	/**
	 * バッファ済みトータルサイズを取得します。<br>
	 * @return バッファ済みトータルサイズ
	 */
	public long getTotalSize() {
		return totalSize;
	}

	/**
	 * Chunkedコンテンツデータの末端を書き込みます。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public void writeEOC() throws IOException {
		flushBuffer();
		StreamUtil.writeLine(stream, "0");
		StreamUtil.writeLine(stream); // TODO フッタ行は･･･
	}

	/**
	 * バイトデータを1バイト書き込みます。<br>
	 * @param data バイトデータ
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.OutputStream#write(int)
	 */
	public void write(int data) throws IOException {
		buffer.write(data);
		if (buffer.size() >= bufferSize) {
			flushBuffer();
		}
	}

	/**
	 * バイト配列を書き込みます。<br>
	 * @param bytes バイト配列
	 * @param offset 書き込むバイト配列のオフセット位置
	 * @param length 書き込むバイト配列の長さ
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.OutputStream#write(byte[], int, int)
	 */
	public void write(byte[] bytes, int offset, int length) throws IOException {
		buffer.write(bytes, offset, length);
		if (buffer.size() >= bufferSize) {
			flushBuffer();
		}
	}

	/**
	 * バイト配列を書き込みます。<br>
	 * @param bytes バイト配列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.OutputStream#write(byte[])
	 */
	public void write(byte[] bytes) throws IOException {
		write(bytes, 0, bytes.length);
		//		if (buffer.size() >= bufferSize) {
		//			flushBuffer();
		//		}
	}

	/**
	 * バッファに溜め込まれている情報を全て書き込みます。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	private void flushBuffer() throws IOException {
		if (buffer.size() <= 0) {
			return;
		}
		if (chunked) {
			// Chunkedデータサイズ行書き込み
			StreamUtil.writeLine(stream, Integer.toHexString(buffer.size()));
		}
		StreamUtil.writes(stream, buffer.toByteArray());
		if (chunked) {
			// Chunked復帰行書き込み
			StreamUtil.writeLine(stream);
		}
		totalSize += buffer.size();
		buffer.reset();
	}

	/**
	 * ストリームをフラッシュします。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.OutputStream#flush()
	 */
	public void flush() throws IOException {
		flushBuffer();
		stream.flush();
	}

	/**
	 * ストリームをクローズします。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.OutputStream#close()
	 */
	public void close() throws IOException {
		flushBuffer();
		stream.flush();
		stream.close();
	}
}
