package org.phosphoresce.commons.socket.http.io;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.phosphoresce.commons.socket.core.SocketGlobal;
import org.phosphoresce.commons.util.StreamUtil;

/**
 * HTTPコンテンツデータ入力ストリームクラス<br>
 * <br>
 * 当クラスはインスタンス初期化時に指定されたコンテンツデータ形式によって、
 * ソケットストリームからのコンテンツボディ読み込みの方式を適正な形で実行し、
 * ユーザーに提供する、コンテンツ入力アダプタストリームクラスです。<br>
 * <br>
 * コンテンツボディデータがContent-Lengthによる明示的なデータ長の提示がある場合や、
 * Chunkedによる未決のデータ長、Content-Lengthによるデータ長の提示がない場合で、
 * それぞれのデータ読み込み方式を内部で切り分けます。<br>
 * 当クラスを使用するユーザーはこれらのデータ読み込み方式を意識する必要はなく、
 * 通常の入力ストリームと同様の方法で読み込むことでコンテンツ取得が可能です。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/11/10	Kitagawa		新規作成
 *-->
 */
public class HttpSocketContentInputStream extends InputStream {

	/** ソケットストリーム */
	private InputStream stream = null;

	/** Chunkedコンテンツデータフラグ */
	private boolean chunked = false;

	/** コンテンツボディデータ長 */
	private long contentLength = -1;

	/** データストリームバッファ */
	private ByteArrayInputStream buffer = new ByteArrayInputStream(new byte[0]);

	/** データストリームバッファサイズ */
	private int bufferSize = SocketGlobal.DEFAULT_STREAM_BUFFER_SIZE;

	/** ストリームデータ末端フラグ */
	private boolean eos = false;

	/** トータルサイズ */
	private long totalSize = 0;

	/**
	 * コンストラクタ<br>
	 */
	private HttpSocketContentInputStream() {
		super();
	}

	/**
	 * コンストラクタ<br>
	 * @param stream ソケットストリーム
	 * @param chunked Chunkedコンテンツデータフラグ
	 * @param contentLength コンテンツボディデータ長
	 * @param bufferSize ストリームデータバッファサイズ
	 * @throws IOException クラス初期化時にストリームからのバッファ取得が行えなかった場合に発生
	 */
	public HttpSocketContentInputStream(InputStream stream, boolean chunked, long contentLength, int bufferSize) throws IOException {
		super();
		this.stream = stream;
		this.chunked = chunked;
		this.contentLength = contentLength;
		this.bufferSize = bufferSize < 0 ? 1 : bufferSize;
		this.eos = false;
		this.totalSize = 0;
	}

	/**
	 * コンストラクタ<br>
	 * @param stream ソケットストリーム
	 * @param chunked Chunkedコンテンツデータフラグ
	 * @param contentLength コンテンツボディデータ長
	 * @throws IOException クラス初期化時にストリームからのバッファ取得が行えなかった場合に発生
	 */
	public HttpSocketContentInputStream(InputStream stream, boolean chunked, long contentLength) throws IOException {
		super();
		this.stream = stream;
		this.chunked = chunked;
		this.contentLength = contentLength;
		this.bufferSize = SocketGlobal.DEFAULT_STREAM_BUFFER_SIZE;
		this.eos = false;
		this.totalSize = 0;
	}

	/**
	 * 扱うストリームがChunked形式であるか判定します。<br>
	 * @return 扱うストリームがChunked形式である場合にtrueを返却
	 */
	public boolean isChunked() {
		return chunked;
	}

	/**
	 * トータルサイズを取得します。<br>
	 * @return トータルサイズ
	 */
	public long getTotalSize() {
		return totalSize;
	}

	/**
	 * ストリームバッファにデータをバッファリングします。<br>
	 * 但し、既にバッファされているデータが残っている場合は処理はスキップします。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	private void readBuffer() throws IOException {
		if (eos || buffer.available() > 0) {
			return;
		}
		if (chunked) {
			int readSize = readChunkedNextSize();
			buffer = new ByteArrayInputStream(StreamUtil.reads(stream, readSize));
			totalSize += buffer.available();
			if (buffer.available() == 0) {
				eos = true;
			}
			StreamUtil.readLine(stream); // Chunked復帰行
		} else if (contentLength == 0) {
			eos = true;
			return;
		} else if (contentLength > 0) {
			int readSize = (int) (((totalSize + bufferSize) > contentLength) ? (contentLength - totalSize) : bufferSize);
			buffer = new ByteArrayInputStream(StreamUtil.reads(stream, readSize));
			totalSize += buffer.available();
			if (contentLength <= totalSize) {
				eos = true;
			}
		} else {
			if (stream.available() <= 0) {
				eos = true;
				return;
			}
			buffer = new ByteArrayInputStream(new byte[] { (byte) stream.read() });
			totalSize += 1;
		}
	}

	/**
	 * 次に提供されるChunkedコンテンツデータ長を取得します。<br>
	 * 当メソッドはコンテンツデータがChunkedである場合に限ります。<br>
	 * @return 次に提供されるChunkedコンテンツデータ長
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	private int readChunkedNextSize() throws IOException {
		String chunkIdentifier = StreamUtil.readLine(stream);
		String chunkSizeBlock = chunkIdentifier.indexOf(";") >= 0 ? chunkIdentifier.substring(0, chunkIdentifier.indexOf(";")).trim() : chunkIdentifier.trim();
		return Integer.parseInt(chunkSizeBlock.trim(), 16);
	}

	/**
	 * ストリームデータを読み込みます。<br>
	 * @return ストリームから読み込まれた1バイトのデータ
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#read()
	 */
	public int read() throws IOException {
		readBuffer();
		return buffer.available() > 0 ? buffer.read() : -1;
	}

	/**
	 * このストリームから読み込むことの出来るデータサイズを取得します。<br>
	 * 尚、ここで提供されるデータサイズは読み込まれる毎に変化します。<br>
	 * 変化する内容はコンテンツデータの形式によってことなります。<br>
	 * 例えば、Chunkedデータの場合は次に提供されるデータの長さが提供され、
	 * コンテンツ長が指定されている場合はバッファされているサイズデータが提供されます。<br>
	 * Chunkedデータでもなく、コンテンツ長の指定がない場合はデータが存在する間、
	 * 常に1が返却され、データがなくなり次第0が返却されることになります。<br>
	 * @return ストリームから読み込むことの出来るデータサイズ
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#available()
	 */
	public int available() throws IOException {
		readBuffer();
		return buffer.available();
	}

	/**
	 * ストリームをクローズします。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#close()
	 */
	public void close() throws IOException {
		stream.close();
	}
}
