package org.phosphoresce.commons.socket.http.container;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.LinkedList;
import java.util.List;

import org.phosphoresce.commons.socket.core.SocketGlobal;
import org.phosphoresce.commons.socket.http.HttpResponseStatus;
import org.phosphoresce.commons.util.StringUtil;

/**
 * HTTPソケットレスポンス情報保持クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/11/14	Kitagawa		新規作成
 *-->
 */
public class HttpSocketResponse extends HttpSocketHeaderContainer {

	/** プロトコル */
	private String protocol;

	/** プロトコルバージョン */
	private String version;

	/** ステータス */
	private HttpResponseStatus status;

	/** メッセージ */
	private String message;

	/** ヘッダ情報 */
	private List headers;

	/** クッキー情報 */
	private List cookies;

	/** コンテンツストリーム */
	private InputStream contentStream;

	// Constructor

	/**
	 * 
	 * コンストラクタ<br>
	 * @param protocol プロトコル
	 * @param version プロトコルバージョン
	 * @param status ステータス
	 * @param message メッセージ
	 */
	public HttpSocketResponse(String protocol, String version, HttpResponseStatus status, String message) {
		super(false);
		this.protocol = protocol;
		this.version = version;
		this.status = status;
		this.message = message;
		this.headers = new LinkedList();
		this.cookies = new LinkedList();
		this.contentStream = null;
	}

	/**
	 * 
	 * コンストラクタ<br>
	 * @param protocol プロトコル
	 * @param version プロトコルバージョン
	 * @param status ステータス
	 * @param message メッセージ
	 */
	public HttpSocketResponse(String protocol, String version, String status, String message) {
		this(protocol, version, HttpResponseStatus.getStatus(message), message);
	}

	/**
	 * 
	 * コンストラクタ<br>
	 * @param protocol プロトコル
	 * @param version プロトコルバージョン
	 * @param status ステータス
	 * @param message メッセージ
	 */
	public HttpSocketResponse(String protocol, String version, int status, String message) {
		this(protocol, version, HttpResponseStatus.getStatus(message), message);
	}

	/**
	 * 
	 * コンストラクタ<br>
	 */
	public HttpSocketResponse() {
		//this(null, null, (HttpResponseStatus) null, null);
		this("HTTP", "1.1", HttpResponseStatus.STATUS_200_OK, HttpResponseStatus.STATUS_200_OK.getMessage());
	}

	// Accessor Method

	/**
	 * プロトコルを取得します。<br>
	 * @return プロトコル
	 */
	public String getProtocol() {
		return protocol == null ? "" : protocol;
	}

	/**
	 * プロトコルを設定します。<br>
	 * @param protocol プロトコル
	 */
	public void setProtocol(String protocol) {
		this.protocol = protocol;
	}

	/**
	 * プロトコルバージョンを取得します。<br>
	 * @return プロトコルバージョン
	 */
	public String getVersion() {
		return version == null ? "" : version;
	}

	/**
	 * プロトコルバージョンを設定します。<br>
	 * @param version プロトコルバージョン
	 */
	public void setVersion(String version) {
		this.version = version;
	}

	/**
	 * ステータスを取得します。<br>
	 * @return ステータス
	 */
	public HttpResponseStatus getStatus() {
		return status;
	}

	/**
	 * ステータスを設定します。<br>
	 * @param status ステータス
	 */
	public void setStatus(HttpResponseStatus status) {
		this.status = status;
		this.message = status == null ? "" : status.getMessage();
	}

	/**
	 * メッセージを取得します。<br>
	 * @return メッセージ
	 */
	public String getMessage() {
		return message == null ? "" : message;
	}

	/**
	 * メッセージを設定します。<br>
	 * @param message メッセージ
	 */
	public void setMessage(String message) {
		this.message = message;
	}

	/**
	 * コンテンツストリームを取得します。<br>
	 * @return コンテンツストリーム
	 */
	public InputStream getContentStream() {
		return contentStream;
	}

	/**
	 * コンテンツストリームを設定します。<br>
	 * @param contentStream コンテンツストリーム
	 */
	public void setContentStream(InputStream contentStream) {
		this.contentStream = contentStream;
	}

	/**
	 * コンテンツデータをテキストとして取得します。<br>
	 * このメソッドを使用することで入力ストリームから提供されるデータが全て読み込まれることに留意してください。<br>
	 * また、ストリームから提供されるデータを全て読み込むため、必ずテキストであるコンテンツであることを
	 * 判定した上で使用する必要があります。膨大なストリームデータが提供される場合、OutOfMemoryとなる可能性があります。<br>
	 * @param charset コンテンツ文字コード
	 * @param bufferSize ストリームバッファサイズ
	 * @return テキストとしてのコンテンツデータ
	 * @throws IOException 入力ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public String getContentText(String charset, int bufferSize) throws IOException {
		ByteArrayOutputStream bufferStream = new ByteArrayOutputStream();
		if (contentStream != null) {
			while (contentStream.available() > 0) {
				int readSize = contentStream.available() > bufferSize ? bufferSize : contentStream.available();
				byte[] bytes = new byte[readSize];
				contentStream.read(bytes);
				bufferStream.write(bytes);
			}
		}
		return charset == null ? new String(bufferStream.toByteArray()) : new String(bufferStream.toByteArray(), charset);
	}

	/**
	 * コンテンツデータをテキストとして取得します。<br>
	 * このメソッドを使用することで入力ストリームから提供されるデータが全て読み込まれることに留意してください。<br>
	 * また、ストリームから提供されるデータを全て読み込むため、必ずテキストであるコンテンツであることを
	 * 判定した上で使用する必要があります。膨大なストリームデータが提供される場合、OutOfMemoryとなる可能性があります。<br>
	 * @param bufferSize ストリームバッファサイズ
	 * @return テキストとしてのコンテンツデータ
	 * @throws IOException 入力ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public String getContentText(int bufferSize) throws IOException {
		return getContentText(null, bufferSize);
	}

	/**
	 * コンテンツデータをテキストとして取得します。<br>
	 * このメソッドを使用することで入力ストリームから提供されるデータが全て読み込まれることに留意してください。<br>
	 * また、ストリームから提供されるデータを全て読み込むため、必ずテキストであるコンテンツであることを
	 * 判定した上で使用する必要があります。膨大なストリームデータが提供される場合、OutOfMemoryとなる可能性があります。<br>
	 * @return テキストとしてのコンテンツデータ
	 * @throws IOException 入力ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public String getContentText() throws IOException {
		return getContentText(null, SocketGlobal.DEFAULT_STREAM_BUFFER_SIZE);
	}

	// Class Method

	/**
	 * レスポンス識別行を取得します。<br>
	 * @return レスポンス識別行
	 */
	public String getIdentifier() {
		StringBuffer buffer = new StringBuffer();
		if (!StringUtil.isEmpty(getProtocol())) {
			buffer.append(getProtocol());
			buffer.append("/");
		}
		if (!StringUtil.isEmpty(getVersion())) {
			buffer.append(getVersion());
			buffer.append(" ");
		}
		if (getStatus() != null) {
			buffer.append(getStatus().getCode());
			buffer.append(" ");
		}
		if (!StringUtil.isEmpty(getMessage())) {
			buffer.append(getMessage());
			buffer.append(" ");
		}
		return buffer.toString().trim();
	}

	/**
	 * 指定されたレスポンス識別行をクラスに対して設定します。<br>
	 * @param identifier レスポンス識別行
	 */
	public void setIdentifier(String identifier) {
		setProtocol(getProtocolByIdentifier(identifier));
		setVersion(getVersionByIdentifier(identifier));
		setStatus(getStatusByIdentifier(identifier));
		setMessage(getMessageByIdentifier(identifier));
	}

	/**
	 * 指定されたヘッダからレスポンスプロトコルを取得します。<br>
	 * @param header ヘッダ文字列
	 * @return レスポンスプロトコル
	 */
	private String getProtocolByIdentifier(String header) {
		if (header == null || header.length() == 0) {
			return null;
		}
		String[] values = header.split(" ", 3);
		if (values.length < 1) {
			return null;
		}
		String[] buffers = values[0].split("/", 2);
		if (buffers.length < 1) {
			return null;
		}
		return buffers[0];
	}

	/**
	 * 指定されたヘッダからレスポンスプロトコルバージョンを取得します。<br>
	 * @param header ヘッダ文字列
	 * @return レスポンスプロトコルバージョン
	 */
	private String getVersionByIdentifier(String header) {
		if (header == null || header.length() == 0) {
			return null;
		}
		String[] values = header.split(" ", 3);
		if (values.length < 1) {
			return null;
		}
		String[] buffers = values[0].split("/", 2);
		if (buffers.length < 2) {
			return null;
		}
		return buffers[1];
	}

	/**
	 * 指定されたヘッダからレスポンスステータスコードを取得します。<br>
	 * @param header ヘッダ文字列
	 * @return レスポンスステータスコード
	 */
	private HttpResponseStatus getStatusByIdentifier(String header) {
		if (header == null || header.length() == 0) {
			return null;
		}
		String[] values = header.split(" ", 3);
		if (values.length < 2) {
			return null;
		}
		return HttpResponseStatus.getStatus(values[1], getMessageByIdentifier(header));
	}

	/**
	 * 指定されたヘッダからレスポンスメッセージを取得します。<br>
	 * @param header ヘッダ文字列
	 * @return レスポンスメッセージ
	 */
	private String getMessageByIdentifier(String header) {
		if (header == null || header.length() == 0) {
			return null;
		}
		String[] values = header.split(" ", 3);
		if (values.length < 3) {
			return null;
		}
		return values[2];
	}

	/**
	 * オブジェクトの文字列表現を返します。<br>
	 * @return オブジェクトの文字列表現
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("{");
		buffer.append("protocol=");
		buffer.append(getProtocol());
		buffer.append(",");
		buffer.append("version=");
		buffer.append(getVersion());
		buffer.append(",");
		buffer.append("status=");
		buffer.append(getStatus());
		buffer.append(",");
		buffer.append("message=");
		buffer.append(getMessage());
		buffer.append(",");
		buffer.append("headers=");
		buffer.append(getHeaders());
		buffer.append(",");
		buffer.append("cookies=");
		buffer.append(getCookies());
		buffer.append("}");
		return buffer.toString();
	}
}
