package org.phosphoresce.commons.socket.http.container;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.phosphoresce.commons.util.StreamUtil;
import org.phosphoresce.commons.util.StringUtil;

/**
 * HTTPソケットリクエストパラメータ管理クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/11/25	Kitagawa		新規作成
 *-->
 */
public class HttpSocketRequestPrameter {

	/** パラメータマップ */
	private Map parameter;

	/** キャラクタセット */
	private String charset;

	/**
	 * コンストラクタ<br>
	 * @param charset キャラクタセット
	 */
	public HttpSocketRequestPrameter(String charset) {
		super();
		this.parameter = new HashMap();
		this.charset = charset;
	}

	/**
	 * コンストラクタ<br>
	 */
	private HttpSocketRequestPrameter() {
		this(null);
	}

	/**
	 * 指定されたクエリ文字列からリクエストクエリパラメータオブジェクトを生成します。<br>
	 * @param charset キャラクタセット
	 * @param query クエリ文字列
	 * @return リクエストクエリパラメータオブジェクト
	 */
	public static HttpSocketRequestPrameter createRequestParameter(String charset, String query) {
		HttpSocketRequestPrameter prameter = new HttpSocketRequestPrameter(charset);
		if (!StringUtil.isEmpty(query)) {
			String[] tokens = query.split("&");
			for (int i = 0; i <= tokens.length - 1; i++) {
				String token = tokens[i];
				String name = token.indexOf("=") >= 0 ? token.substring(0, token.indexOf("=")) : "";
				String value = token.indexOf("=") >= 0 ? token.substring(token.indexOf("=") + 1) : "";
				prameter.put(name, value);
			}
		}
		return prameter;
	}

	/**
	 * 指定された入力ストリームからリクエストクエリパラメータオブジェクトを生成します。<br>
	 * @param charset キャラクタセット
	 * @param stream 入力ストリーム
	 * @param length データ長
	 * @return リクエストクエリパラメータオブジェクト
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @throws UnsupportedEncodingException 指定されているキャラクタセットが不正な場合にスローされます
	 */
	public static HttpSocketRequestPrameter createRequestParameter(String charset, InputStream stream, int length) throws UnsupportedEncodingException, IOException {
		return createRequestParameter(charset, new String(StreamUtil.reads(stream, length), charset));
	}

	/**
	 * 指定されたリクエストからリクエストパラメータオブジェクトを生成します。<br>
	 * 尚、リクエストメソッドがPOSTの場合、当メソッド実行によりストリームが提供するパラメータを併せて読み込みます。<br>
	 * 即ち、当メソッド実行後のリクエストからのストリーム取得は行えません。<br>
	 * @param charset キャラクタセット
	 * @param request リクエストオブジェクト
	 * @return リクエストクエリパラメータオブジェクト
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @throws UnsupportedEncodingException 指定されているキャラクタセットが不正な場合にスローされます
	 */
	public static HttpSocketRequestPrameter createRequestParameter(String charset, HttpSocketRequest request) throws UnsupportedEncodingException, IOException {
		HttpSocketRequestPrameter parameter = request.getQueryParameter(charset);
		if ("POST".equalsIgnoreCase(request.getMethod())) {
			HttpSocketRequestPrameter streamParameter = createRequestParameter(charset, //
					new String(StreamUtil.reads(request.getContentStream(), request.getContentLength()), charset));
			parameter.putAll(streamParameter);
		}
		return parameter;
	}

	/**
	 * 指定されたパラメータ内容を自身のパラメータに全て設定します。<br>
	 * @param parameter パラメータオブジェクト
	 */
	public void putAll(HttpSocketRequestPrameter parameter) {
		this.parameter.putAll(parameter.parameter);
	}

	/**
	 * パラメータを設定します。<br>
	 * @param name パラメータキー
	 * @param value パラメータ値
	 */
	public void put(String name, String value) {
		parameter.put(name, value);
	}

	/**
	 * パラメータを取得します。<br>
	 * @param name パラメータキー
	 * @return パラメータ値
	 * @throws UnsupportedEncodingException 指定されているキャラクタセットが不正な場合にスローされます
	 */
	public String get(String name) throws UnsupportedEncodingException {
		String value = (String) parameter.get(name);
		return value == null ? "" : URLDecoder.decode(value, charset);
	}

	/**
	 * パラメータを削除します。<br>
	 * @param name パラメータキー
	 */
	public void remove(String name) {
		parameter.remove(name);
	}

	/**
	 * パラメータ数を取得します。<br>
	 * @return パラメータ数
	 */
	public int size() {
		return parameter.size();
	}

	/**
	 * パラメータをクリアします。<br>
	 */
	public void clear() {
		parameter.clear();
	}

	/**
	 * パラメータキーセットを取得します。<br>
	 * @return パラメータキーセット
	 */
	public Set keySet() {
		return parameter.keySet();
	}

	/**
	 * パラメータからクエリ文字列を生成します。<br>
	 * @return クエリ文字列
	 * @throws UnsupportedEncodingException 指定されているキャラクタセットが不正な場合にスローされます
	 */
	public String createQueryString() throws UnsupportedEncodingException {
		StringBuffer buffer = new StringBuffer();
		for (Iterator iterator = parameter.keySet().iterator(); iterator.hasNext();) {
			String name = (String) iterator.next();
			if (!StringUtil.isEmpty(name)) {
				buffer.append(name);
				buffer.append("=");
				buffer.append(get(name));
				if (iterator.hasNext()) {
					buffer.append("&");
				}
			}
		}
		return buffer.toString();
	}
}
