package org.phosphoresce.commons.socket.http.container;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;

import org.phosphoresce.commons.socket.core.SocketGlobal;
import org.phosphoresce.commons.util.StringUtil;

/**
 * HTTPソケットリクエスト情報保持クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/11/14	Kitagawa		新規作成
 *-->
 */
public class HttpSocketRequest extends HttpSocketHeaderContainer {

	/** メソッド */
	private String method;

	/** パス */
	private String path;

	/** プロトコル */
	private String protocol;

	/** プロトコルバージョン */
	private String version;

	/** コンテンツストリーム */
	private InputStream contentStream;

	// Constructor

	/**
	 * コンストラクタ<br>
	 * @param method メソッド
	 * @param path パス
	 * @param protocol プロトコル
	 * @param version プロトコルバージョン
	 */
	public HttpSocketRequest(String method, String path, String protocol, String version) {
		super(true);
		this.method = method;
		this.path = path;
		this.protocol = protocol;
		this.version = version;
		this.contentStream = null;
	}

	/**
	 * コンストラクタ<br>
	 */
	public HttpSocketRequest() {
		//this(null, null, null, null);
		this("GET", "/", "HTTP", "1.1");
	}

	// Accessor Method

	/**
	 * メソッドを取得します。<br>
	 * @return メソッド
	 */
	public String getMethod() {
		return method == null ? "" : method;
	}

	/**
	 * メソッドを設定します。<br>
	 * @param method メソッド
	 */
	public void setMethod(String method) {
		this.method = method;
	}

	/**
	 * パスを取得します。<br>
	 * @return パス
	 */
	public String getPath() {
		return path == null ? "" : path;
	}

	/**
	 * パスを設定します。<br>
	 * @param path パス
	 */
	public void setPath(String path) {
		this.path = path;
	}

	/**
	 * プロトコルを取得します。<br>
	 * @return プロトコル
	 */
	public String getProtocol() {
		return protocol == null ? "" : protocol;
	}

	/**
	 * プロトコルを設定します。<br>
	 * @param protocol プロトコル
	 */
	public void setProtocol(String protocol) {
		this.protocol = protocol;
	}

	/**
	 * プロトコルバージョンを取得します。<br>
	 * @return プロトコルバージョン
	 */
	public String getVersion() {
		return version == null ? "" : version;
	}

	/**
	 * プロトコルバージョンを設定します。<br>
	 * @param version プロトコルバージョン
	 */
	public void setVersion(String version) {
		this.version = version;
	}

	/**
	 * コンテンツストリームを取得します。<br>
	 * @return コンテンツストリーム
	 */
	public InputStream getContentStream() {
		return contentStream;
	}

	/**
	 * コンテンツストリームを設定します。<br>
	 * @param contentStream コンテンツストリーム
	 */
	public void setContentStream(InputStream contentStream) {
		this.contentStream = contentStream;
	}

	/**
	 * コンテンツデータをテキストとして取得します。<br>
	 * このメソッドを使用することで入力ストリームから提供されるデータが全て読み込まれることに留意してください。<br>
	 * また、ストリームから提供されるデータを全て読み込むため、必ずテキストであるコンテンツであることを
	 * 判定した上で使用する必要があります。膨大なストリームデータが提供される場合、OutOfMemoryとなる可能性があります。<br>
	 * @param charset コンテンツ文字コード
	 * @param bufferSize ストリームバッファサイズ
	 * @return テキストとしてのコンテンツデータ
	 * @throws IOException 入力ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public String getContentText(String charset, int bufferSize) throws IOException {
		ByteArrayOutputStream bufferStream = new ByteArrayOutputStream();
		if (contentStream != null) {
			while (contentStream.available() > 0) {
				int readSize = contentStream.available() > bufferSize ? bufferSize : contentStream.available();
				byte[] bytes = new byte[readSize];
				contentStream.read(bytes);
				bufferStream.write(bytes);
			}
		}
		return charset == null ? new String(bufferStream.toByteArray()) : new String(bufferStream.toByteArray(), charset);
	}

	/**
	 * コンテンツデータをテキストとして取得します。<br>
	 * このメソッドを使用することで入力ストリームから提供されるデータが全て読み込まれることに留意してください。<br>
	 * また、ストリームから提供されるデータを全て読み込むため、必ずテキストであるコンテンツであることを
	 * 判定した上で使用する必要があります。膨大なストリームデータが提供される場合、OutOfMemoryとなる可能性があります。<br>
	 * @param bufferSize ストリームバッファサイズ
	 * @return テキストとしてのコンテンツデータ
	 * @throws IOException 入力ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public String getContentText(int bufferSize) throws IOException {
		return getContentText(null, bufferSize);
	}

	/**
	 * コンテンツデータをテキストとして取得します。<br>
	 * このメソッドを使用することで入力ストリームから提供されるデータが全て読み込まれることに留意してください。<br>
	 * また、ストリームから提供されるデータを全て読み込むため、必ずテキストであるコンテンツであることを
	 * 判定した上で使用する必要があります。膨大なストリームデータが提供される場合、OutOfMemoryとなる可能性があります。<br>
	 * @return テキストとしてのコンテンツデータ
	 * @throws IOException 入力ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public String getContentText() throws IOException {
		return getContentText(null, SocketGlobal.DEFAULT_STREAM_BUFFER_SIZE);
	}

	// Class Method

	/**
	 * リクエスト識別行を取得します。<br>
	 * @return リクエスト識別行
	 */
	public String getIdentifier() {
		StringBuffer buffer = new StringBuffer();
		if (!StringUtil.isEmpty(getMethod())) {
			buffer.append(getMethod());
			buffer.append(" ");
		}
		if (!StringUtil.isEmpty(getPath())) {
			buffer.append(getPath());
			buffer.append(" ");
		}
		if (!StringUtil.isEmpty(getProtocol())) {
			buffer.append(getProtocol());
			if (!StringUtil.isEmpty(getVersion())) {
				buffer.append("/");
				buffer.append(getVersion());
			}
		}
		return buffer.toString().trim();
	}

	/**
	 * 指定されたリクエスト識別行をクラスに対して設定します。<br>
	 * @param identifier リクエスト識別行
	 */
	public void setIdentifier(String identifier) {
		setMethod(getMethodByIdentifier(identifier));
		setPath(getPathByIdentifier(identifier));
		setProtocol(getProtocolByIdentifier(identifier));
		setVersion(getVersionByIdentifier(identifier));
	}

	/**
	 * 指定されたヘッダからリクエストメソッドを取得します。<br>
	 * @param header ヘッダ文字列
	 * @return リクエストメソッド
	 */
	private String getMethodByIdentifier(String header) {
		if (header == null || header.length() == 0) {
			return null;
		}
		String[] values = header.split(" ", 3);
		if (values.length < 1) {
			return null;
		}
		return values[0];
	}

	/**
	 * 指定されたヘッダからリクエストパスを取得します。<br>
	 * @param header ヘッダ文字列
	 * @return リクエストパス
	 */
	private String getPathByIdentifier(String header) {
		if (header == null || header.length() == 0) {
			return null;
		}
		String[] values = header.split(" ", 3);
		if (values.length < 2) {
			return null;
		}
		return values[1];
	}

	/**
	 * 指定されたヘッダからリクエストプロトコルを取得します。<br>
	 * @param header ヘッダ文字列
	 * @return リクエストプロトコル
	 */
	private String getProtocolByIdentifier(String header) {
		if (header == null || header.length() == 0) {
			return null;
		}
		String[] values = header.split(" ", 3);
		if (values.length < 3) {
			return null;
		}
		String[] buffers = values[2].split("/", 2);
		if (buffers.length < 1) {
			return null;
		}
		return buffers[0];
	}

	/**
	 * 指定されたヘッダからリクエストプロトコルバージョンを取得します。<br>
	 * @param header ヘッダ文字列
	 * @return リクエストプロトコルバージョン
	 */
	private String getVersionByIdentifier(String header) {
		if (header == null || header.length() == 0) {
			return null;
		}
		String[] values = header.split(" ", 3);
		if (values.length < 3) {
			return null;
		}
		String[] buffers = values[2].split("/", 2);
		if (buffers.length < 2) {
			return null;
		}
		return buffers[1];
	}

	/**
	 * リクエストコンテキストパスを取得します。<br>
	 * @return リクエストコンテキストパス
	 */
	public String getContextPath() {
		if (StringUtil.isEmpty(path)) {
			return "/";
		}
		String buffer = path.startsWith("/") ? path : "/" + path;
		return buffer.substring(1).indexOf("/") >= 0 ? buffer.substring(0, buffer.substring(1).indexOf("/") + 1) : buffer;
	}

	/**
	 * リクエストされたリソースパスを取得します。<br>
	 * ここで提供されるパス文字列はコンテキストパスを含まない、それ以降のパス文字列となります。<br>
	 * @return リソースパス
	 */
	public String getResourcePath() {
		if (StringUtil.isEmpty(path)) {
			return "";
		}
		return path.substring(getContextPath().length());
	}

	/**
	 * リクエストクエリパラメータを取得します。<br>
	 * ここで提供するパラメータはリクエストパスに記述されているクエリパラメータのみを提供します。<br>
	 * POSTで送信されたリクエストパラメータについてはここでは提供されません。<br>
	 * @param charset キャラクタセット
	 * @return リクエストクエリパラメータマップ
	 */
	public HttpSocketRequestPrameter getQueryParameter(String charset) {
		return HttpSocketRequestPrameter.createRequestParameter(charset, path.substring(path.indexOf("?") + 1));
	}

	/**
	 * オブジェクトの文字列表現を返します。<br>
	 * @return オブジェクトの文字列表現
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("{");
		buffer.append("method=");
		buffer.append(getMethod());
		buffer.append(",");
		buffer.append("path=");
		buffer.append(getPath());
		buffer.append(",");
		buffer.append("protocol=");
		buffer.append(getProtocol());
		buffer.append(",");
		buffer.append("version=");
		buffer.append(getVersion());
		buffer.append(",");
		buffer.append("headers=");
		buffer.append(getHeaders());
		buffer.append(",");
		buffer.append("cookies=");
		buffer.append(getCookies());
		buffer.append("}");
		return buffer.toString();
	}
}
