package org.phosphoresce.commons.socket.http.container;

import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;

import org.phosphoresce.commons.socket.http.HttpSocketGlobal;
import org.phosphoresce.commons.util.StringUtil;

/**
 * HTTPソケットヘッダコンテナ情報保持クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/11/14	Kitagawa		新規作成
 *-->
 */
abstract class HttpSocketHeaderContainer implements Serializable {

	/** リクエストクッキーヘッダキー */
	protected static final String HEADER_KEY_REQUEST_COOKIE = "Cookie";

	/** レスポンスクッキーヘッダキー */
	protected static final String HEADER_KEY_RESPONSE_COOKIE = "Set-Cookie";

	/** リクエスト用コンテナフラグ */
	private boolean requestContainer;

	/** ヘッダ情報 */
	private List headers;

	/** クッキー情報 */
	private List cookies;

	// Constructor

	/**
	 * コンストラクタ<br>
	 * @param requestContainer リクエスト用コンテナフラグ
	 */
	public HttpSocketHeaderContainer(boolean requestContainer) {
		super();
		this.headers = new LinkedList();
		this.cookies = new LinkedList();
		this.requestContainer = requestContainer;
	}

	/**
	 * コンストラクタ<br>
	 */
	private HttpSocketHeaderContainer() {
		this(true);
	}

	// Field Accessor Method

	/**
	 * ヘッダ情報を取得します。<br>
	 * @return ヘッダ情報
	 */
	protected List getHeaders() {
		return headers;
	}

	/**
	 * ヘッダ情報を設定します。<br>
	 * @param headers ヘッダ情報
	 */
	protected void setHeaders(List headers) {
		this.headers = headers;
	}

	/**
	 * クッキー情報を取得します。<br>
	 * @return cookies
	 */
	protected List getCookies() {
		return cookies;
	}

	/**
	 * クッキー情報を設定します。<br>
	 * @param cookies クッキー情報
	 */
	protected void setCookies(List cookies) {
		this.cookies = cookies;
	}

	// Class Private Method

	/**
	 * ヘッダコンテナがクッキーを保持する際のキーを取得します。<br>
	 * @return ヘッダコンテナがクッキーを保持する際のキー
	 */
	private String getHeaderCookieKey() {
		return requestContainer ? HEADER_KEY_REQUEST_COOKIE : HEADER_KEY_RESPONSE_COOKIE;
	}

	/**
	 * 指定されたヘッダ文字列のヘッダキーを取得します。<br>
	 * @param headerString ヘッダ文字列
	 * @return ヘッダキー
	 */
	private String getHeaderKeyOfHeaderString(String headerString) {
		if (StringUtil.isEmpty(headerString)) {
			return "";
		}
		String[] tokens = headerString.split(":", 2);
		return tokens.length > 0 ? tokens[0].trim() : "";
	}

	/**
	 * 指定されたヘッダ文字列のヘッダ値を取得します。<br>
	 * @param headerString ヘッダ文字列
	 * @return ヘッダ値
	 */
	private String getHeaderValueOfHeaderString(String headerString) {
		if (StringUtil.isEmpty(headerString)) {
			return "";
		}
		String[] tokens = headerString.split(":", 2);
		return tokens.length > 1 ? tokens[1].trim() : "";
	}

	/**
	 * 指定されたヘッダ文字列がクッキー情報であるか判定します。<br>
	 * @param headerString ヘッダ文字列
	 * @return クッキー情報である場合にtrueを返却
	 */
	private boolean isCookieHeader(String headerString) {
		return getHeaderCookieKey().equals(getHeaderKeyOfHeaderString(headerString));
	}

	/**
	 * 指定された単一のクッキー文字列のクッキーキーを取得します。<br>
	 * @param cookieString クッキー文字列
	 * @return クッキーキー
	 */
	private String getCookieKeyOfCookieString(String cookieString) {
		if (StringUtil.isEmpty(cookieString)) {
			return "";
		}
		String[] tokens = cookieString.split("=", 2);
		return tokens.length > 0 ? tokens[0].trim() : "";
	}

	/**
	 * 指定された単一クッキー文字列のクッキー値を取得します。<br>
	 * @param cookieString クッキー文字列
	 * @return クッキー値
	 */
	private String getCookieValueOfCookieString(String cookieString) {
		if (StringUtil.isEmpty(cookieString)) {
			return "";
		}
		String[] tokens = cookieString.split("=", 2);
		return tokens.length > 1 ? tokens[1].trim() : "";
	}

	// Header Accsess Method

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 */
	public void addHeader(String name, String value) {
		if (getHeaderCookieKey().equals(name)) {
			addCookie(value);
		} else {
			HttpSocketHeader header = new HttpSocketHeader();
			header.setName(name);
			header.setValue(value);
			headers.add(header);
		}
	}

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 */
	public void addHeader(String name, int value) {
		addHeader(name, String.valueOf(value));
	}

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 */
	public void addHeader(String name, long value) {
		addHeader(name, String.valueOf(value));
	}

	/**
	 * 指定されたヘッダ文字列からヘッダ情報を追加します。<br>
	 * ここにクッキー情報を保持するヘッダが指定された場合は、
	 * ヘッダ情報としてではなくクッキー情報として追加されます。<br>
	 * @param headerString ヘッダ文字列
	 */
	public void addHeader(String headerString) {
		if (StringUtil.isEmpty(headerString)) {
			return;
		}
		addHeader(getHeaderKeyOfHeaderString(headerString), getHeaderValueOfHeaderString(headerString));
	}

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * また、既に同一のヘッダキーでヘッダ情報が存在する場合、
	 * 既存のヘッダ情報は削除され新たにヘッダ情報が追加されます。<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 */
	public void putHeader(String name, String value) {
		removeHeader(name);
		addHeader(name, value);
	}

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * また、既に同一のヘッダキーでヘッダ情報が存在する場合、
	 * 既存のヘッダ情報は削除され新たにヘッダ情報が追加されます。<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 */
	public void putHeader(String name, int value) {
		removeHeader(name);
		addHeader(name, value);
	}

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * また、既に同一のヘッダキーでヘッダ情報が存在する場合、
	 * 既存のヘッダ情報は削除され新たにヘッダ情報が追加されます。<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 */
	public void putHeader(String name, long value) {
		removeHeader(name);
		addHeader(name, value);
	}

	/**
	 * 指定されたキーと値でヘッダ情報を追加します。<br>
	 * 但し、キーにクッキー用ヘッダキーが指定された場合は、ヘッダ情報
	 * としてではなく、値を元にクッキー情報として追加されます。<br>
	 * また、既に同一のヘッダキーでヘッダ情報が存在する場合、
	 * 既存のヘッダ情報は削除され新たにヘッダ情報が追加されます。<br>
	 * @param headerString ヘッダ文字列
	 */
	public void putHeader(String headerString) {
		if (StringUtil.isEmpty(headerString)) {
			return;
		}
		removeHeader(getHeaderKeyOfHeaderString(headerString));
		addHeader(headerString);
	}

	/**
	 * 指定されたヘッダキーのヘッダ情報を削除します。<br>
	 * 同一キーで複数のヘッダ情報が存在する場合はそれら全てが削除されます。<br>
	 * @param name ヘッダキー
	 */
	public void removeHeader(String name) {
		while (containsHeader(name)) {
			HttpSocketHeader header = getHeader(name);
			headers.remove(header);
		}
	}

	/**
	 * 指定されたキーのヘッダ情報が保持されているか判定します。<br>
	 * @param name ヘッダキー
	 * @return ヘッダ情報が保持されている場合にtrueを返却
	 */
	public boolean containsHeader(String name) {
		return getHeader(name) != null;
	}

	/**
	 * 指定されたヘッダキーのヘッダ情報を取得します。<br>
	 * @param name ヘッダキー
	 * @return ヘッダ情報
	 */
	public HttpSocketHeader getHeader(String name) {
		for (Iterator iterator = headers.iterator(); iterator.hasNext();) {
			HttpSocketHeader header = (HttpSocketHeader) iterator.next();
			if (header.getName().equalsIgnoreCase(name)) {
				return header;
			}
		}
		return null;
	}

	/**
	 * 指定されたヘッダキーのヘッダ値を取得します。<br>
	 * @param name ヘッダキー
	 * @return ヘッダ値
	 */
	public String getHeaderValueOfString(String name) {
		HttpSocketHeader header = getHeader(name);
		return header == null ? "" : header.getValue();
	}

	/**
	 * 指定されたヘッダキーのヘッダ値を取得します。<br>
	 * @param name ヘッダキー
	 * @return ヘッダ値
	 */
	public int getHeaderValueOfInteger(String name) {
		HttpSocketHeader header = getHeader(name);
		return header == null ? -1 : header.getValueOfInteger();
	}

	/**
	 * 指定されたヘッダキーのヘッダ値を取得します。<br>
	 * @param name ヘッダキー
	 * @return ヘッダ値
	 */
	public long getHeaderValueOfLong(String name) {
		HttpSocketHeader header = getHeader(name);
		return header == null ? -1 : header.getValueOfLong();
	}

	/**
	 * ヘッダ文字列配列を取得します。<br>
	 * @return ヘッダ文字列配列
	 */
	public String[] getHeaderStrings() {
		List list = new LinkedList();
		for (Iterator iterator = headers.iterator(); iterator.hasNext();) {
			HttpSocketHeader header = (HttpSocketHeader) iterator.next();
			list.add(header.getHeaderString());
		}
		return (String[]) list.toArray(new String[list.size()]);
	}

	// Cookie Access Method

	/**
	 * 指定されたクッキー文字列からクッキー情報を追加します。<br>
	 * @param cookieString クッキー文字列
	 */
	public void addCookie(String cookieString) {
		if (StringUtil.isEmpty(cookieString)) {
			return;
		}
		if (requestContainer) {
			String[] tokens = cookieString.split(";");
			for (int i = 0; i <= tokens.length - 1; i++) {
				String token = tokens[i].trim();
				HttpSocketCookie cookie = new HttpSocketCookie();
				cookie.setCookieString(token);
				cookies.add(cookie);
			}
		} else {
			HttpSocketCookie cookie = new HttpSocketCookie();
			cookie.setCookieString(cookieString);
			cookies.add(cookie);
		}
	}

	/**
	 * 指定された情報のクッキーを追加します。<br>
	 * @param name クッキー名
	 * @param value クッキー値
	 * @param expires 有効期限
	 * @param path パス
	 * @param domain ドメイン
	 * @param secure セキュアフラグ
	 */
	public void addCookie(String name, String value, Date expires, String path, String domain, boolean secure) {
		StringBuffer buffer = new StringBuffer();
		buffer.append(name == null ? "" : name);
		buffer.append("=");
		buffer.append(value == null ? "" : value);
		if (expires != null) {
			buffer.append("; ");
			buffer.append("expires=");
			buffer.append(new SimpleDateFormat(HttpSocketGlobal.HTTP_HEADER_DATE_FORMAT, Locale.ENGLISH).format(expires));
		}
		if (path != null) {
			buffer.append("; ");
			buffer.append("path=");
			buffer.append(path);
		}
		if (domain != null) {
			buffer.append("; ");
			buffer.append("domain=");
			buffer.append(domain);
		}
		if (secure) {
			buffer.append("; ");
			buffer.append("secure");
		}
		addCookie(buffer.toString());
	}

	/**
	 * 指定された情報のクッキーを追加します。<br>
	 * 当メソッドで追加したクッキーの有効期限はブラウザ終了時までとなります。<br>
	 * @param name クッキー名
	 * @param value クッキー値
	 */
	public void addCookie(String name, String value) {
		addCookie(name, value, null, null, null, false);
	}

	/**
	 * 指定されたクッキー文字列からクッキー情報を追加します。<br>
	 * 既に同一のクッキーキーでクッキー情報が存在する場合、
	 * 既存のクッキー情報は削除され新たにクッキー情報が追加されます。<br>
	 * @param cookieString クッキー文字列
	 */
	public void putCookie(String cookieString) {
		if (StringUtil.isEmpty(cookieString)) {
			return;
		}
		if (requestContainer) {
			String[] tokens = cookieString.split(";");
			for (int i = 0; i <= tokens.length - 1; i++) {
				String token = tokens[i].trim();
				removeCookie(getCookieKeyOfCookieString(token));
				HttpSocketCookie cookie = new HttpSocketCookie();
				cookie.setCookieString(token);
				cookies.add(cookie);
			}
		} else {
			removeCookie(getCookieKeyOfCookieString(cookieString));
			HttpSocketCookie cookie = new HttpSocketCookie();
			cookie.setCookieString(cookieString);
			cookies.add(cookie);
		}
	}

	/**
	 * 指定された情報のクッキーを追加します。<br>
	 * @param name クッキー名
	 * @param value クッキー値
	 * @param expires 有効期限
	 * @param path パス
	 * @param domain ドメイン
	 * @param secure セキュアフラグ
	 */
	public void putCookie(String name, String value, Date expires, String path, String domain, boolean secure) {
		StringBuffer buffer = new StringBuffer();
		buffer.append(name == null ? "" : value);
		buffer.append("=");
		buffer.append(value == null ? "" : value);
		if (expires != null) {
			buffer.append("; ");
			buffer.append("expires=");
			buffer.append(new SimpleDateFormat(HttpSocketGlobal.HTTP_HEADER_DATE_FORMAT, Locale.ENGLISH).format(expires));
		}
		if (path != null) {
			buffer.append("; ");
			buffer.append("path=");
			buffer.append(path);
		}
		if (domain != null) {
			buffer.append("; ");
			buffer.append("domain=");
			buffer.append(domain);
		}
		if (secure) {
			buffer.append("; ");
			buffer.append("secure");
		}
		putCookie(buffer.toString());
	}

	/**
	 * 指定された情報のクッキーを追加します。<br>
	 * 当メソッドで追加したクッキーの有効期限はブラウザ終了時までとなります。<br>
	 * @param name クッキー名
	 * @param value クッキー値
	 */
	public void putCookie(String name, String value) {
		putCookie(name, value, null, null, null, false);
	}

	/**
	 * 指定されたクッキーキーのクッキー情報を削除します。<br>
	 * 同一キーで複数のクッキー情報が存在する場合はそれら全てが削除されます。<br>
	 * @param name クッキーキー
	 */
	public void removeCookie(String name) {
		//		while (containsCookie(name)) {
		//			HttpSocketCookie cookie = getCookie(name);
		//			cookies.remove(cookie);
		//		}
		HttpSocketCookie[] cookies = getCookies(name);
		for (int i = 0; i <= cookies.length - 1; i++) {
			this.cookies.remove(cookies[i]);
		}
	}

	/**
	 * 指定されたキーのクッキー情報が保持されているか判定します。<br>
	 * @param name クッキーキー
	 * @return クッキー情報が保持されている場合にtrueを返却
	 */
	public boolean containsCookie(String name) {
		//		return getCookie(name) != null;
		return getCookieCountOf(name) > 0;
	}

	/**
	 * 指定されたクッキーキーで保持されているクッキー情報数を取得します。<br>
	 * @param name クッキーキー
	 * @return 保持されているクッキー情報数
	 */
	public int getCookieCountOf(String name) {
		return getCookies(name).length;
	}

	/**
	 * 指定されたクッキーキーのクッキー情報を取得します。<br>
	 * @param name クッキーキー
	 * @return クッキー情報
	 */
	public HttpSocketCookie[] getCookies(String name) {
		List list = new LinkedList();
		for (Iterator iterator = cookies.iterator(); iterator.hasNext();) {
			HttpSocketCookie cookie = (HttpSocketCookie) iterator.next();
			if (cookie.getName().equalsIgnoreCase(name)) {
				list.add(cookie);
			}
		}
		return (HttpSocketCookie[]) list.toArray(new HttpSocketCookie[list.size()]);
	}

	/**
	 * 指定されたクッキーキーのクッキー情報を取得します。<br>
	 * 同一クッキーキーで複数のクッキー情報が保持されている場合、末尾の1件が返却されます。<br>
	 * @param name クッキーキー
	 * @return クッキー情報
	 */
	public HttpSocketCookie getCookie(String name) {
		HttpSocketCookie[] cookies = getCookies(name);
		return cookies.length <= 0 ? null : cookies[cookies.length - 1];
	}

	/**
	 * 指定されたクッキーキーのクッキー値を取得します。<br>
	 * 同一クッキーキーで複数のクッキー情報が保持されている場合、末尾の1件が返却されます。<br>
	 * @param name クッキーキー
	 * @return クッキー値
	 */
	public String getCookieValueOfString(String name) {
		HttpSocketCookie cookie = getCookie(name);
		return cookie == null ? "" : cookie.getValue();
	}

	/**
	 * 指定されたクッキーキーのクッキー値を取得します。<br>
	 * 同一クッキーキーで複数のクッキー情報が保持されている場合、末尾の1件が返却されます。<br>
	 * @param name クッキーキー
	 * @return クッキー値
	 */
	public int getCookieValueOfInteger(String name) {
		HttpSocketCookie cookie = getCookie(name);
		return cookie == null ? -1 : cookie.getValueOfInteger();
	}

	/**
	 * 指定されたクッキーキーのクッキー値を取得します。<br>
	 * 同一クッキーキーで複数のクッキー情報が保持されている場合、末尾の1件が返却されます。<br>
	 * @param name クッキーキー
	 * @return クッキー値
	 */
	public long getCookieValueOfLong(String name) {
		HttpSocketCookie cookie = getCookie(name);
		return cookie == null ? -1 : cookie.getValueOfLong();
	}

	/**
	 * クッキー文字列配列を取得します。<br>
	 * @return クッキー文字列配列
	 */
	public String[] getCookieStrings() {
		List list = new LinkedList();
		if (cookies.size() == 0) {
			return new String[] {};
		}
		if (requestContainer) {
			StringBuffer cookieBuffer = new StringBuffer();
			for (Iterator iterator = cookies.iterator(); iterator.hasNext();) {
				HttpSocketCookie cookie = (HttpSocketCookie) iterator.next();
				cookieBuffer.append(cookie.getCookieString());
				cookieBuffer.append("; ");
			}
			String cookieString = cookieBuffer.toString().trim();
			cookieString = cookieString.endsWith(";") ? cookieString.substring(0, cookieString.length() - 1) : cookieString;
			list.add(getHeaderCookieKey() + ": " + cookieString);
		} else {
			for (Iterator iterator = cookies.iterator(); iterator.hasNext();) {
				HttpSocketCookie cookie = (HttpSocketCookie) iterator.next();
				list.add(getHeaderCookieKey() + ": " + cookie.getCookieString());
			}
		}
		return (String[]) list.toArray(new String[list.size()]);
	}

	// Useful Header Access Method

	/**
	 * コンテンツボディデータの長さを取得します。<br>
	 * ヘッダ情報よりデータ長を特定できない場合は-1が返却されます。<br>
	 * @return コンテンツボディデータの長さ
	 */
	public int getContentLength() {
		//		if (requestContainer) {
		//			return containsHeader("Content-Length") ? getHeaderValueOfInteger("Content-Length") : 0;
		//		} else {
		return getHeaderValueOfInteger("Content-Length");
		//		}
	}

	/**
	 * コンテンツボディデータの長さを設定します。<br>
	 * @param length コンテンツボディデータの長さ
	 */
	public void setContentLength(long length) {
		if (length < 0) {
			removeHeader("Content-Length");
		} else {
			addHeader("Content-Length", length);
		}
	}

	/**
	 * コンテンツボディデータがChunkedデータであるかを判定します。<br>
	 * @return コンテンツボディデータがChunkedデータである場合にtrueを返却
	 */
	public boolean isChunkedContent() {
		return "chunked".equalsIgnoreCase(getHeaderValueOfString("Transfer-Encoding"));
	}

	/**
	 * コネクションがKeep-Aliveであるか判定します。<br>
	 * @return Keep-Aliveである場合にtrueを返却
	 */
	public boolean isKeepAliveConnection() {
		return "keep-alive".equalsIgnoreCase(getHeaderValueOfString("Connection"));
	}
}
