package org.phosphoresce.commons.socket.http.container;

import org.phosphoresce.commons.util.StringUtil;

/**
 * HTTPソケットヘッダ情報保持クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/11/14	Kitagawa		新規作成
 *-->
 */
public class HttpSocketHeader {

	/** ヘッダキー */
	private String name;

	/** ヘッダ値 */
	private String value;

	/**
	 * コンストラクタ<br>
	 * @param name ヘッダキー
	 * @param value ヘッダ値
	 */
	protected HttpSocketHeader(String name, String value) {
		super();
		if (!isValidName(name)) {
			throw new IllegalArgumentException(name + " is invalid header name");
		}
		this.name = name;
		this.value = value;
	}

	/**
	 * コンストラクタ<br>
	 */
	protected HttpSocketHeader() {
		this(null, null);
	}

	/**
	 * ヘッダキーを取得します。<br>
	 * @return ヘッダキー
	 */
	public String getName() {
		return name == null ? "" : name;
	}

	/**
	 * ヘッダキーを設定します。<br>
	 * @param name ヘッダキー
	 */
	public void setName(String name) {
		if (!isValidName(name)) {
			throw new IllegalArgumentException(name + " is invalid header name");
		}
		this.name = name;
	}

	/**
	 * ヘッダ値を取得します。<br>
	 * @return ヘッダ値
	 */
	public String getValue() {
		return value == null ? "" : value;
	}

	/**
	 * ヘッダ値を取得します。<br>
	 * @return ヘッダ値
	 */
	public int getValueOfInteger() {
		return StringUtil.isEmpty(value) || !StringUtil.isNumeric(value) ? -1 : StringUtil.parsePrimitiveInt(value);
	}

	/**
	 * ヘッダ値を取得します。<br>
	 * @return ヘッダ値
	 */
	public long getValueOfLong() {
		return StringUtil.isEmpty(value) || !StringUtil.isNumeric(value) ? -1 : StringUtil.parsePrimitiveLong(value);
	}

	/**
	 * ヘッダ値を設定します。<br>
	 * @param value ヘッダ値
	 */
	public void setValue(String value) {
		this.value = value;
	}

	/**
	 * ヘッダ値を設定します。<br>
	 * @param value ヘッダ値
	 */
	public void setValue(int value) {
		this.value = String.valueOf(value);
	}

	/**
	 * ヘッダ文字列を取得します。<br>
	 * @return ヘッダ文字列
	 */
	public String getHeaderString() {
		StringBuffer buffer = new StringBuffer();
		if (!StringUtil.isEmpty(getName())) {
			buffer.append(getName());
			buffer.append(": ");
			buffer.append(getValue());
		}
		String result = buffer.toString().trim();
		return result;
	}

	/**
	 * ヘッダ文字列からヘッダ情報を設定します。<br>
	 * @param header ヘッダ文字列
	 */
	public void setHeaderString(String header) {
		name = null;
		value = null;
		if (header == null || header.length() == 0) {
			return;
		}
		String[] tokens = header == null ? new String[0] : header.split(":", 2);
		String name = null;
		String value = null;
		if (tokens.length >= 1) {
			name = tokens[0].trim();
		}
		if (tokens.length >= 2) {
			value = tokens[1].trim();
		}
		setName(name);
		setValue(value);
	}

	/**
	 * 指定されたヘッダキーが有効なものであるか判定します。<br>
	 * @param name ヘッダキー
	 * @return 有効な場合にtrueを返却
	 */
	private boolean isValidName(String name) {
		if (StringUtil.isEmpty(name)) {
			return true;
		} else {
			return true;
		}
	}

	/**
	 * オブジェクトの文字列表現を返します。<br>
	 * @return オブジェクトの文字列表現
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("{");
		buffer.append("name=");
		buffer.append(getName());
		buffer.append(",");
		buffer.append("value=");
		buffer.append(getValue());
		buffer.append("}");
		return buffer.toString();
	}
}
