package org.phosphoresce.commons.socket.http.container;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.phosphoresce.commons.util.StringUtil;

/**
 * HTTPソケットクッキー情報保持クラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/11/14	Kitagawa		新規作成
 *-->
 */
public class HttpSocketCookie implements Serializable, Cloneable {

	/** 予約済みクッキーキー */
	private static final String[] RESERVED_NAMES = new String[] { "Expires", "Path", "Domain", "Secure", "Version", "Comment", "Discard", "Max-Age" };

	/** クッキーキー */
	private String name;

	/** クッキー値 */
	private String value;

	/** クッキーオプション */
	private Map options;

	/**
	 * コンストラクタ<br>
	 * @param name クッキーキー
	 * @param value クッキー値
	 */
	protected HttpSocketCookie(String name, String value) {
		super();
		if (!isValidName(name)) {
			throw new IllegalArgumentException(name + " is invalid cookie name");
		}
		this.name = name;
		this.value = value;
		this.options = new HashMap();
	}

	/**
	 * コンストラクタ<br>
	 */
	protected HttpSocketCookie() {
		this(null, null);
	}

	/**
	 * クッキーキーを取得します。<br>
	 * @return クッキーキー
	 */
	public String getName() {
		return name == null ? "" : name;
	}

	/**
	 * クッキーキーを設定します。<br>
	 * @param name クッキーキー
	 */
	public void setName(String name) {
		if (!isValidName(name)) {
			throw new IllegalArgumentException(name + " is invalid cookie name");
		}
		this.name = name;
	}

	/**
	 * クッキー値を取得します。<br>
	 * @return クッキー値
	 */
	public String getValue() {
		return value == null ? "" : value;
	}

	/**
	 * クッキー値を取得します。<br>
	 * @return クッキー値
	 */
	public int getValueOfInteger() {
		return StringUtil.isEmpty(value) || !StringUtil.isNumeric(value) ? -1 : StringUtil.parsePrimitiveInt(value);
	}

	/**
	 * クッキー値を取得します。<br>
	 * @return クッキー値
	 */
	public long getValueOfLong() {
		return StringUtil.isEmpty(value) || !StringUtil.isNumeric(value) ? -1 : StringUtil.parsePrimitiveLong(value);
	}

	/**
	 * クッキー値を設定します。<br>
	 * @param value クッキー値
	 */
	public void setValue(String value) {
		this.value = value;
	}

	/**
	 * クッキー値を設定します。<br>
	 * @param value クッキー値
	 */
	public void setValue(int value) {
		this.value = String.valueOf(value);
	}

	/**
	 * 指定された大小文字区別なしのオプションキーのキャッシュされている
	 * 実際のキーを取得します。<br>
	 * @param name オプションキー(大小文字区別なし)
	 * @return キャッシュされている実際のキー
	 */
	private String getCachedOptionName(String name) {
		if (name == null) {
			return null;
		} else if ("".equals(name)) {
			return "";
		} else {
			for (Iterator iterator = options.keySet().iterator(); iterator.hasNext();) {
				String cachedName = (String) iterator.next();
				if (name.equalsIgnoreCase(cachedName)) {
					return cachedName;
				}
			}
		}
		throw new IllegalArgumentException(name + "is not exist cached options");
	}

	/**
	 * 指定されたオプションキーでオプションが保持されているか判定します。<br>
	 * @param name オプションキー
	 * @return オプションが保持されている場合にtrueを返却
	 */
	public boolean containsOption(String name) {
		for (Iterator iterator = options.keySet().iterator(); iterator.hasNext();) {
			String cachedName = (String) iterator.next();
			if (name.equalsIgnoreCase(cachedName)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * クッキーオプションを設定します。<br>
	 * @param name オプションキー(大小文字区別なし)
	 * @param value オプション値
	 */
	public void setOption(String name, String value) {
		if (containsOption(name)) {
			options.put(getCachedOptionName(name), value);
		} else {
			options.put(name, value);
		}
	}

	/**
	 * クッキーオプションを取得します。<br>
	 * @param name オプションキー(大小文字区別なし)
	 * @return オプション値
	 */
	public String getOption(String name) {
		if (containsOption(name)) {
			String value = (String) options.get(getCachedOptionName(name));
			return value == null ? "" : value;
		} else {
			return "";
		}
	}

	/**
	 * クッキー文字列を取得します。<br>
	 * @return クッキー文字列
	 */
	public String getCookieString() {
		StringBuffer buffer = new StringBuffer();
		if (!StringUtil.isEmpty(getName())) {
			buffer.append(getName());
			if (!StringUtil.isEmpty(getValue())) {
				buffer.append("=");
				buffer.append(getValue());
			}
			buffer.append("; ");
			for (Iterator iterator = options.keySet().iterator(); iterator.hasNext();) {
				String optionName = (String) iterator.next();
				String optionValue = getOption(optionName);
				buffer.append(optionName);
				if (!StringUtil.isEmpty(optionValue)) {
					buffer.append("=");
					buffer.append(optionValue);
					buffer.append("; ");
				}
			}
		}
		String result = buffer.toString().trim();
		if (result.endsWith(";")) {
			result = result.substring(0, result.length() - 1);
		}
		return result;
	}

	/**
	 * クッキー文字列からクッキー情報を設定します。<br>
	 * @param cookie クッキー文字列
	 */
	public void setCookieString(String cookie) {
		name = null;
		value = null;
		options.clear();
		if (cookie == null || cookie.length() == 0) {
			return;
		}
		String[] sets = cookie == null ? new String[0] : cookie.split(";");
		for (int i = 0; i <= sets.length - 1; i++) {
			String[] tokens = sets[i] == null ? new String[0] : sets[i].split("=", 2);
			String name = null;
			String value = null;
			if (tokens.length >= 1) {
				name = tokens[0].trim();
			}
			if (tokens.length >= 2) {
				value = tokens[1].trim();
			}
			if (i == 0) {
				setName(name);
				setValue(value);
				continue;
			} else {
				setOption(name, value);
			}
		}
	}

	/**
	 * 指定されたクッキーキーが有効なものであるか判定します。<br>
	 * @param name クッキーキー
	 * @return 有効な場合にtrueを返却
	 */
	private boolean isValidName(String name) {
		if (StringUtil.isEmpty(name)) {
			return true;
		} else if (StringUtil.containsIgnoreCase(RESERVED_NAMES, name)) {
			return false;
		} else if (name.startsWith("$")) {
			return false;
		} else {
			return true;
		}
	}

	/**
	 * オブジェクトの文字列表現を返します。<br>
	 * @return オブジェクトの文字列表現
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("{");
		buffer.append("name=");
		buffer.append(getName());
		buffer.append(",");
		buffer.append("value=");
		buffer.append(getValue());
		buffer.append(",");
		buffer.append("options=");
		buffer.append(options.toString());
		buffer.append("}");
		return buffer.toString();
	}
}
