package org.phosphoresce.commons.socket.http;

import java.io.Serializable;

/**
 * レスポンスステータス列挙型クラス<br>
 * <br>
 * 当クラスの等価比較条件はコードのみとなります。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/29	Kitagawa		新規作成
 *-->
 */
public final class HttpResponseStatus implements Serializable {

	/*
	 * 情報ステータス
	 */

	/** ステータス100 */
	public static final HttpResponseStatus STATUS_100_CONTINUE = new HttpResponseStatus( //
			100 //
			, "Continue" //
			, "" //
			, "処理を継続しています。続きのリクエストを送信してください。" //
	);

	/** ステータス101 */
	public static final HttpResponseStatus STATUS_101_SWITCHING_PROTOCOLS = new HttpResponseStatus( //
			101 //
			, "Switching Protocols" //
			, "" //
			, "Upgrade ヘッダで指定したプロトコルに変更して再要求してください。" //
	);

	/*
	 * 成功ステータス
	 */

	/** ステータス200 */
	public static final HttpResponseStatus STATUS_200_OK = new HttpResponseStatus( //
			200 //
			, "OK" //
			, "Action complete successfully." //
			, "成功しました。" //
	);

	/** ステータス201 */
	public static final HttpResponseStatus STATUS_201_CREATED = new HttpResponseStatus( //
			201 //
			, "Created" //
			, "Success following a POST command." //
			, "Location ヘッダで指定した場所に新しいコンテンツが作成されました。" //
	);

	/** ステータス202 */
	public static final HttpResponseStatus STATUS_202_ACCEPTED = new HttpResponseStatus( //
			202 //
			, "Accepted" //
			, "The request has been accepted for processing, but the processing has not been completed." //
			, "要求は受理されました。ただし処理は完了していません。" //
	);

	/** ステータス203 */
	public static final HttpResponseStatus STATUS_203_NON_AUTHORITATIVE_INFORMATION = new HttpResponseStatus( //
			203 //
			, "Non-Authoritative Information" //
			, "Response to a GET command, indicates that the returned meta information is from a private overlaid web." //
			, "応答ヘッダはオリジナルサーバーが返したものとは異なりますが、処理は成功です。" //
	);

	/** ステータス204 */
	public static final HttpResponseStatus STATUS_204_NO_CONTENT = new HttpResponseStatus( //
			204 //
			, "No Content" //
			, "Server has received the request but there is no information to send back." //
			, "コンテンツはありませんが、処理は成功しました。" //
	);

	/** ステータス205 */
	public static final HttpResponseStatus STATUS_205_RESET_CONTENT = new HttpResponseStatus( //
			205 //
			, "Reset Content" //
			, "Request fulfilled and user agent should reset document that caused the request. " //
					+ "Used to allow user input followed by reseting the input form. The response MUST NOT include an entity." //
			, "要求を受理したので、現在のコンテンツ（画面）を破棄してください。。" //
	);

	/** ステータス206 */
	public static final HttpResponseStatus STATUS_206_PARTIAL_CONTENT = new HttpResponseStatus( //
			206 //
			, "Partial Content" //
			, "The requested file was partially sent. Usually caused by stopping or refreshing a web page." //
			, "コンテンツを一部のみ返却します。" //
	);

	/*
	 * 転送ステータス
	 */

	/** ステータス300 */
	public static final HttpResponseStatus STATUS_300_MULTIPLE_CHOICES = new HttpResponseStatus( //
			300 //
			, "Multiple Choices" //
			, "Requested resource corresponds to a set of representations, each with its own location. " //
					+ "Negotiation information is provided so that the representation can be selected and redirect appropriately." //
			, "コンテンツ入手方法について複数の選択肢があります。" //
	);

	/** ステータス301 */
	public static final HttpResponseStatus STATUS_301_MOVED_PERMANENTLY = new HttpResponseStatus( //
			301 //
			, "Moved Permanently" //
			, "Requested a directory instead of a file. The web server substituted the index.htm file." //
			, "Location ヘッダで指定された別の場所に移動しました。" //
	);

	/** ステータス302 */
	public static final HttpResponseStatus STATUS_302_FOUND = new HttpResponseStatus( //
			302 //
			, "Found" //
			, "The requested resource resides temporarily under a different URI. " //
					+ "Since the redirection might be altered on occasion, the client SHOULD continue to use the Request-URI for future requests." //
			, "Location ヘッダで指定された別の場所に見つかりました。そちらを見てください。" //
	);

	/** ステータス303 */
	public static final HttpResponseStatus STATUS_303_SEE_OTHER = new HttpResponseStatus( //
			303 //
			, "See Other" //
			, "The response to the request is at a different URI and should be retrieved there using a GET method. " //
					+ "This allows the output of a POST-activated script to redirect the user agent to a selected resource." //
			, "Location ヘッダで指定された他の場所を見てください。" //
	);

	/** ステータス304 */
	public static final HttpResponseStatus STATUS_304_NOT_MODIFIED = new HttpResponseStatus( //
			304 //
			, "Not Modified" //
			, "The cached version of the requested file is the same as the file to be sent." //
			, "更新されていません。If-Modified-Since ヘッダを用いた場合に返却されます。" //
	);

	/** ステータス305 */
	public static final HttpResponseStatus STATUS_305_USE_PROXY = new HttpResponseStatus( //
			305 //
			, "Use Proxy" //
			, "The requested resource must be accessed through the proxy given in the Location field. " //
					+ "The recipient should repeat this request via the proxy. This response must only be generated by origin servers." //
			, "Location ヘッダで指定したプロキシを使用してください。" //
	);

	/** ステータス306 */
	public static final HttpResponseStatus STATUS_306_UNUSED = new HttpResponseStatus( //
			306 //
			, "(Unused)" //
			, "Reserved for future use." //
			, "未使用。" //
	);

	/** ステータス307 */
	public static final HttpResponseStatus STATUS_307_TEMPORARY_REDIRECT = new HttpResponseStatus( //
			307 //
			, "Temporary Redirect" //
			, "The requested resource resides temporarily under a different URI." //
			, "別の場所に一時的に移動しています。" //
	);

	/*
	 * クライアントエラーステータス
	 */

	/** ステータス400 */
	public static final HttpResponseStatus STATUS_400_BAD_REQUEST = new HttpResponseStatus( //
			400 //
			, "Bad Request" //
			, "The request could not be understood by the server due to malformed syntax. " //
					+ "The client SHOULD NOT repeat the request without modifications." //
			, "要求が不正です。" //
	);

	/** ステータス401 */
	public static final HttpResponseStatus STATUS_401_UNAUTHORIZED = new HttpResponseStatus( //
			401 //
			, "Unauthorized" //
			, "User fail to provide a valid user name / password required for access to file / directory." //
			, "認証されていません。" //
	);

	/** ステータス402 */
	public static final HttpResponseStatus STATUS_402_PAYMENT_REQUIRED = new HttpResponseStatus( //
			402 //
			, "Payment Required" //
			, "Reserved for future use." //
			, "支払いが必要です。" //
	);

	/** ステータス403 */
	public static final HttpResponseStatus STATUS_403_FORBIDDEN = new HttpResponseStatus( //
			403 //
			, "Forbidden" //
			, "The server understood the request, but is refusing to fulfill it. " //
					+ "Authorization will not help and the request SHOULD NOT be repeated." //
			, "アクセスが認められていません。" //
	);

	/** ステータス404 */
	public static final HttpResponseStatus STATUS_404_NOT_FOUND = new HttpResponseStatus( //
			404 //
			, "Not Found" //
			, "The server has not found anything matching the Request-URI. " //
					+ "No indication is given of whether the condition is temporary or permanent." //
			, "見つかりません。" //
	);

	/** ステータス405 */
	public static final HttpResponseStatus STATUS_405_METHOD_NOT_ALLOWED = new HttpResponseStatus( //
			405 //
			, "Method Not Allowed" //
			, "The method in the Request-Line is not allowed for the resource identified by the Request-URI. " //
					+ "The response must include an Allow header containing a list of valid methods for the requested resource." //
			, "指定したメソッドはサポートされていません。" //
	);

	/** ステータス406 */
	public static final HttpResponseStatus STATUS_406_NOT_ACCEPTABLE = new HttpResponseStatus( //
			406 //
			, "Not Acceptable" //
			, "The resource identified by the request is only capable of generating response entities " //
					+ "which have content characteristics not acceptable according to the accept headers sent in the request." //
			, "許可されていません。" //
	);

	/** ステータス407 */
	public static final HttpResponseStatus STATUS_407_PROXY_AUTHENTICATION_REQUIRED = new HttpResponseStatus( //
			407 //
			, "Proxy Authentication Required" //
			, "This code is similar to 401 (Unauthorized), but indicates that the client must first authenticate itself with the proxy." //
			, "プロキシ認証が必要です。" //
	);

	/** ステータス408 */
	public static final HttpResponseStatus STATUS_408_REQUEST_TIMEOUT = new HttpResponseStatus( //
			408 //
			, "Request Timeout" //
			, "The client did not produce a request within the time that the server was prepared to wait. " //
					+ "The client MAY repeat the request without modifications at any later time." //
			, "リクエストがタイムアウトしました。" //
	);

	/** ステータス409 */
	public static final HttpResponseStatus STATUS_409_CONFLICT = new HttpResponseStatus( //
			409 //
			, "Conflict" //
			, "The request could not be completed due to a conflict with the current state of the resource." //
			, "リクエストがコンフリクト（衝突・矛盾）しました。" //
	);

	/** ステータス410 */
	public static final HttpResponseStatus STATUS_410_GONE = new HttpResponseStatus( //
			410 //
			, "Gone" //
			, "The requested resource is no longer available at the server and no forwarding address is known." //
			, "要求されたコンテンツは無くなってしまいました。" //
	);

	/** ステータス411 */
	public static final HttpResponseStatus STATUS_411_LENGTH_REQUIRED = new HttpResponseStatus( //
			411 //
			, "Length Required" //
			, "The server refuses to accept the request without a defined Content-Length of the message-body. The client may repeat the request if it adds it." //
			, "Content-Length ヘッダを付加して要求してください。" //
	);

	/** ステータス412 */
	public static final HttpResponseStatus STATUS_412_PRECONDITION_FAILED = new HttpResponseStatus( //
			412 //
			, "Precondition Failed" //
			, "A condition in the header field data passed by the client evaluated false on the server. " //
					+ "This lets the client prevent the requested method from being applied to a resource other than the one intended." //
			, "If-... ヘッダで指定された条件に合致しませんでした。" //
	);

	/** ステータス413 */
	public static final HttpResponseStatus STATUS_413_REQUEST_ENTITY_TOO_LARGE = new HttpResponseStatus( //
			413 //
			, "Request Entity Too Large" //
			, "The server refused to process the request because the request entity too large. " //
					+ "The server may close the connection to prevent the client from continuing the request." //
			, "要求されたエンティティが大きすぎます。" //
	);

	/** ステータス414 */
	public static final HttpResponseStatus STATUS_414_REQUEST_URI_TOO_LONG = new HttpResponseStatus( //
			414 //
			, "Request-URI Too Long" //
			, "The server is refusing to service the request because the Request-URI is longer than the server is willing to interpret." //
			, "要求された URI が長すぎます。" //
	);

	/** ステータス415 */
	public static final HttpResponseStatus STATUS_415_UNSUPPORTED_MEDIA_TYPE = new HttpResponseStatus( //
			415 //
			, "Unsupported Media Type" //
			, "The server is refusing to service the request because the entity of the request is in a format not supported by " // 
					+ "the requested resource for the requested method." //
			, "サポートされていないメディアタイプです。" //
	);

	/** ステータス416 */
	public static final HttpResponseStatus STATUS_416_REQUESTED_RANGE_NOT_SATISFIABLE = new HttpResponseStatus( //
			416 //
			, "Requested Range Not Satisfiable" //
			, "A server SHOULD return a response with this status code if a request included a Range request-header field." //
			, "要求されたレンジが不正です。" //
	);

	/** ステータス417 */
	public static final HttpResponseStatus STATUS_417_EXPECTATION_FAILED = new HttpResponseStatus( //
			417 //
			, "Expectation Failed" //
			, "The server could not meet the expectation in an Expect request-header field. " //
					+ "If the server is a proxy, the server has unambiguous evidence that the request could not be met by the next-hop server." //
			, "Expect ヘッダで指定された拡張要求は失敗しました。" //
	);

	/*
	 * サーバーエラーステータス
	 */

	/** ステータス500 */
	public static final HttpResponseStatus STATUS_500_INTERNAL_SERVER_ERROR = new HttpResponseStatus( //
			500 //
			, "Internal Server Error" //
			, "The server encountered an unexpected condition which prevented it from fulfilling the request." //
			, "サーバーで予期しないエラーが発生しました。" //
	);

	/** ステータス501 */
	public static final HttpResponseStatus STATUS_501_NOT_IMPLEMENTED = new HttpResponseStatus( //
			501 //
			, "Not Implemented" //
			, "The server does not support the facility required." //
			, "実装されていません。" //
	);

	/** ステータス502 */
	public static final HttpResponseStatus STATUS_502_BAD_GATEWAY = new HttpResponseStatus( //
			502 //
			, "Bad Gateway" //
			, "The server, while acting as a gateway or proxy, " //
					+ "received an invalid response from the upstream server it accessed in attempting to fulfill the request." //
			, "ゲートウェイが不正です。" //
	);

	/** ステータス503 */
	public static final HttpResponseStatus STATUS_503_SERVICE_UNAVAILABLE = new HttpResponseStatus( //
			503 //
			, "Service Unavailable" //
			, "The server cannot process the request due to a system overload. This (should) be a temporary condition." //
			, "サービスは利用可能ではありません。" //
	);

	/** ステータス504 */
	public static final HttpResponseStatus STATUS_504_GATEWAY_TIMEOUT = new HttpResponseStatus( //
			504 //
			, "Gateway Timeout" //
			, "The service did not respond within the time frame that the gateway was willing to wait." //
			, "ゲートウェイがタイムアウトしました。" //
	);

	/** ステータス505 */
	public static final HttpResponseStatus STATUS_505_HTTP_VERSION_NOT_SUPPORTED = new HttpResponseStatus( //
			505 //
			, "HTTP Version Not Supported" //
			, "The server does not support, or refuses to support, the HTTP protocol version that was used in the request message." //
			, "このHTTPバージョンはサポートされていません。" //
	);

	/** 全ステータス配列 */
	private static final HttpResponseStatus[] ALL_STATUS = new HttpResponseStatus[] { //
	//
			STATUS_100_CONTINUE, //
			STATUS_101_SWITCHING_PROTOCOLS, //
			STATUS_200_OK, //
			STATUS_201_CREATED, //
			STATUS_202_ACCEPTED, //
			STATUS_203_NON_AUTHORITATIVE_INFORMATION, //
			STATUS_204_NO_CONTENT, //
			STATUS_205_RESET_CONTENT, //
			STATUS_206_PARTIAL_CONTENT, //
			STATUS_300_MULTIPLE_CHOICES, //
			STATUS_301_MOVED_PERMANENTLY, //
			STATUS_302_FOUND, //
			STATUS_303_SEE_OTHER, //
			STATUS_304_NOT_MODIFIED, //
			STATUS_305_USE_PROXY, //
			STATUS_306_UNUSED, //
			STATUS_307_TEMPORARY_REDIRECT, //
			STATUS_400_BAD_REQUEST, //
			STATUS_401_UNAUTHORIZED, //
			STATUS_402_PAYMENT_REQUIRED, //
			STATUS_403_FORBIDDEN, //
			STATUS_404_NOT_FOUND, //
			STATUS_405_METHOD_NOT_ALLOWED, //
			STATUS_406_NOT_ACCEPTABLE, //
			STATUS_407_PROXY_AUTHENTICATION_REQUIRED, //
			STATUS_408_REQUEST_TIMEOUT, //
			STATUS_409_CONFLICT, //
			STATUS_410_GONE, //
			STATUS_411_LENGTH_REQUIRED, //
			STATUS_412_PRECONDITION_FAILED, //
			STATUS_413_REQUEST_ENTITY_TOO_LARGE, //
			STATUS_414_REQUEST_URI_TOO_LONG, //
			STATUS_415_UNSUPPORTED_MEDIA_TYPE, //
			STATUS_416_REQUESTED_RANGE_NOT_SATISFIABLE, //
			STATUS_417_EXPECTATION_FAILED, //
			STATUS_500_INTERNAL_SERVER_ERROR, //
			STATUS_501_NOT_IMPLEMENTED, //
			STATUS_502_BAD_GATEWAY, //
			STATUS_503_SERVICE_UNAVAILABLE, //
			STATUS_504_GATEWAY_TIMEOUT, //
			STATUS_505_HTTP_VERSION_NOT_SUPPORTED, //
	};

	/** ステータスコード */
	private int code;

	/** ステータスメッセージ */
	private String message;

	/** 説明(en) */
	private String description_en;

	/** 説明(ja) */
	private String description_ja;

	/**
	 * コンストラクタ<br>
	 */
	private HttpResponseStatus() {
		super();
	}

	/**
	 * コンストラクタ<br>
	 * @param code ステータスコード
	 * @param message ステータスメッセージ
	 * @param description_en 説明(en)
	 * @param description_ja 説明(ja)
	 */
	public HttpResponseStatus(int code, String message, String description_en, String description_ja) {
		super();
		this.code = code;
		this.message = message;
		this.description_en = description_en;
		this.description_ja = description_ja;
	}

	/**
	 * ステータスコードを取得します。<br>
	 * @return ステータスコード
	 */
	public int getCode() {
		return code;
	}

	/**
	 * ステータスメッセージを取得します。<br>
	 * @return ステータスメッセージ
	 */
	public String getMessage() {
		return message;
	}

	/**
	 * 説明(en)を取得します。<br>
	 * @return 説明(en)
	 */
	public String getDescriptionEn() {
		return description_en;
	}

	/**
	 * 説明(ja)を取得します。<br>
	 * @return 説明(ja)
	 */
	public String getDescriptionJa() {
		return description_ja;
	}

	/**
	 * 指定されたステータスコードのステータスオブジェクトを取得します。<br>
	 * 指定されたコードに対応するステータスオブジェクトが存在しない場合は
	 * 一時的に作成される指定されたコードを保持したステータスオブジェクトが返却されます。<br>
	 * @param code ステータスコード
	 * @return ステータスオブジェクト
	 */
	public static HttpResponseStatus getStatus(int code) {
		for (int i = 0; i <= ALL_STATUS.length - 1; i++) {
			HttpResponseStatus status = ALL_STATUS[i];
			if (code == status.getCode()) {
				return status;
			}
		}
		return new HttpResponseStatus(code, "", "", "");
	}

	/**
	 * 指定されたステータスコードのステータスオブジェクトを取得します。<br>
	 * 指定されたコードに対応するステータスオブジェクトが存在しない場合は
	 * 一時的に作成される指定されたコードを保持したステータスオブジェクトが返却されます。<br>
	 * @param code ステータスコード
	 * @param defaultMessage ステータスオブジェクトに設定するディフォルトメッセージ
	 * @return ステータスオブジェクト
	 */
	public static HttpResponseStatus getStatus(int code, String defaultMessage) {
		HttpResponseStatus status = getStatus(code);
		if (status.getMessage().equals(defaultMessage)) {
			return status;
		} else {
			return new HttpResponseStatus(code, defaultMessage == null ? "" : defaultMessage, status.getDescriptionEn(), status.getDescriptionJa());
		}
	}

	/**
	 * 指定されたステータスコードのステータスオブジェクトを取得します。<br>
	 * 指定されたコードに対応するステータスオブジェクトが存在しない場合はnullを返却します。<br>
	 * @param code ステータスコード
	 * @return ステータスオブジェクト
	 */
	public static HttpResponseStatus getStatus(String code) {
		return getStatus(Integer.parseInt(code));
	}

	/**
	 * 指定されたステータスコードのステータスオブジェクトを取得します。<br>
	 * 指定されたコードに対応するステータスオブジェクトが存在しない場合は
	 * 一時的に作成される指定されたコードを保持したステータスオブジェクトが返却されます。<br>
	 * @param code ステータスコード
	 * @param defaultMessage ステータスオブジェクトに設定するディフォルトメッセージ
	 * @return ステータスオブジェクト
	 */
	public static HttpResponseStatus getStatus(String code, String defaultMessage) {
		return getStatus(Integer.parseInt(code), defaultMessage);
	}

	/**
	 * オブジェクトのハッシュコード値を返します。<br>
	 * @return オブジェクトのハッシュコード値
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + code;
		//result = PRIME * result + ((description == null) ? 0 : description.hashCode());
		//result = PRIME * result + ((message == null) ? 0 : message.hashCode());
		return result;
	}

	/**
	 * このオブジェクトと他のオブジェクトが等しいかどうかを示します。<br>
	 * @param object 比較対象オブジェクト
	 * @return オブジェクト同士が等価である場合にtrueを返却
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object object) {
		if (this == object)
			return true;
		if (object == null)
			return false;
		if (getClass() != object.getClass())
			return false;
		final HttpResponseStatus other = (HttpResponseStatus) object;
		if (code != other.code)
			return false;
		//		if (description == null) {
		//			if (other.description != null)
		//				return false;
		//		} else if (!description.equals(other.description))
		//			return false;
		//		if (message == null) {
		//			if (other.message != null)
		//				return false;
		//		} else if (!message.equals(other.message))
		//			return false;
		return true;
	}

	/**
	 * オブジェクトの文字列表現を返します。<br>
	 * @return オブジェクトの文字列表現
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer();
		buffer.append("{");
		buffer.append("code=");
		buffer.append(code);
		buffer.append(",");
		buffer.append("message=");
		buffer.append(message);
		buffer.append(",");
		buffer.append("description_en=");
		buffer.append(description_en);
		buffer.append(",");
		buffer.append("description_ja=");
		buffer.append(description_ja);
		buffer.append("}");
		return buffer.toString();
	}
}
