package org.phosphoresce.commons.socket.core.io;

import java.io.IOException;
import java.net.Socket;

/**
 * ソケットストリーム中継ストリームクラス<br>
 * <br>
 * 当クラスは二つのソケットストリーム間のデータ転送を行うためのインタフェースを提供します。<br>
 * 各種転送インタフェースを使用したタイミングで入力元から指示された仕様でデータを入力し、
 * そのデータの形式で出力先ソケットストリームにデータを出力します。<br>
 * <br>
 * また、クラスが保持する入力側、出力側のストリームベクトルを変更してストリームの流れを
 * 変更することも可能です(但し、ストリーム処理中に変更することは不可能)。<br>
 * <br>
 * 当クラスは基本的に上記のような転送を目的としたクラスで、転送を行うためのインタフェースを
 * 提供しますが、例外的に個別の入力、出力を行うことも考えられます。その場合、個別のストリーム
 * オブジェクトをユーザーが用意して使用することで実装を行うことも可能ですが、当クラスでは、
 * これらの場合にも統合的にストリームを扱うために、個別の入出力インタフェースも提供します。<br>
 * 但し、個別に入出力を行った場合、それらの情報はユーザー側で状況を把握しておく必要があります。<br>
 * <br>
 * <br>
 * 当クラスを継承したサブクラスを設ける場合、ストリームの処理インタフェースを追加する際には
 * ストリーム処理中フラグを妥当な値に設定する処理を実装することが必要です。<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/10/30	Kitagawa		新規作成
 *-->
 */
public class SocketRelayStream {

	/** 入力元ソケットストリーム */
	protected SocketStream source;

	/** 出力先ソケットストリーム */
	protected SocketStream destination;

	/** ストリーム処理中フラグ */
	protected boolean streaming;

	/**
	 * コンストラクタ<br>
	 */
	public SocketRelayStream() {
		super();
		this.source = null;
		this.destination = null;
		this.streaming = false;
	}

	/**
	 * コンストラクタ<br>
	 * @param source 入力元ソケットストリーム
	 * @param destination 出力先ソケットストリーム
	 */
	public SocketRelayStream(SocketStream source, SocketStream destination) {
		super();
		this.source = source;
		this.destination = destination;
		this.streaming = false;
	}

	/**
	 * コンストラクタ<br>
	 * @param source 入力元ソケット
	 * @param destination 出力先ソケット
	 * @throws IOException 正常にソケットストリームを生成できなかった場合にスローされます
	 */
	public SocketRelayStream(Socket source, Socket destination) throws IOException {
		super();
		this.source = source == null ? null : new SocketStream(source);
		this.destination = destination == null ? null : new SocketStream(destination);
		this.streaming = false;
	}

	// Source Object Accessor Method

	/**
	 * 入力元ソケットを取得します。<br>
	 * 入力元ソケットに対して直接操作を行う場合、リレーストリームが何らかの処理が
	 * 行われているかどうかを留意する必要があります。<br>
	 * @return 入力元ソケット
	 */
	public Socket getSourceSocket() {
		return source.getSocket();
	}

	/**
	 * 指定された入力元ソケットを設定します。<br>
	 * 新たな入力元ソケットを設定する場合、リレーストリームが何らかの処理が
	 * 行われているかどうかを留意する必要があります。<br>
	 * @param source 入力元ソケット
	 * @throws IOException 指定されたソケットからストリームオブジェクトを生成できなかった場合にスローされます
	 */
	public void setSourceSocket(Socket source) throws IOException {
		if (source == null) {
			this.source = null;
		} else {
			this.source = new SocketStream(source);
		}
	}

	/**
	 * 入力元ソケットストリームを取得します。<br>
	 * 入力元ソケットストリームに対して直接操作を行う場合、リレーストリームが何らかの処理が
	 * 行われているかどうかを留意する必要があります。<br>
	 * @return 入力元ソケットストリーム
	 */
	public SocketStream getSourceStream() {
		return source;
	}

	/**
	 * 指定された入力元ソケットストリームを設定します。<br>
	 * 新たな入力元ソケットストリームを設定する場合、リレーストリームが何らかの処理が
	 * 行われているかどうかを留意する必要があります。<br>
	 * @param source 入力元ソケットストリーム
	 */
	public void setSourceStream(SocketStream source) {
		this.source = source;
	}

	/**
	 * 出力先ソケットを取得します。<br>
	 * 出力先ソケットに対して直接操作を行う場合、リレーストリームが何らかの処理が
	 * 行われているかどうかを留意する必要があります。<br>
	 * @return 出力先ソケット
	 */
	public Socket getDestinationSocket() {
		return destination.getSocket();
	}

	/**
	 * 指定された出力先ソケットを設定します。<br>
	 * 新たな出力先ソケットを設定する場合、リレーストリームが何らかの処理が
	 * 行われているかどうかを留意する必要があります。<br>
	 * @param destination 入力元ソケット
	 * @throws IOException 指定されたソケットからストリームオブジェクトを生成できなかった場合にスローされます
	 */
	public void setDestinationSocket(Socket destination) throws IOException {
		if (destination == null) {
			this.destination = null;
		} else {
			this.destination = new SocketStream(destination);
		}
	}

	/**
	 * 出力先ソケットストリームを取得します。<br>
	 * 出力先ソケットストリームに対して直接操作を行う場合、リレーストリームが何らかの処理が
	 * 行われているかどうかを留意する必要があります。<br>
	 * @return 出力先ソケットストリーム
	 */
	public SocketStream getDestinationStream() {
		return destination;
	}

	/**
	 * 指定された出力先ソケットストリームを設定します。<br>
	 * 新たな出力先ソケットストリームを設定する場合、リレーストリームが何らかの処理が
	 * 行われているかどうかを留意する必要があります。<br>
	 * @param destination 出力先ソケットストリーム
	 */
	public void setDestinationStream(SocketStream destination) {
		this.destination = destination;
	}

	// Class Method

	/**
	 * 入力元と出力先を入れ替えてストリームの方向を逆にします。<br>
	 * @throws IOException スレッド処理などでストリーム処理中に方向を変更しようとした場合にスローされます
	 */
	public void reverseStreamVector() throws IOException {
		if (streaming) {
			throw new IOException("Stream is processing");
		}
		SocketStream temporary = source;
		source = destination;
		destination = temporary;
	}

	/**
	 * ストリーム処理中であるか判定します。<br>
	 * @return ストリーム処理中である場合にtrueを返却
	 */
	public boolean isStreaming() {
		return streaming;
	}

	// Common Stream Method

	/**
	 * 入力元及び、出力先のソケットストリームをクローズします。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 */
	public void close() throws IOException {
		streaming = true;
		try {
			if (source != null) {
				source.close();
			}
			if (destination != null) {
				destination.close();
			}
		} finally {
			streaming = false;
		}
	}

	// InputStream Side Transfer Method

	/**
	 * 入力ストリームのメソッドの次の呼び出し側からブロックされることなく、
	 * 入力ストリームから読み込むことができる(またはスキップできる)バイト数を返します。<br>
	 * @return ブロックしないで入力ストリームから読み込むことができるバイト数
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#available()
	 */
	public int available() throws IOException {
		streaming = true;
		try {
			return source.available();
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力ストリームから次のバイトデータを読み込みます。<br>
	 * @return バイトデータ
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#read()
	 */
	public int read() throws IOException {
		streaming = true;
		try {
			return source.read();
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力ストリームからlengthバイトまでのデータをバイト配列に読み込みます。<br>
	 * @param bytes 読込先バイト配列
	 * @param offset データが書き込まれる配列のオフセット
	 * @param length 読み込むバイト数
	 * @return バッファに読み込まれたバイトの合計数(ストリームの終わりに達してデータがない場合は-1を返却)
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#read(byte[], int, int)
	 */
	public int read(byte[] bytes, int offset, int length) throws IOException {
		streaming = true;
		try {
			return source.read(bytes, offset, length);
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力ストリームからバイト数を読み込み、それをバッファ配列に格納します。<br>
	 * @param bytes 読込先バイト配列
	 * @return バッファに読み込まれたバイトの合計数(ストリームの終わりに達してデータがない場合は-1を返却)
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#read(byte[])
	 */
	public int read(byte[] bytes) throws IOException {
		streaming = true;
		try {
			return source.read(bytes);
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力ストリームから単一行を読み込みます。<br>
	 * <b>現在のメソッド仕様では、行のデリミタとなるコードは\nまたは\r\n固定となります。</b><br>
	 * @return 読み込まれた一行
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#readLine()
	 */
	public String readLine() throws IOException {
		streaming = true;
		try {
			return source.readLine();
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力ストリームから読み込めるバイトデータを全て読み込みます。<br>
	 * 終了判定はストリームのavaivableが提供するブロックされずに読み込めるサイズを条件とします。<br>
	 * 即ち、ストリームがブロックされている状態の箇所で読み込みが完了します。それ以降に再度
	 * ストリームがデータを提供した場合でも、当メソッドは自身で復帰することはありません。<br>
	 * @return 読み込んだバイト配列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#reads()
	 */
	public byte[] reads() throws IOException {
		streaming = true;
		try {
			return source.reads();
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力ストリームからlengthバイト分のバイトデータを読み込み配列として提供します。<br>
	 * 当メソッドはソケットからの動的なストリームデータを読み込むことを前提としているため、
	 * 内部でバッファリングしながらの読み込みを行いません。なぜなら、ソケットストリームでは
	 * 正確にavaivableでの終了判定が行えない為、ブロックされてしまうことがある為です。
	 * @param length 読み込むバイト数
	 * @return 読み込んだバイト配列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#reads(int)
	 */
	public byte[] reads(int length) throws IOException {
		streaming = true;
		try {
			return source.reads(length);
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力ストリームからのデータをlengthバイトだけスキップしてその範囲のデータを破棄します。<br>
	 * @param length 破棄するバイト数
	 * @return 実際にスキップされたバイト数
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see java.io.InputStream#skip(long)
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#skip(long)
	 */
	public long skip(long length) throws IOException {
		streaming = true;
		try {
			return source.skip(length);
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力ストリームの現在位置にマークを設定します。<br>
	 * @param readlimit マーク位置が無効になる前に読み込み可能なバイトの最大リミット
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#mark(int)
	 */
	public void mark(int readlimit) throws IOException {
		streaming = true;
		try {
			source.mark(readlimit);
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力ストリームがmarkおよびresetメソッドをサポートしているかどうかを判定します。<br>
	 * @return サポートしている場合にtrueを返却
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#markSupported()
	 */
	public boolean markSupported() throws IOException {
		streaming = true;
		try {
			return source.markSupported();
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力ストリームの位置を、入力ストリームで最後にmarkメソッドが呼び出されたときのマーク位置に再設定します。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#reset()
	 */
	public void reset() throws IOException {
		streaming = true;
		try {
			source.reset();
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力元ソケットストリームから単一行を読み飛ばします。<br>
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public void skipLine() throws IOException {
		streaming = true;
		try {
			source.skipLine();
		} finally {
			streaming = false;
		}
	}

	// OutputStream Side Transfer Method

	/**
	 * 出力ストリームをフラッシュし、バッファに入っている出力バイトをすべて強制的に書き込みます。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#flush()
	 */
	public void flush() throws IOException {
		streaming = true;
		try {
			destination.flush();
		} finally {
			streaming = false;
		}
	}

	/**
	 * オフセットoffsetから始まる指定のバイト配列から出力ストリームにlengthバイトを書き込みます。<br>
	 * @param bytes 書き込むバイトデータ配列
	 * @param offset データの開始オフセット
	 * @param length 書き込むバイト数
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#write(byte[], int, int)
	 */
	public void write(byte[] bytes, int offset, int length) throws IOException {
		streaming = true;
		try {
			destination.write(bytes, offset, length);
		} finally {
			streaming = false;
		}
	}

	/**
	 * 指定のバイト配列を出力ストリームに書き込みます。<br>
	 * @param bytes 書き込むバイトデータ配列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#write(byte[])
	 */
	public void write(byte[] bytes) throws IOException {
		streaming = true;
		try {
			destination.write(bytes);
		} finally {
			streaming = false;
		}
	}

	/**
	 * 出力ストリームに指定されたバイトを書き込みます。<br>
	 * @param b バイトデータ
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#write(int)
	 */
	public void write(int b) throws IOException {
		streaming = true;
		try {
			destination.write(b);
		} finally {
			streaming = false;
		}
	}

	/**
	 * 指定された文字列を出力ストリームに書き込みます。<br>
	 * @param line 書き込む文字列
	 * @param withSeparator 文字列末尾に行セパレータ(\r\n)を付与して書き込む場合はtrueを指定
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#writeLine(java.lang.String, boolean)
	 */
	public void writeLine(String line, boolean withSeparator) throws IOException {
		streaming = true;
		try {
			destination.writeLine(line, withSeparator);
		} finally {
			streaming = false;
		}
	}

	/**
	 * 指定された文字列を出力ストリームに書き込みます。<br>
	 * 尚、文字列出力末尾には改行コードが付与されて書き込まれます。<br>
	 * @param line 書き込む文字列
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#writeLine(java.lang.String)
	 */
	public void writeLine(String line) throws IOException {
		streaming = true;
		try {
			destination.writeLine(line);
		} finally {
			streaming = false;
		}
	}

	/**
	 * 出力ストリームに行セパレータを書き込みます。<br>
	 * 通常の文字列出力時には単独での行セパレータ出力は不要となりますが、
	 * データデリミタ識別用に意図的にセパレータのみの出力を行う場合に使用します。<br>
	 * @throws IOException ストリーム操作時に入出力例外が発生した場合にスローされます
	 * @see org.phosphoresce.commons.socket.core.io.SocketStream#writeSeparate()
	 */
	public void writeSeparate() throws IOException {
		streaming = true;
		try {
			destination.writeSeparate();
		} finally {
			streaming = false;
		}
	}

	// Forwarding Method

	/**
	 * 入力元ソケットストリームから出力先ソケットストリームに対して
	 * 1バイトのデータを転送します。<br>
	 * @return 転送した1バイトデータ
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public int foward() throws IOException {
		streaming = true;
		try {
			int b = source.read();
			destination.write(b);
			return b;
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力元ソケットストリームから出力先ソケットストリームに対して
	 * 指定されたバイト数のデータを転送します。<br>
	 * @param length 転送するバイト数
	 * @return 転送したバイトデータ
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public byte[] forward(int length) throws IOException {
		streaming = true;
		try {
			byte[] bytes = source.reads(length);
			destination.write(bytes);
			return bytes;
		} finally {
			streaming = false;
		}
	}

	/**
	 * 指定されたバイト数分のデータを指定されたバッファサイズごとに転送します。<br>
	 * @param length 転送サイズ
	 * @param bufferSize バッファサイズ
	 * @return 転送したバイト数
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public int forward(int length, int bufferSize) throws IOException {
		streaming = true;
		try {
			int avaivable = length;
			int readSize = -1;
			int readed = 0;
			while (true) {
				if (avaivable >= bufferSize) {
					readSize = bufferSize;
				} else {
					readSize = avaivable;
				}
				avaivable -= readSize;
				byte[] bytes = forward(readSize);
				readed += bytes.length;
				if (avaivable <= 0) {
					break;
				}
			}
			return readed;
		} finally {
			streaming = false;
		}
	}

	/**
	 * ストリームの最終までを指定されたバッファサイズごとに転送します。<br>
	 * 終了判定はストリームのavaivableが提供するブロックされずに読み込めるサイズを条件とします。<br>
	 * @param bufferSize バッファサイズ
	 * @return 転送したバイト数
	 * @throws IOException
	 */
	public int forwardEOS(int bufferSize) throws IOException {
		streaming = true;
		try {
			int readed = 0;
			while (true) {
				if (source.available() <= 0) {
					break;
				}
				int readSize = bufferSize > source.available() ? source.available() : bufferSize;
				readed += readSize;
				byte[] bytes = source.reads(readSize);
				destination.write(bytes);
			}
			return readed;
		} finally {
			streaming = false;
		}
	}

	/**
	 * 入力元ソケットストリームから出力先ソケットストリームに対して
	 * 単一行のデータを転送します。<br>
	 * @return 転送した単一行データ
	 * @throws IOException ストリーム操作中に入出力例外が発生した場合にスローされます
	 */
	public String forwardLine() throws IOException {
		streaming = true;
		try {
			String line = source.readLine();
			destination.writeLine(line);
			return line;
		} finally {
			streaming = false;
		}
	}
}
