package org.phosphoresce.common.graphics.util;

import java.awt.AlphaComposite;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Image;
import java.awt.Transparency;
import java.awt.color.ColorSpace;
import java.awt.image.BufferedImage;
import java.awt.image.ColorConvertOp;

/**
 * イメージ関連グラフィックスユーティリティクラス<br>
 * 
 * @author Kitagawa<br>
 * 
 *<!--
 * 更新日		更新者			更新内容
 * 2008/06/04	Kitagawa		新規作成
 *-->
 */
public final class ImageGraphicsUtil {

	/**
	 * コンストラクタ<br>
	 */
	private ImageGraphicsUtil() {
		super();
	}

	/**
	 * 指定されたイメージをBufferedImageに変換します。<br>
	 * 但し、指定されたイメージがBufferedImageである場合、そのまま返却されます。<br>
	 * @param image イメージオブジェクト
	 * @return BufferedImageオブジェクト
	 */
	public static BufferedImage convertBufferedImage(Image image) {
		if (image == null) {
			return null;
		} else if (image instanceof BufferedImage) {
			return (BufferedImage) image;
		} else {
			BufferedImage buffer = createEmptyBufferedImage(ImageUtil.getImageSize(image));
			Graphics graphics = buffer.getGraphics();
			graphics.drawImage(image, 0, 0, null);
			graphics.dispose();
			return buffer;
		}
	}

	/**
	 * 指定されたサイズとTransparencyタイプのBufferedImageを生成します。<br>
	 * @param dimension サイズ
	 * @param type Transparencyタイプ
	 * @return 空のBufferedImageオブジェクト
	 */
	public static BufferedImage createEmptyBufferedImage(Dimension dimension, int type) {
		GraphicsEnvironment environment = GraphicsEnvironment.getLocalGraphicsEnvironment();
		GraphicsDevice device = environment.getDefaultScreenDevice();
		GraphicsConfiguration configuration = device.getDefaultConfiguration();
		BufferedImage image = configuration.createCompatibleImage(dimension.width, dimension.height, type);
		return image;
	}

	/**
	 * 指定されたサイズとTransparencyタイプのBufferedImageを生成します。<br>
	 * @param width 幅
	 * @param height 高さ
	 * @param type Transparencyタイプ
	 * @return 空のBufferedImageオブジェクト
	 */
	public static BufferedImage createEmptyBufferedImage(int width, int height, int type) {
		return createEmptyBufferedImage(new Dimension(width, height), type);
	}

	/**
	 * 指定されたサイズのBufferedImageを生成します。<br>
	 * イメージのTransparencyタイプはTRANSLUCENTとなります。<br>
	 * @param dimension サイズ
	 * @return 空のBufferedImageオブジェクト
	 */
	public static BufferedImage createEmptyBufferedImage(Dimension dimension) {
		return createEmptyBufferedImage(dimension, Transparency.TRANSLUCENT);
	}

	/**
	 * 指定されたサイズとTransparencyタイプのBufferedImageを生成します。<br>
	 * イメージのTransparencyタイプはTRANSLUCENTとなります。<br>
	 * @param width 幅
	 * @param height 高さ
	 * @return 空のBufferedImageオブジェクト
	 */
	public static BufferedImage createEmptyBufferedImage(int width, int height) {
		return createEmptyBufferedImage(width, height, Transparency.TRANSLUCENT);
	}

	/**
	 * 指定されたコンポーネントのイメージを指定されたTransparencyタイプで生成します。<br>
	 * @param component コンポーネント
	 * @param type Transparencyタイプ
	 * @return コンポーネント描画イメージ
	 */
	public static BufferedImage createComponentImage(Component component, int type) {
		BufferedImage buffer = createEmptyBufferedImage(component.getSize(), type);
		Graphics graphics = buffer.getGraphics();
		component.paintAll(graphics);
		graphics.dispose();
		return buffer;
	}

	/**
	 * 指定されたコンポーネントのイメージを生成します。<br>
	 * @param component コンポーネントオブジェクト
	 * @return コンポーネント描画イメージ
	 */
	public static BufferedImage createComponentImage(Component component) {
		return createComponentImage(component, Transparency.TRANSLUCENT);
	}

	/**
	 * 指定されたイメージをグレースケールしたイメージを提供します。<br>
	 * @param image イメージ
	 * @return グレースケールされたイメージ
	 */
	public static BufferedImage scaleGlay(Image image) {
		ColorConvertOp op = new ColorConvertOp(ColorSpace.getInstance(ColorSpace.CS_GRAY), null);
		BufferedImage buffer = op.filter(convertBufferedImage(image), null);
		return buffer;
	}

	/**
	 * 指定されたイメージに対してアルファ処理を行ったイメージを提供します。<br>
	 * @param image イメージ
	 * @param alpha アルファ値(0.0f～1.0f)
	 * @return アルファ処理されたイメージ
	 */
	public static BufferedImage scaleAlpha(Image image, float alpha) {
		BufferedImage render = createEmptyBufferedImage(ImageUtil.getImageSize(image));
		Graphics2D graphics = (Graphics2D) render.getGraphics();
		graphics.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER, alpha));
		graphics.drawImage(image, 0, 0, null);
		graphics.dispose();
		return render;
	}
}
